# frozen_string_literal: true

require 'litany/cli/base'
require 'litany/project/project_workspace'
require 'litany/project/project_workspace_delegate'

module Litany
  module CLI
    # Compiles the current Litany workspace
    class Init < Litany::CLI::Base
      include ProjectWorkspaceDelegate

      parameter '[DIRECTORY]', 'project workspace directory', default: Dir.pwd, &proc { |dir| File.expand_path(dir) }

      def execute
        exit(1) unless ProjectWorkspace.new(directory: directory, delegate: self).initialize_workspace
      end

      def use_non_empty_workspace?
        agree 'The workspace directory is not empty, do you still wish to initialize your project here?', default: false
      end

      def overwrite_existing_workspace?
        question = 'There is an existing Litany workspace here!  Do you wish to overwrite it?  <%= color("Warning: This cannot be undone!", :underline, :yellow) %>'
        agree question, confirm: true, default: false
      end

      def new_workspace_name
        ask 'What is the new projects name?', default: File.basename(directory), validator: /.*/
      end

      def allowed_stages
        stages = ask 'What stages are considered valid for this project?  Please use a comma separated list.', default: 'general'
        stages.split(',').collect(&:downcase).collect(&:to_sym)
      end

      def aws_profile(for_stage:)
        ask "What aws profile should we use for this the #{for_stage} stage?", default: 'default', validator: /^[a-z-]+$/
      end
    end
  end
end
