# frozen_string_literal: true

require 'litany/resource'
require 'litany/mixins'

module Litany
  class AutoScalingVolume < MetaResource
    property :device, nil, String do |value|
      raise "Invalid value `#{value}` for device.  Examples: `/dev/xvd[b-c][a-z]`." unless value =~ %r/\/dev\/xvd[b-c][a-z]/
    end
    property :encrypted, true, [true, false] # not supported inside of eb
    property :iops, nil, [nil, 100..20_000]
    property :size, nil, 4..16_384
    property :volume_type, :gp2, [:gp2, :io1, :general, :provisioned] do |value|
      map = {general: :gp2, provisioned: :io1}
      map.include?(value) ? map[value] : value
    end

    def to_eb_string
      if iops.nil?
        "#{device}=:#{size}:true:#{volume_type}"
      else
        "#{device}=:#{size}:true:io1:#{iops}"
      end
    end

    def to_json(*args)
      ## TODO: SnapshotId

      props = {
        DeviceName: device,
        Ebs: {
          DeleteOnTermination: true,
          Encrypted: encrypted,
          VolumeSize: size,
          VolumeType: volume_type
        }
      }

      unless iops.nil?
        props[:Ebs][:Iops] = iops
        props[:Ebs][:VolumeType] = :io1
      end

      props.to_json(*args)
    end
  end
end
