# frozen_string_literal: true

require 'litany/mixins'
require 'litany/resource'

module Litany
  class RDSClusterParameterGroup < Resource
    cfn_type 'AWS::RDS::DBClusterParameterGroup'
    parent_resource :rds_cluster

    property :description, nil, [nil, String]
    property :family, 'aurora5.6', ['aurora5.6', 'aurora-mysql5.7', 'aurora-postgres9.6']

    def parameter(key, value)
      valid_parameters = [
        :auto_increment_increment, :auto_increment_offset, :binlog_checksum, :binlog_format, :binlog_row_image, :binlog_rows_query_log_events,
        :character_set_database, :character_set_filesystem, :completion_type, :default_storage_engine, :innodb_autoinc_lock_mode, :innodb_checksum_algorithm,
        :innodb_checksums, :innodb_cmp_per_index_enabled, :innodb_commit_concurrency, :innodb_data_home_dir, :innodb_doublewrite, :innodb_file_per_table,
        :innodb_flush_log_at_trx_commit, :innodb_ft_max_token_size, :innodb_ft_min_token_size, :innodb_ft_num_word_optimize, :innodb_ft_sort_pll_degree,
        :innodb_online_alter_log_max_size, :innodb_optimize_fulltext_only, :innodb_page_size, :innodb_print_all_deadlocks, :innodb_purge_batch_size,
        :innodb_purge_threads, :innodb_rollback_on_timeout, :innodb_rollback_segments, :innodb_spin_wait_delay, :innodb_strict_mode, :innodb_support_xa,
        :innodb_sync_array_size, :innodb_sync_spin_loops, :innodb_table_locks, :innodb_undo_directory, :innodb_undo_logs, :innodb_undo_tablespaces,
        :lc_time_names, :lower_case_table_names, :master_info_repository, :master_verify_checksum, :server_id, :skip_character_set_client_handshake,
        :skip_name_resolve, :sync_frm, :time_zone
      ]

      raise "Invalid parameter `#{key} cannot be set." unless valid_parameters.include? key

      mappings = {
        master_info_repository: :'master-info-repository',
        skip_character_set_client_handshake: :'skip-character-set-client-handshake'
      }

      parameters[mappings.fetch(key, key)] = value
    end

    def parameters
      @parameters ||= {}
    end

    def properties
      {
        Description: description || "Parameter group for #{rds_cluster.resource_name}",
        Family: family,
        Parameters: parameters,
        # These Tags cannot be updated due to a bug in CF (?)
        Tags: [
          {
            Key: 'ProjectName',
            Value: project.name.pascal_case
          },
          {
            Key: 'Name',
            Value: output_name
          }
        ]
      }
    end
  end
end
