# frozen_string_literal: true

require 'litany/mixins'
require 'litany/resource'

module Litany
  class SecretsManagerSecret < Resource
    include Taggable

    cfn_type 'AWS::SecretsManager::Secret'

    flag :exclude_lowercase, false
    flag :exclude_numbers, false
    flag :exclude_punctuation, false
    flag :exclude_uppercase, false
    flag :include_spaces, false
    flag :require_each_type, true

    property :description, nil, String
    property :excluded_characters, '', String
    property :generate_key, 'password', String
    property :length, 32, Integer
    property :template, '{}', String

    def properties
      # Skipping KmsKeyId, Name, SecretString
      {
        Description: description,
        GenerateSecretString: {
          ExcludeCharacters: excluded_characters,
          ExcludeLowercase: exclude_lowercase?,
          ExcludeNumbers: exclude_numbers?,
          ExcludePunctuation: exclude_punctuation?,
          ExcludeUppercase: exclude_uppercase?,
          GenerateStringKey: generate_key,
          IncludeSpace: include_spaces?,
          PasswordLength: length,
          RequireEachIncludedType: require_each_type?,
          SecretStringTemplate: template
        }
      }
    end

    def resolve(key)
      sub("{{resolve:secretsmanager:${#{resource_name}}::#{key}}}")
    end
  end
end
