# frozen_string_literal: true

module Litany
  class SecretPropertyValue < Base

    # @param key [String, Symbol] the key this property value is attached to, used in logging situations
    # @param value [Object] the value to store, ultimately returned as a string when rendered into json
    def initialize(key, value)
      @key = key
      @value = value
    end

    # Non-leaking inspect statement
    def inspect
      "<SecretPropertyValue: Key: #{@key}>"
    end

    # Used when compiling objects to json
    def to_json(*args)
      to_s.to_json(*args)
    end

    # If redaction is enabled at the class level this returns a placeholder, otherwise returns the value as a string
    #   Integrates with the implementation of ArcanaSecret to resolve any secrets
    # @return [String]
    def to_s
      if self.class.redact?
        '<Redacted Secret>'
      else
        @value.respond_to?(:resolve) ? @value.resolve : @value.to_s
      end
    end

    class << self
      # Enables redaction of secret values
      # @return [true]
      def enable_redaction
        @redact = true
      end

      # Disables redaction of secret values
      # @return [false]
      def disable_redaction
        @redact = false
      end

      # Tests if redaction is currently enabled
      # @return [Boolean]
      def redact?
        @redact
      end
    end
  end
end

