# How to Integrate with another VPC

> Pro tip: This guide assumes you're attempting to integrate a VPC that has been created with a CIDR that Twitch's Network team issued.  If this is not the case, that's eh, bad? You should abort and discuss.

## Part 1: Account Whitelisting

In `security-lit`'s `settings.lit` there is a list of trusted accounts.  You first must ensure that the account you wish to peer with is on this list.  Please keep the ids ordered alphabetically by account alias.

```ruby
set :trusted_account_ids, [
  '708209600803', # orko-production
  '182878055170', # syslog-production
  '857171411193', # twilight-production
  '157787312516', # twilight-test
  '001591884629', # twilight-shadow
  '994136867826', # tools-production
  '523910602058', # tools-shadow
]
```

The following command ensures that the whitelisted accounts are converged in AWS.  You are required to execute this if you update the list.
```bash
lit upload -s production -e pdx -S PeeringIAMPermissions
```

## Part 2: Create the Peering Connection

> Important: Part 2 is performed inside of the litany project managing the VPC you're attempting to peer.  The expectation is that this is *not* `security-lit`.

1. Ensure the below line of DSL is in the litany definition of the VPC you're attempting to connect to teleport.  Nothing in this line should change.
    ```ruby
    peer :bastion, vpc: settings.bastion.vpc, cidr: settings.bastion.cidr, account: '103132490357', arn: 'arn:aws:iam::103132490357:role/vpc-peering-authorizer' if settings.bastion?
    ```
1. In that litany project's settings we must define the values for `bastion.cidr` and `bastion.vpc`.  I recommend creating a `bastion.lit` file under the settings folder.  The file should be portable across stages and projects.  In that file you should define a `inside` block for the region you're integrating.
    ```ruby
    inside :pdx do
      set :bastion, vpc: 'vpc-02c32df2e64cfc731', cidr: '10.204.132.0/22'
    end
    ```
    * **VPC**: use the vpc id for that region's security-services VPC.
    * **CIDR**: use the cidr value for that region as defined in `security-lit`'s `settings.lit` file.
1. Perform a `lit upload` command to upload the VPC stack you're integrating with bastion.

## Part 3: Add Routes to Security Services

> Important: Switch back to working in `security-lit` for Part 3.

Finally, we must add return routes from the security services VPC for that region.  In `settings.lit` each regional block should have a list of `peer_routes` defined.

```ruby
  set :peer_routes,
      twilight_shadow: { cidr: '10.201.164.0/22', target: 'pcx-053cf7b4feb5df31f' },
      twilight_guru_shadow: { cidr: '10.201.244.0/22', target: 'pcx-0fd64f8ce373d8930' }
```

Each peering relationship should be represented by a line on this list.  There are three parts to the data needed:

* **Name**: The name must be unique inside the list of routes for that region.  Generally, we should use descriptive names, so we know at a glance what it's referring to.
* **CIDR**: This is the CIDR of the vpc on the other end.  The CIDR can be obtained from either the AWS console or that project's litany configs.
* **Target**: This is the resource ID for the peering connection inside of the security-services account.  The ID can be found under the VPC section in the AWS console.

Perform a `lit upload` command to upload the VPC stack.

## Validation

After all three parts above you should be able to connect to resources inside the remote VPC.  Below is an example SSH command.  You need to replace the ip address with a ***LAN ip*** from the newly connected VPC and make sure your TC variable is pointing at the right teleport remote cluster.
```bash
TC=csi-security-services-pdx ssh 10.201.166.166
```
