import jetbrains.buildServer.configs.kotlin.v2018_2.ParameterDisplay
import jetbrains.buildServer.configs.kotlin.v2018_2.buildFeatures.dockerSupport
import jetbrains.buildServer.configs.kotlin.v2018_2.buildSteps.dockerCommand
import jetbrains.buildServer.configs.kotlin.v2018_2.buildSteps.script
import jetbrains.buildServer.configs.kotlin.v2018_2.version
import twitch.instrumentorum.instrum
import twitch.instrumentorum.project.features.ecrConnection
import java.io.File

version = "2018.2"

instrum {
    description = "A zero-configuration ECR proxy."

    sourceRoot("git@git-aws.internal.justin.tv:CPE-Ops/stevedore.git")

    params {
        text("deploy.environments", "pdx", "Deploy Environments", "Comma separated lists of environment variables to deploy to.")
        text("deploy.service", "stevedore", "ECS Service", "The name of the ECS Service", ParameterDisplay.HIDDEN, readOnly = true)
        text("ecr.repo", "stevedore", "ECR Repository", "The name used the ECR Repository", readOnly = true)
    }

    features {
        ecrConnection {
            id = "AmazonECR"
            name = "Amazon ECR"
        }
    }

    buildGroup {
        build("Initialize Build Chain") {
            steps {
                script {
                    name = "Discover Version Number"
                    scriptContent = File("scripts/git_version_number.rb").readText()
                }
            }
        }

        build("Build Container") {
            features {
                dockerSupport {
                    cleanupPushedImages = true
                    loginToRegistry = on {
                        dockerRegistryId = "AmazonECR"
                    }
                }
            }

            steps {
                dockerCommand {
                    name = "Build Container"

                    commandType = build {
                        source = path {
                            path = "Dockerfile"
                        }

                        namesAndTags = """
                            994136867826.dkr.ecr.us-west-2.amazonaws.com/%ecr.repo%:%build.number%
                            994136867826.dkr.ecr.us-west-2.amazonaws.com/%ecr.repo%:latest
                        """.trimIndent()
                    }
                }

                dockerCommand {
                    name = "Push Container"

                    commandType = push {
                        namesAndTags = """
                            994136867826.dkr.ecr.us-west-2.amazonaws.com/%ecr.repo%:%build.number%
                            994136867826.dkr.ecr.us-west-2.amazonaws.com/%ecr.repo%:latest
                        """.trimIndent()

                        removeImageAfterPush = true
                    }
                }
            }
        }

        build("Deploy to Production") {
            params {
                text("deploy.account", "994136867826", "Deploy Account", "Which account are we deploying to?", ParameterDisplay.HIDDEN, readOnly = true)
                text("deploy.cluster", "services", "ECS Cluster", "Name of the ECS Cluster", ParameterDisplay.HIDDEN, readOnly = true)
                text("deploy.count", "2", "Container Count", "How many containers to deploy.", regex = "^\\d+\$", validationMessage = "Count must be a positive number.")
                text("deploy.role", "", "Deploy IAM Role", "Which IAM role should we assume to do the deploy?", ParameterDisplay.HIDDEN, readOnly = true)
                text("task.role", "StevedoreTaskIAMRole-USWest2-d6d1b019c08ceb6a4261681d70f764a8ad", "Task Role", "Which IAM role to use for the task definition task role.", ParameterDisplay.HIDDEN, readOnly = true)
                text("task.revisions", "{}", "Task Revisions", "A json serialized mapping of environment to task revision ids.", ParameterDisplay.HIDDEN)
            }

            failureConditions {
                executionTimeoutMin = 5
            }

            steps {
                script {
                    name = "Update Task Definition"
                    scriptContent = """
                        #!/usr/bin/env ruby
                        # frozen_string_literal: true

                        ## common.rb
                        ${File("scripts/common.rb").readText()}

                        ## update_task_definition.rb
                        ${File("scripts/update_task_definition.rb").readText()}
                    """.trimIndent()
                }

                script {
                    name = "Update Service"
                    scriptContent = """
                        #!/usr/bin/env ruby
                        # frozen_string_literal: true

                        ## common.rb
                        ${File("scripts/common.rb").readText()}

                        ## update_task_definition.rb
                        ${File("scripts/update_service.rb").readText()}
                    """.trimIndent()
                }

                script {
                    name = "Verify Update"
                    scriptContent = """
                        #!/usr/bin/env ruby
                        # frozen_string_literal: true

                        ## common.rb
                        ${File("scripts/common.rb").readText()}

                        ## update_task_definition.rb
                        ${File("scripts/verify_update.rb").readText()}
                    """.trimIndent()
                }
            }
        }
    }

}
