# frozen_string_literal: true

require_relative 'bootstrap'

class TwilightTest < Minitest::Test
  def twitch
    @twitch ||= Faraday.new(TWITCH_URL) do |conn|
      conn.use :cookie_jar
      conn.adapter Faraday.default_adapter
    end
  end

  def teardown
    sleep 0.25
  end

  # standard routing tests
  Twitch::Route.list.each do |route|
    define_method :"test_path_#{route.path.tr('/', '_')}" do
      res = twitch.get(route.path) do |req|
        req.headers['Cookie'] = 'impact_excluded=1;'
      end

      # verify that we are hitting our Fastly config
      refute_nil res.headers['x-served-by'], 'Expected X-Served-By header to be present'
      assert_match /^cache-[a-z]{3}[0-9]{4}-[A-Z]{3}/, res.headers['x-served-by']

      # verify expected backend
      # TODO: make this check reliable. not currently any consistent header to fully differentiate backends
      refute_nil res.headers['x-backend'], 'Expected X-Backend header to be present'
      assert_equal route.backend, Twitch::Route.actual_backend(res), "Response headers: \n#{res.headers.inspect}" \
        unless route.skip_backend_check

      # verify that we have our expected http status
      assert_equal route.status, res.status, "Unexpected response of #{res.status} on #{route.path}, expected #{route.status}"

      # verify that we have proper location header for 3xx
      unless route.location.nil?
        if res.headers['location'].include?(TWITCH_URL)
          # redirect is relative, should match TWITCH_URL and not www
          assert_includes res.headers['location'], route.location.gsub('https://www.twitch.tv', TWITCH_URL)
        else
          # redirect is not relative and should match the www url defined in routes
          assert_includes res.headers['location'], route.location
        end
      end
    end
  end

  # legacy experiment routing test, staging only until we go live
  if TWITCH_URL == 'https://twilight.twitch.tv'
    Twitch::Route.list.each do |route|
      define_method :"test_path_legacy_experiment_#{route.path.tr('/', '_')}" do
        res = twitch.get(route.path) do |req|
          req.headers['Cookie'] = 'impact_included=1;'
        end

        # verify expected backend
        unless route.skip_backend_check
          backend = :legacy if route.backend == :twilight
          backend ||= route.backend
          assert_equal backend, Twitch::Route.actual_backend(res), "Response headers: \n#{res.headers.inspect}"
        end

        # verify that we have our expected http status
        assert_equal route.status, res.status, "Unexpected response of #{res.status} on #{route.path}, expected #{route.status}"

        # verify that we have proper location header for 3xx
        unless route.location.nil?
          if res.headers['location'].include?(TWITCH_URL)
            # redirect is relative, should match TWITCH_URL and not www
            assert_includes res.headers['location'], route.location.gsub('https://www.twitch.tv', TWITCH_URL)
          else
            # redirect is not relative and should match the www url defined in routes
            assert_includes res.headers['location'], route.location
          end
        end
      end
    end
  end
end
