# frozen_string_literal: true

require 'twitch/audit/models/client_configuration'

describe Twitch::Audit::ClientConfiguration do
  context 'when creating an instance' do
    subject(:no_args) { described_class.new }
    subject(:incomplete) { described_class.new dead_letter_bucket: 'test-bucket' }
    subject(:symbol_for_bucket) { described_class.new dead_letter_bucket: :test, service_name: 'tests', sns_topic_arn: 'arn:aws:sns:us-west-2:123456789012:test-sns' }
    subject(:invalid_arn) { described_class.new dead_letter_bucket: 'test-bucket', service_name: 'tests', sns_topic_arn: 'arn:aws:sns:us-west-2:12345678:test-sns' }
    subject(:minimal_valid) { described_class.new dead_letter_bucket: 'test-bucket', service_name: 'tests', sns_topic_arn: 'arn:aws:sns:us-west-2:123456789012:test-sns' }
    subject(:overridden_valid) { described_class.new dead_letter_bucket: 'test-bucket', service_name: 'tests', sns_topic_arn: 'arn:aws:sns:us-west-2:123456789012:test-sns', submission_interval: 10 }

    it('errors without required attributes') do
      expect { no_args }.to raise_error(Dry::Struct::Error)
      expect { incomplete }.to raise_error(Dry::Struct::Error)
    end

    it('errors if types mismatch') do
      expect { symbol_for_bucket }.to raise_error(Dry::Struct::Error)
      expect { invalid_arn }.to raise_error(Dry::Struct::Error)
    end

    it('has correct default values') do
      expect { minimal_valid }.to_not raise_error
      expect(minimal_valid.submission_batch_size).to equal(10)
      expect(minimal_valid.submission_timeout).to equal(30)
      expect(minimal_valid.submission_interval).to equal(1)
    end

    it('has correct overridden values') do
      expect { overridden_valid }.to_not raise_error
      expect(overridden_valid.submission_interval).to_not equal(1)
      expect(overridden_valid.submission_interval).to equal(10)
    end
  end
end
