
vcl 4.0;

import awsrest;
import std;
import var;

acl settings_ips {
  "localhost";
  "127.0.0.1";
  "172.16.0.0"/12; # Docker was using this range, needed for development.
}

probe healthcheck {
  .url = "/vicarius-public-healthcheck.html";
  .window = 5;
  .threshold = 3;
  .interval = 2s;
  .timeout = 1s;
}

sub no_healthy_backend {
  # This sub is used by the stage files to tell us there's no healthy backends.
  var.global_set("unhealthy_backends", true);
}

sub vcl_init {
  # We assume backends are healthy until we can prove otherwise.
  var.global_set("unhealthy_backends", false);
}

sub vcl_recv {
  # This is a kludge to allow us to update the access keys for signing on an ongoing basis.
  if (req.method == "SETTINGS" && client.ip ~ settings_ips) {
    var.global_set("access_key", req.http.X-Access-Key);
    var.global_set("secret_key", req.http.X-Secret-Key);
    var.global_set("session_token", req.http.X-Session-Token);
    var.global_set("last_key_update", now);

    return(synth(200, "Negative. I am a meat popsicle."));
  }

  if (req.method != "GET" && req.method != "HEAD" && req.method != "OPTIONS") {
    return(synth(405, "Method Not Allowed"));
  }

  if (req.http.transfer-encoding) {
   return(fail);
  }

  if (req.url ~ "^([^\?])+\.(css|js)\.map(\?.*)?$" || req.url ~ "^/(\?.*)?$") {
    return(synth(404, "Not Found"));
  }

  # We assume there's a healthy backend, and rely on pick_backend to tell us otherwise.
  var.global_set("unhealthy_backends", false);
  call pick_backend;

  if (req.url ~ "^/debug/running") {
    if (var.global_get("unhealthy_backends") == "true" || (now - std.time(var.global_get("last_key_update"), now - 10y) > 30m) ) {
      # Either we can't reach _any_ of our backends or the last key update is over 30 minutes ago.
      # Result? Seppuku.
      return(synth(503, "AWWW No!"));
    }

    set req.url = "/vicarius-private-healthcheck.html";
  }

  # Cleans up and standardizes the URL to protect against a few bad things.
  set req.url = awsrest.formurl(std.querysort(req.url));
}

sub vcl_hash {
  # Varies based on our settings for the CDN.
  # Cache key automagically has URL and HOST in it.
  hash_data(req.http.Access-Control-Request-Headers);
  hash_data(req.http.Access-Control-Request-Method);
  hash_data(req.http.Origin);
}

sub vcl_backend_fetch {
  # We sign the request, for fun and profit.
  unset bereq.http.x-amz-cf-id;
  set bereq.http.host = var.global_get("target_host");

  awsrest.v4_generic(
    service           = "s3",
    region            = var.global_get("target_region"),
    access_key        = var.global_get("access_key"),
    secret_key        = var.global_get("secret_key"),
    token             = var.global_get("session_token"),
    signed_headers    = "host;",
    canonical_headers = "host:" + bereq.http.host + awsrest.lf()
  );
}

sub vcl_backend_response {
  set beresp.http.X-Vicarius-Origin = beresp.backend.name;
  set beresp.http.X-Vicarius-Region = std.getenv("VICARIUS_REGION");

  # By maniuplating the request url here we are able to leverage the cache control overrides natively.  Meaning this is currently a 60s cache because it gets seen as "/config/settings.js".
  if (beresp.status == 404 && bereq.url ~ "(?i)^/config/settings.*([0-9a-f]{31,32})\.js(on)?(?:\?.*)?$") {
    set bereq.url = regsub(bereq.url, "(?i)^/config/settings(.*)(?:\.[a-fA-F0-9]{31,32})\.js(on)?(?:\?.*)?$", "/config/settings\1.js\2");
    std.log("New URL is: " + bereq.url);
    return(retry);
  }

  // Support Integration Test for 5XX Retries
  if (bereq.url ~ "^/debug/return_502") {
    set beresp.status = 502;
  }

  # On fetch error, retry backend 3 times
  if (beresp.status >= 500) {
    if (bereq.retries < 3) {
      set beresp.http.X-Vicarius-Retries = bereq.retries;
      return(retry);
    } else {
      set beresp.http.X-Vicarius-Retries = bereq.retries;
      set beresp.http.Cache-Control = "public, max-age=1";
      set beresp.ttl = 1s;
      set beresp.grace = 5s;
      return (deliver);
    }
  }

  # Serve Stale
  # Docs: https://varnish-cache.org/docs/5.0/users-guide/vcl-grace.html
  set beresp.grace = 24h;

  # Cache Control Overrides
  if (bereq.url ~ "/vicarius-private-healthcheck.html") {
    set beresp.http.Cache-Control = "public, max-age=10";
    set beresp.ttl = 10s;
    set beresp.grace = 10m;
  } elsif (bereq.url ~ "(?i)^/config/settings.*(?<![0-9a-f]{32})\.js(on)?$" || (bereq.url ~ "(?i)^/player/" && bereq.url !~ "(?i)^/player/(js|css)/player\.[^.]+\.(css|js)")) {
    set beresp.http.Cache-Control = "public, max-age=60";
    set beresp.ttl = 1m;
  } elsif (bereq.url ~ "(?i)^/localization/" || bereq.url ~ "(?i)^/config/manifest(\.(development|test))?\.json(\?.*)?$") {
    set beresp.http.Cache-Control = "public, max-age=300";
    set beresp.ttl = 5m;
  } else {
    set beresp.http.Cache-Control = "public, max-age=31536000, immutable";
    set beresp.ttl = 365d;
  }

  # Compression stuff
  if ((beresp.status == 200 || beresp.status == 404) && (beresp.http.content-type ~ "^(text\/html|application\/x\-javascript|text\/css|application\/javascript|text\/javascript|application\/json|application\/vnd\.ms\-fontobject|application\/x\-font\-opentype|application\/x\-font\-truetype|application\/x\-font\-ttf|application\/xml|font\/eot|font\/opentype|font\/otf|image\/svg\+xml|image\/vnd\.microsoft\.icon|text\/plain|text\/xml|application\/wasm)\s*($|;)" || bereq.url ~ "\.(css|js|html|eot|ico|otf|ttf|json|svg|wasm)($|\?)" ) ) {
    # always set vary to make sure uncompressed versions dont always win
    if (!beresp.http.Vary ~ "Accept-Encoding") {
      if (beresp.http.Vary) {
        set beresp.http.Vary = beresp.http.Vary + ", Accept-Encoding";
      } else {
         set beresp.http.Vary = "Accept-Encoding";
      }
    }
    if (bereq.http.Accept-Encoding == "gzip") {
      set beresp.do_gzip = true;
    }
  }
}

sub vcl_deliver {
  # We replace 404s and 403s with a ugly page to limit information leaks.
  if (resp.status == 404) {
    return(synth(404, "Not Found"));
  }

  if (resp.status == 403) {
    return(synth(403, "Verboten"));
  }

  # Debug headers
  set resp.http.X-Vicarius-TransactionID = req.xid;
  if (obj.hits > 0) {
    set resp.http.X-Vicarius-Hits = obj.hits;
    set resp.http.X-Vicarius-TTL = obj.age + "/" + obj.ttl;
  } else {
    set resp.http.X-Vicarius-Hits = "-1";
  }

  # Remove any set by the backend.
  unset resp.http.Access-Control-Allow-Origin;
  unset resp.http.Timing-Allow-Origin;

  # Allow Origin headers
  if (req.http.Origin ~ "(?i)^https?://(?:(?:\w[\w\.-]+)\.)?twitch\.tv(:\d+)?$" || req.http.Origin ~ "(?i)^https?://(.*\.)?twitch-shadow\.net(:\d+)?$" || req.http.Origin ~ "(?i)^https?://(.*\.)?twitch\.tech(:\d+)?$" || req.http.Origin ~ "(?i)^https?://(.*\.)?sage\.xarth\.tv(:\d+)?$" || req.http.Origin ~ "(?i)https?://(localhost|127\.0\.0\.1)(:\d+)?$") {
    # set resp.http.Timing-Allow-Origin = req.http.Origin;
    # set resp.http.Access-Control-Allow-Origin = req.http.Origin;

    // Temp setup in prep for 10/30, will revert after www change out.
    set resp.http.Access-Control-Allow-Origin = "*";
    set resp.http.Timing-Allow-Origin = "*";
  } elsif (!req.http.Origin && req.url ~ "(?i)^([^\?])+\.(css|js)(\?.*)?$") {
    set resp.http.Access-Control-Allow-Origin = "*";
    set resp.http.Timing-Allow-Origin = "*";
  }

  # More information leak stopping.
  unset resp.http.Server;
  unset resp.http.Via;
  unset resp.http.X-Varnish;
}

sub vcl_synth {
  # Ugly custom error pages.
  unset resp.http.Server;
  unset resp.http.Via;
  unset resp.http.X-Varnish;

  set resp.http.Content-Type = "text/html; charset=utf-8";
  set resp.http.Retry-After = "5";
  set resp.http.X-TransactionID = req.xid;

  set resp.body = "<!DOCTYPE html><html><head><title>" + resp.status + " " + resp.reason + "</title></head><body><h1>" + resp.status + " " + resp.reason + "</h1><p>You've found yourself in an unprecedented situation. To the east is a long and dark corridor, to the west is a field of marigolds. Which way do you wish to go?</p></body></html>";
  return(deliver);
}
