# frozen_string_literal: true

require_relative 'bootstrap'
require 'minitest/autorun'

class OverridesTest < Minitest::Test
  @@protocols = %i[http https].freeze
  @@twitch = @@protocols.map { |protocol| [protocol, Twitch::Web.create(protocol)] }.to_h

  def teardown
    sleep 0.5
  end

  def test_logged_in_beta_redirect_with_cookie
    10.times do
      res = @@twitch[:https].get('/') do |req|
        req.headers['Cookie'] = 'api_token=123; twilight_beta=1'
      end
      assert_equal 302, res.status
      assert_equal res.headers['location'], 'https://go.twitch.tv/'
      sleep 0.5
    end
  end

  def test_logged_in_beta_no_redirect_with_opt_out_cookie
    10.times do
      res = @@twitch[:https].get('/') do |req|
        req.headers['Cookie'] = 'api_token=123; twilight_beta=1; twilight_beta_opt_out=1'
      end
      assert_equal 200, res.status
      sleep 0.5
    end
  end

  def test_forced_beta_redirect_with_cookie
    res = @@twitch[:https].get('/') do |req|
      req.headers['Cookie'] = 'twilight_beta_force_redirect=1'
    end
    assert_equal 302, res.status
    assert_equal res.headers['location'], 'https://go.twitch.tv/'
  end

  def test_force_beta_redirect_with_query_string
    res = @@twitch[:https].get('/') do |req|
      req.params['beta-redirect'] = 'true'
    end
    assert_equal 302, res.status
    assert_equal res.headers['location'], 'https://go.twitch.tv/?beta-redirect=true'
  end

  def test_forced_no_redirect_with_cookie
    res = @@twitch[:https].get('/') do |req|
      req.headers['Cookie'] = 'twilight_beta_force_redirect=1; twilight_beta_opt_out=1'
    end
    assert_equal 200, res.status
  end

  def test_forced_no_redirect_with_query_string
    res = @@twitch[:https].get('/') do |req|
      req.headers['Cookie'] = 'twilight_beta_force_redirect=1'
      req.params['no-beta-redirect'] = 'true'
    end
    assert_equal 200, res.status
  end

  def test_mobile_redirect_precedence
    res = @@twitch[:https].get('/') do |req|
      req.headers[:user_agent] = MOBILE_USER_AGENT
      req.headers['Cookie'] = 'twilight_beta_force_redirect=1'
    end
    assert_equal 302, res.status
    assert_equal res.headers['location'], 'https://m.twitch.tv/?desktop-redirect=true'
  end

  def test_no_ps4_redirect
    res = @@twitch[:https].get('/') do |req|
      req.headers[:user_agent] = PS4_USER_AGENT
      req.headers['Cookie'] = 'twilight_beta_force_redirect=1'
    end
    assert_equal 200, res.status
  end

  def test_no_safari_11_redirect
    SAFARI_11_USER_AGENTS.each do |safari_ua|
      res = @@twitch[:https].get('/') do |req|
        req.headers[:user_agent] = safari_ua
        req.headers['Cookie'] = 'twilight_beta_force_redirect=1'
      end
      assert_equal 200, res.status
      sleep 0.5
    end
  end

  BOTS.each do |bot|
    define_method :"test_no_beta_redirect_for_#{bot}" do
      res = @@twitch[:https].get('/lirik') do |req|
        req.headers[:user_agent] = bot
        req.headers['Cookie'] = 'twilight_beta_force_redirect=1'
      end
      assert_equal 200, res.status, "Unexpected response of #{res.status} for #{bot} bot, expected 200"
    end
  end
end
