﻿using System;
using System.Net;
using System.Threading;
using System.Diagnostics;
using Curse.Friends.Enums;
using Curse.Friends.WebService.Requests;
using Curse.Logging;
using Curse.LoginService.RestClient;
using System.Security;
using Curse.ReportingService.RestClient;
using Curse.ServiceAuthentication.Models;
using Curse.Friends.RestClient;
using Curse.Friends.WebService.Responses;
using Curse.Voice.RestClient;
using Curse.Voice.Service.Requests;
using Curse.ClientService.Models;

namespace Curse.Cerium.Tests
{
    class Program
    {
        private static readonly Guid MachineKey = new Guid("79e8d2b3-f7af-49c8-aa3b-476a5eac46d2");

        static SecureString GetPassword()
        {
            var pwd = new SecureString();
            while (true)
            {
                var i = Console.ReadKey(true);
                if (i.Key == ConsoleKey.Enter)
                {
                    break;
                }
                else if (i.Key == ConsoleKey.Backspace)
                {
                    if (pwd.Length > 0)
                    {
                        pwd.RemoveAt(pwd.Length - 1);
                        Console.Write("\b \b");
                    }
                }
                else
                {
                    pwd.AppendChar(i.KeyChar);
                    Console.Write("*");
                }
            }
            return pwd;
        }

        static void TestBugReporting()
        {
         
            var sampleArray = new byte[1024];

            var resp = ReportingServiceRestClient.Instance.SubmitBugReport(new BugReport
            {
                Title = "test",
                ClientVersion = new Version(6, 1, 0, 0),
                Platform = BugReportPlatform.Mac,
                Type = BugReportType.BugOrDefect,
                UsernameOrEmail = "mcomperda@curse.com",
                Attachments = new[] { new BugReportAttachment { FileContents = sampleArray, FileName = "Test.zip"} }
            });
        }

        private static void TestUsageProfile()
        {
            
            ReportingServiceRestClient.Instance.SaveUsageProfile(new UsageProfile { ClientVersion = "6.1.0.0", Platform = UsageProfilePlatform.iOS, CountryCode = "US" });
        }

        private static void TestUserReport()
        {

            UserReportingServiceRestClient.Instance.ReportUserAbuse(new UserAbuseReportRequest {Reason = UserAbuseReportReason.Harassment, ReportDescription = "Test", ReportedUserID = 128, ReportingUserPlatform = ClientPlatform.iPhone});
        }


        static void Main(string[] args)
        {            
            CeriumConfiguration.DisableCertificateValidation();

            Logger.Init(@"C:\Curse\Logs\", "CeriumTests");            

            ReportingServiceRestClient.Initialize("https://clientservice-v6.curse.dev/ClientReportingService.svc/json");
            UserReportingServiceRestClient.Initialize("https://clientservice-v6.curse.dev/ClientUserReportingService.svc/json");

            TestBugReporting();            

            Console.Write("Enter your username: ");
            string username = Console.ReadLine();
            Console.Write("Enter your password: ");
            var password = GetPassword();

            var credentials = new NetworkCredential(username, password);

            Console.Clear();
            Console.Write("Logging in...");

            LoginResponse loginResponse;

            try
            {
                LoginServiceRestClient.Initialize("https://clientservice-v6.curse.dev/ClientLoginService.svc/json/");
                

                var isAvailable = LoginServiceRestClient.Instance.CheckUsernameAvailability(credentials.UserName);
                Console.WriteLine("Username Available: " + isAvailable);

                var register = LoginServiceRestClient.Instance.Register("tester@curse.com", "tester", "test");
                Console.WriteLine("Registration Status: " + register.Status);

                loginResponse = LoginServiceRestClient.Instance.Login(credentials.UserName, credentials.Password);
                if (loginResponse.Status != AuthenticationStatus.Success)
                {
                    Console.WriteLine("Login failed: " + loginResponse.Status);
                    Console.ReadLine();
                    return;
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("Login failed: " + ex.Message);
                Console.ReadLine();
                return;
            }

            Console.WriteLine("Success!");

            ReportingServiceRestClient.SetAuthToken(loginResponse.Session.UserID, loginResponse.Session.Token);
            TestUsageProfile();

            UserReportingServiceRestClient.SetAuthToken(loginResponse.Session.UserID, loginResponse.Session.Token);
            TestUserReport();
            

            FriendsRestClient.Initialize("https://friends-service.cursevoice.stg/FriendsService.svc/json/");
            FriendsRestClient.SetAuthToken(loginResponse.Session.UserID, loginResponse.Session.Token);

            var hostlist = FriendsRestClient.Instance.GetHostList();

            VoiceRestClient.Initialize("https://voice-service.cursevoice.stg/VoiceService.svc/json/");
            VoiceRestClient.SetAuthToken(loginResponse.Session.UserID, loginResponse.Session.Token);
            
            var resp = FriendsRestClient.Instance.CallFriend(new CallFriendRequest { ClientVersion = "6.2.0.0", SendInvitation = false, FriendID = 128, Force = true });

            if (resp.Status == VoiceSessionRequestStatus.Successful)
            {
                var voiceResp = VoiceRestClient.Instance.GetVoiceSession(new GetVoiceSessionRequest { ClientVersion = "6.2.0.0", InviteCode = resp.InviteUrl });
            }
            
            Register();
            
            TestFriends();

            Console.ReadLine();
        }


        private static void TestFriends()
        {

            var friends = FriendsRestClient.Instance.GetMyFriends();
            var profile = FriendsRestClient.Instance.GetUserProfile(1);
            var changeProfile =
                FriendsRestClient.Instance.ChangeProfile(new ChangeProfileRequest
                {
                    AboutMe = profile.AboutMe,
                    AvatarUrl = profile.AvatarUrl,
                    City = profile.City,
                    CountryCode = profile.CountryCode,
                    Name = profile.Name,
                    State = profile.State
                });

            var changeStatus = FriendsRestClient.Instance.ChangeStatus(new ChangeStatusRequest
                                    {
                                        MachineKey = MachineKey,
                                        Status = UserConnectionStatus.Away
                                    });

            changeStatus = FriendsRestClient.Instance.ChangeStatus(new ChangeStatusRequest
            {
                MachineKey = MachineKey,
                Status = UserConnectionStatus.Online
            });

            var groups = FriendsRestClient.Instance.GetMyGroups();
            foreach (var group in groups.Groups)
            {
                var groupDetails = FriendsRestClient.Instance.GetGroupDetails(new GroupDetailsRequest { GroupID = group.GroupID });
                if (groupDetails.Status == BasicServiceResponseStatus.Successful)
                {
                    var toggleFavorite = FriendsRestClient.Instance.ToggleFavoriteGroup(new FavoriteGroupRequest { GroupID = group.GroupID });
                    toggleFavorite = FriendsRestClient.Instance.ToggleFavoriteGroup(new FavoriteGroupRequest { GroupID = group.GroupID });

                    var changeGroup = FriendsRestClient.Instance.ChangeGroupInfo(new ChangeGroupRequest
                    {
                        GroupID = group.GroupID,
                        Title = group.GroupTitle
                    });

                    var mute = FriendsRestClient.Instance.ChangeGroupNotificationPreferences(new ChangeGroupNotificationPreferencesRequest
                    {
                        GroupID = group.GroupID,
                        Preference = NotificationPreference.Disabled
                    });

                    mute = FriendsRestClient.Instance.ChangeGroupNotificationPreferences(new ChangeGroupNotificationPreferencesRequest
                    {
                        GroupID = group.GroupID,
                        Preference = NotificationPreference.Enabled
                    });
                }
            }

            var friendRequest = FriendsRestClient.Instance.RequestFriendship(new FriendshipRequest
            {
                FriendID = 10,
                InvitationMessage = "Test",
                IsFromSuggestion = false,
                KnownIdentity = "Bryan McLemore"
            });

            var renameFrame = FriendsRestClient.Instance.RenameFriend(new RenameFriendRequest { FriendID = 10, Nickname = "Kaelten" });


            var confirm = FriendsRestClient.Instance.ConfirmFriendship(new ConfirmFriendshipRequest
            {
                FriendID = 10
            });

            var decline = FriendsRestClient.Instance.DeclineFriendship(new DeclineFriendshipRequest { FriendID = 10 });

            var find = FriendsRestClient.Instance.FindFriends(new FriendSearchRequest { QueryString = "kaelten" });

            var createGroup =
                FriendsRestClient.Instance.CreateGroup(new CreateGroupRequest
                {
                    AccessLevel = GroupRole.Guest,
                    Title = "My Test Group",
                    RecipientsUserIDs = new[] { 128 },
                    Type = GroupType.Normal
                });

            var leaveGroup = FriendsRestClient.Instance.LeaveGroup(new LeaveGroupRequest { GroupID = createGroup.GroupID });
        }

        private static void Register()
        {


            Console.WriteLine("Registering self...");
            var sw = Stopwatch.StartNew();
            var resp = FriendsRestClient.Instance.RegisterSelf(new RegisterSelfRequest { MachineKey = MachineKey, Platform = DevicePlatform.Windows });
            sw.Stop();
            Console.WriteLine("Register self " + sw.ElapsedMilliseconds);

            Console.WriteLine();
            Console.WriteLine("Getting frirends...");
            sw.Restart();
            var friends = FriendsRestClient.Instance.GetMyFriends();
            sw.Stop();
            Console.WriteLine("Retrieved " + friends.Friends.Length + " friends in " + sw.ElapsedMilliseconds);

            Console.WriteLine();
            Console.WriteLine("Getting frirends...");
            sw.Restart();
            var groups = FriendsRestClient.Instance.GetMyGroups();
            sw.Stop();
            Console.WriteLine("Retrieved " + groups.Groups.Length + " groups in " + sw.ElapsedMilliseconds);

            sw.Restart();
            foreach (var group in groups.Groups)
            {
                var groupDetails = FriendsRestClient.Instance.GetGroupDetails(new GroupDetailsRequest { GroupID = group.GroupID });
            }
            sw.Stop();
            Console.WriteLine("Retrieved " + groups.Groups.Length + " group details in " + sw.ElapsedMilliseconds);

            Console.WriteLine();
            Console.WriteLine("Getting push prefs...");
            sw.Restart();
            var pushPrefs = FriendsRestClient.Instance.GetMyPushNotificationPreferences();
            sw.Stop();
            Console.WriteLine("Retrieved " + groups.Groups.Length + " groups in " + sw.ElapsedMilliseconds);

            sw.Restart();
            var pushResp = FriendsRestClient.Instance.ChangePushNotificationPreferences(new ChangePushNotificationPreferencesRequest
            {
                FriendMessagePushPreference = PushNotificationPreference.All,
                GroupMessagePushPreference = PushNotificationPreference.Favorites,
                FriendRequestPushEnabled = true
            });
            sw.Stop();
            Console.WriteLine("Updated push notification prefs in " + sw.ElapsedMilliseconds);


            Console.WriteLine("-------------------------------------------------------");
        }
    }
}
