﻿using System;
using System.Collections.Specialized;
using System.Net.Http;
using System.Net.Http.Headers;
using Newtonsoft.Json;
using System.Collections.Generic;
using System.Text;

namespace Curse.Cerium
{
    public abstract class CeriumRestClient<T> where T : class
    {
        public static CeriumConfiguration Configuration = new CeriumConfiguration();                

        protected static string ServiceBaseUrl;

        protected virtual HttpClient GetHttpClient()
        {
            HttpClientHandler handler;
            if(Configuration.CustomHttpHandler != null)
            {
                handler = (HttpClientHandler)Activator.CreateInstance(Configuration.CustomHttpHandler);    
            } else
            {
                handler = new HttpClientHandler();
            }
            handler.UseProxy = !Configuration.DisableProxyDetection;

            var httpClient = new HttpClient(handler) { Timeout = TimeSpan.FromSeconds(Configuration.TimeoutSeconds) } ;
            if(Configuration.UserAgent != null)
            {
                httpClient.DefaultRequestHeaders.TryAddWithoutValidation("User-Agent", Configuration.UserAgent);
            }
            if(Configuration.ContentType != null)
            {
                httpClient.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue(Configuration.ContentType));
            }
            
            var customHeaders = GetCustomHeaders();
            if (customHeaders != null)
            {
                foreach(var key in customHeaders.AllKeys)
                {
                    httpClient.DefaultRequestHeaders.TryAddWithoutValidation(key, customHeaders[key]);
                }
            }
            
            return httpClient;
        }

        private static string GetUrl(string action)
        {
            return ServiceBaseUrl + action;
        }

        protected virtual TResponse Post<TRequest, TResponse>(TRequest body, string action = null)
        {
            if (action == null)
            {
                action = typeof (TRequest).Name;
                if (!action.EndsWith("Request"))
                {
                    throw new ArgumentException("Action cannot be inferred from request type!", "action");
                }
                action = action.Substring(0, action.LastIndexOf("Request", StringComparison.InvariantCulture));
            }                        

            string json;

            using (new SimpleTimer(action, "Serialization"))
            {
                json = JsonConvert.SerializeObject(body);
            }
            
            HttpResponseMessage response = null;
            var url = GetUrl(action);
            using (new SimpleTimer(action, "Request to " + url))
            {
                using (var httpClient = GetHttpClient())
                {
                    response = httpClient.PostAsync(url, new StringContent(json, Encoding.UTF8, Configuration.ContentType)).Result;                    
                }
            }
           
            TResponse value;
            using (new SimpleTimer(action, "Deserialization"))
            {
                var s = response.Content.ReadAsStringAsync().Result;
                value = JsonConvert.DeserializeObject<TResponse>(s);
            }

            return value;
        }
        
        protected virtual TResponse Get<TResponse>(string action)
        {
            using (var httpClient = GetHttpClient())
            {
                var url = GetUrl(action);
                string response;
                using (new SimpleTimer(action, "Request to " + url))
                {
                    response = httpClient.GetStringAsync(url).Result;
                }

                TResponse value;
                using (new SimpleTimer(action, "Deserialization"))
                {
                    value = JsonConvert.DeserializeObject<TResponse>(response);
                }

                return value;
            }
        }
        
        protected virtual NameValueCollection GetCustomHeaders()
        {
            return null;
        }
    }
}
