﻿using System.Diagnostics;
using System.Linq;
using System.Runtime.InteropServices;
using Curse.Aerospike;
using Curse.CloudServices.Jobs;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Statistics;
using Curse.Friends.Statistics.Models;
using Curse.Logging;
using System;
using System.Collections.Generic;
using Curse.Extensions;
using Aerospike.Client;

namespace Curse.Friends.Jobs
{
    public class PeriodicStatisticsJob : BaseJob
    {

        private static readonly LogCategory Logger = new LogCategory("PeriodicStatisticsJob");

        public override JobScheduleMode ScheduleMode
        {
            get 
            {
                return JobScheduleMode.Delegate;
            }
        }
            
        public override bool IsScheduled(DateTime lastRunTime)
        {

#if DEBUG
            return true;
#endif

            if(DateTime.UtcNow.Subtract(lastRunTime).TotalMinutes >= 30)
            {
                return true;
            }

            if (DateTime.UtcNow.Subtract(lastRunTime).TotalMinutes < 2) // If it ran in the last minute, we don't need to run it again!
            {
                return false;
            }

            var minute = DateTime.UtcNow.Minute;
            return minute == 0 || minute == 15 || minute == 30 || minute == 45;
        }

        public override void Run()
        {
            var startDate = FriendsStatsManager.GetQuarter().AddMinutes(-15);

            foreach (var config in AerospikeConfiguration.Configurations)
            {
                RunForRegion(config, startDate);

                Logger.Info("Garbage collecting...");
                GC.Collect(0, GCCollectionMode.Forced, true);
                Logger.Info("Finished garbage collection!");
            }            
        }

        private void RunForRegion(AerospikeConfiguration config, DateTime startDate)
        {
            Logger.Info("Calculating stats for: " + config.RegionKey + " with a start date of " + startDate);

            var stats = new FriendsRegionStats
            {
                RegionID = config.RegionIdentifier,
                StartDate = startDate
            };

            try
            {
                TimeStats("friend suggestion", CalculateFriendSuggestionStats, config, stats);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to calculate friend suggestion stats: " + config.RegionKey);
                return;
            }

            try
            {
                TimeStats("friendship", CalculateFriendStats, config, stats);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to calculate friend stats for region: " + config.RegionKey);
                return; 
            }

            try
            {
                TimeStats("user", CalculateUserStats, config, stats);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to calculate user stats: " + config.RegionKey);
                return;
            }


            if (config == AerospikeConfiguration.Configurations.FirstOrDefault(p => p.IsLocal))
            {
                try
                {
                    TimeStats("friend hint", CalculateFriendHintStats, config, stats);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to calculate friend hint stats for region: " + config.RegionKey);
                    return;
                }
            }

            Logger.Info("Done with stats for: " + config.RegionKey);
            FriendsStatsManager.AddRegionStats(stats);
        }

        private static void TimeStats(string statsName, Action<AerospikeConfiguration, FriendsRegionStats> statsAction, AerospikeConfiguration configuration, FriendsRegionStats stats)
        {
            Logger.Info("Calculating " + statsName + " stats...");
            var sw = Stopwatch.StartNew();
            statsAction(configuration, stats);
            sw.Stop();
            Logger.Info("Finished " + statsName + " stats in " + sw.Elapsed.TotalSeconds.ToString("###,##0.00") + " seconds");
        }

        private void CalculateFriendStats(AerospikeConfiguration config, FriendsRegionStats stats)
        {
            Logger.Info("Running query 'calculateFriendshipStats'...");
            var result = Friendship.TryGetAggregatResult(3, config, p => p.IndexMode, IndexMode.Default, "statistics", "calculateFriendshipStats", new Value.IntegerValue(config.RegionIdentifier));
            Logger.Info("Finished query 'calculateFriendshipStats'");

            stats.TotalPendingFriendRequests = Convert.ToInt64(result["pending"]);
            stats.TotalFavorites = Convert.ToInt64(result["favorites"]);            

            if (result.ContainsKey("confirmed"))
            {
                stats.TotalFriendships = Convert.ToInt64(result["confirmed"]);
            }

            if (result.ContainsKey("differentRegion"))
            {
                stats.TotalOutOfRegionFriends = Convert.ToInt64(result["differentRegion"]);
            }
                  
            Logger.Info("Friend stats completed for " + config.RegionKey, result);
        }

        private void CalculateFriendHintStats(AerospikeConfiguration config, FriendsRegionStats stats)
        {
            Logger.Info("Running query 'calculateFriendHintStats'...");
            var result = FriendHint.TryGetAggregatResult(3, config, p => p.IndexMode, IndexMode.Default, "statistics", "calculateFriendHintStats");
            Logger.Info("Finished query 'calculateFriendHintStats'");

            stats.IdentitiesByGame = ConvertDictionary(result["gameHints"], Convert.ToInt32, Convert.ToInt64);
            stats.IdentitiesByPlatform = ConvertDictionary(result["platformHints"], p => (FriendPlatform)Convert.ToInt32(p), Convert.ToInt64);                                 
        }

        private void CalculateUserStats(AerospikeConfiguration config, FriendsRegionStats stats)
        {
            Logger.Info("Running query 'calculateUserStats'...");
            var dict = User.TryGetAggregatResult(3, config, p => p.IndexMode, IndexMode.Default, "statistics", "calculateUserStats");
            Logger.Info("Finished query 'calculateUserStats'");

            stats.UsersWithFriends = Convert.ToInt32(dict["usersWithFriends"]);            
            stats.MaxFriendsPerUser = Convert.ToInt32(dict["maxFriends"]);
            stats.UsersWithProfiles = Convert.ToInt32(dict["userProfileCount"]);
            stats.TotalUsers = Convert.ToInt64(dict["total"]);
            stats.AverageFriendsPerUser = Convert.ToDouble(dict["overallAverageFriends"]);
            stats.AverageFriendsPerEngagedUser = Convert.ToDouble(dict["enagedUsersAverageFriends"]);

            var usersByGame = (Dictionary<object, object>)dict["gameCounts"];
            stats.UserCountByCurrentGame = new Dictionary<int, long>();
            foreach (var kvp in usersByGame)
            {
                stats.UserCountByCurrentGame.Add(Convert.ToInt32(kvp.Key), Convert.ToInt64(kvp.Value));
            }

            var usersStatus = (Dictionary<object, object>)dict["statusCounts"];
            stats.UserCountByStatus = new Dictionary<UserConnectionStatus, long>();
            foreach (var kvp in usersStatus)
            {
                stats.UserCountByStatus.Add((UserConnectionStatus)Convert.ToInt32(kvp.Key), Convert.ToInt64(kvp.Value));
            }

        }

        private static Dictionary<TKey, TValue> ConvertDictionary<TKey, TValue>(object value, Func<object, TKey> keyConverter, Func<object, TValue> valueConverter)
        {
            var originalDictionary = (Dictionary<object, object>)value;
            var newDictionary = new Dictionary<TKey, TValue>();
            foreach (var kvp in originalDictionary)
            {
                newDictionary.Add(keyConverter(kvp.Key), valueConverter(kvp.Value));
            }
            return newDictionary;
        }        

        private void CalculateFriendSuggestionStats(AerospikeConfiguration config, FriendsRegionStats stats)
        {
            Logger.Info("Running query 'calculateFriendSuggestionStats'...");
            var result = FriendSuggestion.TryGetAggregatResult(3, config, p => p.IndexMode, IndexMode.Default, "statistics", "calculateFriendSuggestionStats");
            Logger.Info("Finished query 'calculateFriendSuggestionStats'");
            
            stats.TotalFriendSuggestions = Convert.ToInt32(result["total"]);
            stats.FriendSuggestionsByStatus = ConvertDictionary(result["statusCounts"], Convert.ToInt32, Convert.ToInt32);
            stats.FriendSuggestionsByType = ConvertDictionary(result["typeCounts"], Convert.ToInt32, Convert.ToInt32);
            

            var platformStats = ConvertDictionary(result["platformStats"], Convert.ToInt32, o => ConvertDictionary(o, Convert.ToString, Convert.ToInt32));
            stats.FriendSuggestionsByPlatform = new Dictionary<int, int>();
            stats.FriendSuggestionsAcceptedByPlatform = new Dictionary<int, int>();
            stats.FriendSuggestionsPendingByPlatform = new Dictionary<int, int>();
            stats.FriendSuggestionsDeclinedByPlatform = new Dictionary<int, int>();

            foreach (var stat in platformStats)
            {
                stats.FriendSuggestionsByPlatform[stat.Key] = stat.Value.GetValueOrDefault("total");
                stats.FriendSuggestionsAcceptedByPlatform[stat.Key] = stat.Value.GetValueOrDefault("acceptedCount");
                stats.FriendSuggestionsPendingByPlatform[stat.Key] = stat.Value.GetValueOrDefault("pendingCount");
                stats.FriendSuggestionsDeclinedByPlatform[stat.Key] = stat.Value.GetValueOrDefault("declinedCount");
                    
            }

            var gameStats = ConvertDictionary(result["gameStats"], Convert.ToInt32, o => ConvertDictionary(o, Convert.ToString, Convert.ToInt32));
            stats.FriendSuggestionsByGame = new Dictionary<int, int>();
            stats.FriendSuggestionsAcceptedByGame = new Dictionary<int, int>();
            stats.FriendSuggestionsPendingByGame = new Dictionary<int, int>();
            stats.FriendSuggestionsDeclinedByGame = new Dictionary<int, int>();

            foreach (var stat in gameStats)
            {
                stats.FriendSuggestionsByGame[stat.Key] = stat.Value.GetValueOrDefault("total");
                stats.FriendSuggestionsAcceptedByGame[stat.Key] = stat.Value.GetValueOrDefault("acceptedCount");
                stats.FriendSuggestionsPendingByGame[stat.Key] = stat.Value.GetValueOrDefault("pendingCount");
                stats.FriendSuggestionsDeclinedByGame[stat.Key] = stat.Value.GetValueOrDefault("declinedCount");
            }            
        }
    }
}
