﻿using Microsoft.WindowsAzure.Storage.Table;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using TestApp.Extensions;

namespace TestApp.Models
{
    public class BaseEntity : TableEntity
    {
        public BaseEntity() { }
        public BaseEntity(string partitionKey, string rowKey) : base(partitionKey, rowKey) { }

        internal CloudTable _table;
    }

    public class BaseEntity<T> : BaseEntity where T : BaseEntity, new()
    {
        static BaseEntity()
        {
            var table = GetTable();
            table.CreateIfNotExists();
        }

        public BaseEntity(string partitionKey, string rowKey) : base(partitionKey, rowKey) { }

        public BaseEntity(int partitionKey, int rowKey) : base(partitionKey.ToString(), rowKey.ToString()) { }

        public CloudTable Table
        {
            get
            {                
                return _table ?? GetTable();
            }
        }

        public static T Get(string partitionKey, string rowKey)
        {
            if (string.IsNullOrEmpty(partitionKey))
            {
                throw new ArgumentNullException("partitionKey");
            }

            if (string.IsNullOrEmpty(rowKey))
            {
                throw new ArgumentNullException("rowKey");
            }

            TableOperation retrieveOperation = TableOperation.Retrieve<T>(partitionKey, rowKey);
            CloudTable table = GetTable();
            TableResult query = table.Execute(retrieveOperation);
            T result = (T)query.Result;
            result._table = table;
            return result;
        }

        public static CloudTable GetTable()
        {
            var client = AzureClientHelper.Instance.GetClient();
            var table = client.GetTableReference(typeof(T).Name);
            return table;
        }

        public void Delete()
        {
            TableOperation deleteOperation = TableOperation.Delete(this);
            Table.Execute(deleteOperation);
        }

        public void Insert()
        {
            TableOperation insertOperation = TableOperation.Insert(this);
            Table.Execute(insertOperation);
        }

        public void Update()
        {
            TableOperation updateOperation = TableOperation.Replace(this);
            Table.Execute(updateOperation);
        }

        public static T[] GetAll()
        {
            var query = new TableQuery<T>();            
            return GetTable().ExecuteQuery(query).ToArray();
        }

        public static T[] GetAllByPartitionKey(string partitionKey)
        {            
            TableQuery<T> query = new TableQuery<T>().Where(TableQuery.GenerateFilterCondition("PartitionKey", QueryComparisons.Equal, partitionKey));
            return GetTable().ExecuteQuery(query).ToArray();
        }

        public static void BulkInsert(IEnumerable<T> values)
        {
            var table = GetTable();
            var batch = new TableBatchOperation();
            foreach (var value in values)
            {
                batch.Insert(value);
            }            
            table.ExecuteBatch(batch);            
        }

        public static void BulkMerge(IEnumerable<T> values)
        {
            var table = GetTable();
            var batch = new TableBatchOperation();
            foreach (var value in values)
            {
                batch.Merge(value);
            }            
            table.ExecuteBatch(batch);
        }

        public static void BulkInsertOrMerge(IEnumerable<T> values)
        {
            var table = GetTable();
            var batch = new TableBatchOperation();
            foreach (var value in values)
            {
                batch.InsertOrMerge(value);
            }            
            table.ExecuteBatch(batch);
        }

        public static int BulkDeleteByPartionKey(string partitionKey)
        {
            var table = GetTable();
            var values = GetAllByPartitionKey(partitionKey);
            var count = values.Count();

            var partitions = values.Batch(100);
            foreach (var part in partitions)
            {
                var batch = new TableBatchOperation();

                foreach (var value in part)
                {
                    batch.Delete(value);
                }

                if (batch.Any())
                {
                    table.ExecuteBatch(batch);
                }
            }

            return count;

        }
    }
}
