﻿using System.Data.SqlTypes;
using Aerospike.Client;
using Curse.Aerospike.Tests.Models;
using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Curse.Aerospike.Tests
{
    class Program
    {
        const string Namespace = "cursevoice-global";
        const string SetName = "Test";
        const int UserID = 6;

        static void Main(string[] args)
        {
            
                        
            try
            {
                while (true)
                {
                    Console.Clear();
                    Console.WriteLine("Welcome to the Aerospike test app. This app is designed to quickly prototype out framework-level improvements to our Aerospike wrapper.");
                    Console.WriteLine("1. Test querying large lists");
                    Console.WriteLine("2. Test wiping and inserting data to a large list");
                    Console.WriteLine("3. Test large records");
                    Console.WriteLine("4. Test large list config");
                    Console.WriteLine("5. Test cluster affinity");
                    var selection = Console.ReadKey(true).Key;
                    Console.Clear();

                    switch (selection)
                    {
                        case ConsoleKey.Escape:
                            return;
                        case ConsoleKey.D1:
                            TestLargeLists(false);
                            break;
                        case ConsoleKey.D2:
                            TestLargeLists(true);
                            break;
                        case ConsoleKey.D3:
                            TestLargeRecords();
                            break;
                        case ConsoleKey.D4:
                            TestLargeListConfig();
                            break;
                        case ConsoleKey.D5:
                            TestAffinity();
                            break;
                        default:
                            Console.WriteLine("Invalidation selection! Press any key to continue...");
                            Console.ReadKey(true);
                            continue;

                    }
                    //TestGenericDictionaries();

                    //TestLargeListConfig();
                }
            }
            finally
            {
                Console.WriteLine("Shutting down Aerospike config...");
                AerospikeConfiguration.Shutdown();
            }
    
            return;

            var sw = Stopwatch.StartNew();
            try
            {
                var allRecords = Test.GetAllLocal();
                sw.Stop();
                Console.WriteLine("Retrieved " + allRecords.Length + " records in " + sw.ElapsedMilliseconds.ToString("###,##0.00"));
            }
            catch (Exception ex)
            {
                Console.WriteLine("Failed: " + ex.Message);
            }

            var m = new Test { UserID = 1, OtherUserID = 128, Name = "NeT", Status = TestStatus.Awesome, IsOnline = false, Timestamp = DateTime.UtcNow };
            m.InsertLocal();
            m = Test.GetLocal(1, 128);
            m.Timestamp = DateTime.UtcNow;            
            m.Update(p => p.Timestamp);
            m = Test.GetLocal(1, 128);

            var ids = new List<int>();

            for (int i = 0; i < 10000; i++)
            {
                ids.Add(i);
            }


            sw = Stopwatch.StartNew();
            Parallel.ForEach(ids, i =>
            {
                var testModel = new Test { UserID = 1, OtherUserID = i, Name = "User-" + i, Status = TestStatus.Awesome };
                testModel.InsertLocal();
            });

            {
                sw.Stop();
                var msPerRow = sw.ElapsedMilliseconds / (double)ids.Count;
                Console.WriteLine("Inserted " + ids.Count + " friends in " + sw.ElapsedMilliseconds.ToString("###,##0.00") + "ms (" + msPerRow.ToString("###,##0.00") + "ms per)");
            }

            sw = Stopwatch.StartNew();

            Parallel.ForEach(ids, i =>
            {
                var testModel = Test.GetLocal(1, i);
            });

            {
                sw.Stop();
                var msPerRow = sw.ElapsedMilliseconds / (double)ids.Count;
                Console.WriteLine("Retrieved " + ids.Count + " friends in " + sw.ElapsedMilliseconds.ToString("###,##0.00") + "ms (" + msPerRow.ToString("###,##0.00") + "ms per)");
            }

            while (true)
            {
                Console.WriteLine("--------------------------------------------------------------------");
                GetByUserID(ids);
                QueryByUserID();
                QueryByOtherUserID();
                Thread.Sleep(1000);

            }

            // Read record
            //Record record = client.Get(null, key);
            //var name = record.bins["UserID"];
            //var age = record.bins["OtherUserID"];                       

            Console.ReadKey();
        }

        private static void TestGenericDictionaries()
        {
            var m = new Test { UserID = 1, OtherUserID = 128, Name = "NeT", Status = TestStatus.Awesome, IsOnline = false, Timestamp = DateTime.UtcNow };
            m.TestDictionary = new Dictionary<int, TestSubField>();
            m.TestDictionary.Add(1, new TestSubField { TestBool = true, TestNumber = 1, TestString = "test" });
            m.InsertLocal();

            m = Test.GetLocal(1, 128);
        }

        static void TestLargeListConfig()
        {
            var listTest = new Test { UserID = 1, OtherUserID = 4, Name = "NeT", Status = TestStatus.Awesome, IsOnline = false, Timestamp = DateTime.UtcNow };
            try
            {
                listTest.InsertLocal();
                listTest = Test.GetLocal(1, 4);
            }
            catch (Exception ex)
            {
                Console.Write("Failed to insert or get a LDT record: " + ex.Message);
                Console.ReadKey();
                return;
            }

            var sw = Stopwatch.StartNew();
            var values = listTest.TestLargeList.GetValues();
            sw.Stop();

            Console.WriteLine("Retrieved " + values.Count() + " values in " + sw.ElapsedMilliseconds + "ms");

            try
            {
                sw.Restart();
                listTest.TestLargeList.Clear();
                Console.WriteLine("Cleared in " + sw.ElapsedMilliseconds + "ms");
            }
            catch (Exception ex)
            {
                Console.WriteLine("Failed to clear large list: " + ex.Message);
                Console.ReadKey();
                return;
            }

            sw.Restart();
            for (var i = 0; i < 1; i++)
            {
                for (var j = 1; j <= 100; j++)
                {
                    var dict = new Dictionary<string, object>();
                    var element = new TestMessage
                    {
                        ID = Guid.NewGuid(),
                        Body = "This is value 1 for item " + j,
                        SenderID = UserID,
                        Timestamp = DateTime.UtcNow
                    };
                    listTest.TestLargeList.Add(element);                    
                }
            }

            sw.Stop();
            Console.WriteLine("Inserted valued in " + sw.ElapsedMilliseconds + "ms");

            sw.Restart();
            var found = listTest.TestLargeList.Range(DateTime.UtcNow.AddDays(-1).ToEpoch(), DateTime.UtcNow.ToEpoch());
            sw.Stop();
            Console.WriteLine("After insertion, found: " + found.Count() + " records in " + sw.ElapsedMilliseconds + "ms");
            Console.ReadKey();

        }


        /// <summary>
        /// Attempt to insert update a record that exceeds 128 KB
        /// </summary>
        private static void TestLargeRecords()
        {
            var hs = new HashSet<int>();
            for (var i = 0; i < 102400; i++)
            {
                hs.Add(i);
            }

            var listTest = new Test { UserID = 1, OtherUserID = 128, Name = "NeT", Status = TestStatus.Awesome, IsOnline = false, Timestamp = DateTime.UtcNow, FriendIDs  = hs};
            listTest.InsertLocal();
            listTest = Test.GetLocal(1, 128);
        }

        static void TestLargeLists(bool testInserts)
        {
            var listTest = new Test { UserID = 1, OtherUserID = 128, Name = "NeT", Status = TestStatus.Awesome, IsOnline = false, Timestamp = DateTime.UtcNow };
            listTest.InsertLocal();
            listTest = Test.GetLocal(1, 128);
            
            if (testInserts)
            {
                listTest.TestLargeList.Clear();

                const int maxItems = 100000;

                listTest = Test.GetLocal(1, 128);
                var counter = 1;
                var timestamp = DateTime.UtcNow;
                while (true)
                {
                    
                    if (counter%100 == 0)
                    {
                        timestamp = timestamp.AddMinutes(-1);
                    }
                    else
                    {
                        timestamp = timestamp.AddSeconds(-1);
                    }

                    var message = new TestMessage()
                    {
                        Body = "This is a test message (#" + counter + ") of average length. Let's hope it scales!",
                        ID = Guid.NewGuid(),
                        SenderID = 1,
                        Timestamp = timestamp
                    };

                    listTest.TestLargeList.Add(message);

                    if (counter % 100 == 0)
                    {
                        Console.Title = counter.ToString("###,##0");
                    }

                    if (++counter > maxItems)
                    {
                        break;
                    }
                }
            }
            else
            {
                
                int mins;
                Console.WriteLine("How many minutes of messages would you like to retrieve? (Default to 30)");
                var input = Console.ReadLine();
                if (!int.TryParse(input, out mins))
                {
                    mins = 30;
                }
                var startTime = DateTime.UtcNow.AddMinutes(-mins);
                var endTime = DateTime.UtcNow;

                var sw = Stopwatch.StartNew();
                var rangeFilter = listTest.TestLargeList.Range(startTime.ToEpoch(), endTime.ToEpoch());
                sw.Stop();

                Console.WriteLine("Searching for messages from " + startTime.ToLocalTime() + " to " + endTime.ToLocalTime());
                Console.WriteLine("Query completed in " + sw.Elapsed.TotalMilliseconds.ToString("###,##0.00") + " milliseconds. " + rangeFilter.Count().ToString("###,##0") + " items were returned of " + listTest.TestLargeList.Count.ToString("###,##0") + " total. Showing the first 10...");

                sw = Stopwatch.StartNew();
                var list = rangeFilter.ToList();
                sw.Stop();
                Console.WriteLine("Hydration completed in " + sw.Elapsed.TotalMilliseconds.ToString("###,##0.00") + " milliseconds");

                foreach (var message in list.OrderBy(p => p.Timestamp).Take(10).OrderBy(p => p.Timestamp).ToArray())
                {
                    Console.WriteLine(message.SenderID + " said at " + message.Timestamp.ToLocalTime() + ": " + message.Body);
                }
                Console.ReadKey();
            }            
        }

        static void TestLargeMaps()
        {
            var hs = new HashSet<int>()
            { 
                1,2,3,4,5,6,10
            };

            Stopwatch sw;

            Console.WriteLine("Press any key to begin tests...");
            Console.ReadKey(true);

            var listTest = new Test { UserID = 1, OtherUserID = 128, Name = "NeT", Status = TestStatus.Awesome, IsOnline = false, Timestamp = DateTime.UtcNow, FriendIDs = hs };
            listTest.InsertLocal();
            listTest = Test.GetLocal(1, 128);

            
            // Test Clearing            
            if (listTest.TestLargeDictionary.Count > 0)
            {
                Console.Write("Clearing test set of " + listTest.TestLargeDictionary.Count + " items... ");
                sw = Stopwatch.StartNew();
                listTest.TestLargeDictionary.Clear();
                sw.Start();
                Console.WriteLine("Completed in " + sw.ElapsedMilliseconds.ToString("###,##0.00") + "ms");
            }

            // Test Individual Insert Speed
            Console.WriteLine("Press any key to continue tests... ");
            Console.ReadKey(true);
            const int maxItems = 0;

            Console.Write("Individually adding " + maxItems + " items...");
            sw = Stopwatch.StartNew();
            for (var i = 1; i <= maxItems; i++)
            {
                listTest.TestLargeDictionary.Add(i, "Testing #" + i);
                if (i%100 == 0)
                {
                    Console.Title = i.ToString("###,##0");
                }
            }
            Console.WriteLine("Completed in " + sw.ElapsedMilliseconds.ToString("###,##0.00") + "ms");

            // Test Bulk Insert Speed
            Console.WriteLine("Press any key to finish tests...");
            Console.ReadKey(true);
            //return;
                        
            const int maxBulkItems = 10000;
            const int startingId = 100000;
            var dict = new Dictionary<int, string>();
            for (var i = startingId; i <= startingId + maxBulkItems; i++)
            {
                dict.Add(i, "Testing #" + i);
            }

            Console.Write("Bulk adding test set of " + dict.Count + " items... ");
            var page = 0;
            const int perPage = 4100;
            while (true)
            {
                sw = Stopwatch.StartNew();
                var currentItems = dict.Skip(page * perPage).Take(perPage).ToDictionary(p => p.Key, p => p.Value);
                if (currentItems.Count == 0)
                {
                    break;                    
                }
                
                listTest.TestLargeDictionary.BulkAdd(currentItems);
                                
                Console.WriteLine("Completed in " + sw.ElapsedMilliseconds.ToString("###,##0.00") + "ms");
                ++page;
            }
            Console.WriteLine("Press any key to finish tests...");
            Console.ReadKey(true);
            return;

            var values = listTest.TestLargeDictionary.Values;

            foreach (var kvp in values)
            {
                Console.WriteLine(kvp.Key + ": " + kvp.Value);
            }
            Console.WriteLine("Done!");
            Console.ReadLine();
        }

        static void GetByUserID(IEnumerable<int> otherUserIDs)
        {
            var sw = Stopwatch.StartNew();
            Parallel.ForEach(otherUserIDs, i =>
            {
                var testModel = Test.GetLocal(1, i);
            });

            sw.Stop();
            var msPer = sw.Elapsed.TotalMilliseconds / (double)otherUserIDs.Count();
            Console.WriteLine("Retrieved " + otherUserIDs.Count() + " friends in " + sw.Elapsed.TotalMilliseconds.ToString("###,##0.000") + "ms (" + msPer.ToString("###,##0.000") + "ms per)");
        }

        static void QueryByUserID()
        {
            var sw = Stopwatch.StartNew();
            var allFriends = Test.GetAllLocal(x => x.UserID, 1);
            sw.Stop();
            var msPer = sw.Elapsed.TotalMilliseconds / (double)allFriends.Length;
            Console.WriteLine("Queried " + allFriends.Length + " friends by UserID in " + sw.Elapsed.TotalMilliseconds.ToString("###,##0.000") + "ms (" + msPer.ToString("###,##0.000") + "ms per)");
        }

        static void QueryByOtherUserID()
        {
            var sw = Stopwatch.StartNew();
            var allFriends = Test.GetAllLocal(x => x.OtherUserID, 1);
            sw.Stop();
            var msPer = sw.Elapsed.TotalMilliseconds / (double)allFriends.Length;
            Console.WriteLine("Queried " + allFriends.Length + " friends by OtherUserID in " + sw.Elapsed.TotalMilliseconds.ToString("###,##0.000") + "ms (" + msPer.ToString("###,##0.000") + "ms per)");
        }

        static void CreateIndex(AerospikeClient client, string binName)
        {
            var task = client.CreateIndex(null, Namespace, SetName, "IDX_" + SetName + "_" + binName, binName, IndexType.NUMERIC);
            task.Wait();
        }

        static void CreateManyFriends(AerospikeClient client, int userID, int numberToCreate)
        {
            for (int i = 1; i < numberToCreate; i++)
            {
                if (i == userID)
                {
                    continue;
                }

                var key = new Key(Namespace, SetName, userID + "-" + i.ToString());

                // Create Bins
                var bin1 = new Bin("UserID", userID);
                var bin2 = new Bin("OtherUserID", i);

                // Write record
                client.Put(null, key, bin1, bin2);
            }
        }

        static void QueryByUserID(AerospikeClient client, int userID)
        {
            var stmt = new Statement();
            stmt.SetNamespace(Namespace);
            stmt.SetSetName(SetName);
            stmt.SetFilters(Filter.Equal("UserID", userID));
            var rs = client.Query(null, stmt);

            while (rs.Next())
            {
                Key key = rs.Key;
                Record record = rs.Record;
            }
        }

        static void TestAffinity()
        {
            var m = new Test { UserID = 1, OtherUserID = 128, Name = "NeT", Status = TestStatus.Awesome, IsOnline = false, Timestamp = DateTime.UtcNow };
            m.InsertLocal();
            m = Test.GetLocal(1, 128);
            m.Timestamp = DateTime.UtcNow;
            m.Update(p => p.Timestamp);
            m = Test.GetLocal(1, 128);

            var n = new AffinityTest { ID = Guid.NewGuid(), Timestamp = DateTime.UtcNow };
            n.InsertLocal();
            n = AffinityTest.GetLocal(n.ID);
            n.Timestamp = DateTime.UtcNow;
            n.Update(p => p.Timestamp);
            n = AffinityTest.GetLocal(1, 128);
            
        }
    }


}
