using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Xml.Serialization;

namespace Curse.Aerospike.Tools.Configuration
{
    [Serializable]
    public class AerospikeModel
    {
        public string Name { get; set; }

        public bool IsGlobal { get; set; }

        public int Affinity { get; set; }

        private List<IndexedProperty> _indexedProperties = new List<IndexedProperty>();
        public List<IndexedProperty> IndexedProperties
        {
            get { return _indexedProperties; }
            set { _indexedProperties = value; }
        }

        private List<KeyProperty> _keyProperties = new List<KeyProperty>();
        public List<KeyProperty> KeyProperties
        {
            get{return _keyProperties;} 
            set { _keyProperties = value; }
        }

        public override string ToString()
        {
            return string.Format("Set {0}, {1} Key Parts, {2} Indexes", Name, KeyProperties.Count,
                IndexedProperties.Count);
        }


        private static readonly string DefaultModelsPath = AppDomain.CurrentDomain.BaseDirectory + "\\DefaultModels.asmodels";
        private static readonly string ReflectedModelsPath = AppDomain.CurrentDomain.BaseDirectory + "\\Dependencies";

        private static volatile bool _isLoaded;
        private static AerospikeModelCollection _models;

        public static List<AerospikeModel> GetModelsForNamespace(string @namespace, HashSet<string> setNames)
        {
            if (!_isLoaded)
            {
                LoadDefaultModels();
                _isLoaded = true;
            }
            return _models.Models.Where(m => setNames.Contains(m.Name, StringComparer.InvariantCultureIgnoreCase)).ToList();
        }

        private static void LoadDefaultModels()
        {
            _models = new AerospikeModelCollection();

            ImportModels(DefaultModelsPath);

            if (Directory.Exists(ReflectedModelsPath))
            {
                foreach (var file in Directory.GetFiles(ReflectedModelsPath, "*.dll"))
                {
                    ImportModels(file);
                }
            }
        }

        public static void ImportModels(string modelFile)
        {
            if (modelFile.EndsWith(".asmodels"))
            {
                LoadModelFile(modelFile);
            }
            else
            {
                LoadReflectedModels(modelFile);
            }
        }

        private static void LoadModelFile(string modelFile)
        {
            try
            {
                var serializer = new XmlSerializer(typeof(AerospikeModelCollection));
                using (var stream = File.OpenRead(modelFile))
                {
                    var importedModels = (AerospikeModelCollection)serializer.Deserialize(stream);
                    foreach (var model in importedModels.Models)
                    {
                        _models.Add(model);
                    }
                    stream.Close();
                }
            }
            catch
            {
                Debugger.Break();
            }
        }

        private static void LoadReflectedModels(string dllFile)
        {
            LoadTablesFromAssembly(Assembly.LoadFrom(dllFile));
        }

        private static void LoadTablesFromAssembly(Assembly assembly)
        {
            var types = assembly.GetTypes().Where(t => t.GetCustomAttribute<TableDefinitionAttribute>() != null);

            foreach (var type in types)
            {
                var indexedProperties = type.GetProperties().Where(p =>
                {
                    {
                        var columnAttribute = p.GetCustomAttribute<ColumnAttribute>();
                        return columnAttribute != null && columnAttribute.IsIndexed;
                    }
                }).Select(p => new IndexedProperty
                {
                    Name = p.GetCustomAttribute<ColumnAttribute>().Name,
                    DataType = GetCorrectedType(p.PropertyType)
                });

                var keyProperties = type.GetProperties().Where(p =>
                {
                    var columnAttribute = p.GetCustomAttribute<ColumnAttribute>();
                    return columnAttribute != null && columnAttribute.KeyOrdinal > 0;
                }).Select(p =>
                {
                    var columnAttribute = p.GetCustomAttribute<ColumnAttribute>();
                    return new KeyProperty
                    {
                        Name = p.Name,
                        DataType = GetCorrectedType(p.PropertyType),
                        Ordinal = columnAttribute.KeyOrdinal,
                        IsOptional = columnAttribute.IsOptional
                    };
                });

                var tableDefinition = type.GetCustomAttribute<TableDefinitionAttribute>();
                _models.Add(new AerospikeModel
                {
                    Name = type.Name,
                    KeyProperties = keyProperties.ToList(),
                    IndexedProperties = indexedProperties.ToList(),
                    IsGlobal = (tableDefinition.KeySpace??"").ToLower().Contains("global"),
                    
                });
            }
        }

        private static Type GetCorrectedType(Type originalType)
        {
            return originalType.IsEnum ? typeof (AerospikeEnum) : originalType;
        }

        public static void ExportModels(string filename)
        {
            try
            {
                var serializer = new XmlSerializer(typeof(AerospikeModelCollection));
                using (var stream = File.Create(filename))
                {
                    serializer.Serialize(stream, _models);
                    stream.Close();
                }
            }
            catch
            {
                Debugger.Break();
            }
        }


    }
}
