﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;
using System.Linq;
using System.Windows;
using Aerospike.Client;
using Curse.Aerospike.Tools.Configuration;

namespace Curse.Aerospike.Tools.Filtering
{
    public abstract class AerospikeQuery
    {
        public const int MaxTableRows = 1000;

        public string BinName { get; private set; }

        protected AerospikeQuery(string binName)
        {
            BinName = binName;
        }

        public abstract QueryResults Run(AerospikeSet set, string sortColumn = null);

        public static AerospikeQuery CreateQuery(IndexedProperty property)
        {
            if (property.DataType.IsAssignableFrom(typeof (string))
                || property.DataType.IsAssignableFrom(typeof (Guid)))
            {
                return new StringQuery(property.Name);
            }
            else
            {
                return new LongQuery(property.Name);
            }
        }

        public static AerospikeQuery CreateKeyQuery(List<KeyProperty> keyProperties)
        {
            return new KeyLookupQuery(keyProperties);
        }

        public static AerospikeQuery CreateNonFilteringQuery()
        {
            return new NoFilterQuery();
        }

        private static void UpdateDataTableColumns(DataTable dataTable, Dictionary<string, object> bins)
        {
            if (!dataTable.Columns.Contains("InternalKey"))
            {
                dataTable.Columns.Add(new DataColumn("InternalKey", typeof(Key)) { ReadOnly = true });
            }
            foreach (var b in bins)
            {
                if (!dataTable.Columns.Contains(b.Key))
                {
                    Type type;
                    bool shouldBeReadOnly = false;
                    if (b.Value == null || b.Value is DBNull)
                    {
                        type = typeof(string);
                        shouldBeReadOnly = true;
                    }
                    else
                    {
                        type = b.Value.GetType();
                    }
                    var dc = new DataColumn(b.Key, type) { ReadOnly = shouldBeReadOnly };
                    dataTable.Columns.Add(dc);
                }
            }
        }

        protected static QueryResults Query(AerospikeSet set, Filter filter, string sortColumn = null)
        {
            var dataTable = new DataTable();
            var client = set.Parent.Parent.Client;

            var records = new Dictionary<Key, Record>();
            var recordsetTruncated = false;

            try
            {
                var sm = new Statement();
                sm.SetNamespace(set.Parent.Name);
                sm.SetSetName(set.Name);

                if (filter != null)
                {
                    sm.SetFilters(filter);
                }

                var recordCounter = 0;

                //if(false)
                using (var rs = client.Query(null, sm))
                {

                    while (rs.Next())
                    {
                        if (recordCounter++ == MaxTableRows)
                        {
                            recordsetTruncated = true;
                            break;
                        }
                        records.Add(rs.Key, rs.Record);
                    }
                }

                //if(true)
                //client.ScanAll(null, set.Parent.Name, set.Name, (key, record) =>
                //{
                //    if (recordCounter++ == MaxTableRows)
                //    {
                //        recordsetTruncated = true;
                //        throw new AerospikeException.ScanTerminated();
                //    }
                //    records.Add(key, record);
                //});
            }
            catch (Exception ex)
            {
                Debug.WriteLine(ex.Message);
                Debugger.Break();
            }

            // Build out the cols
            if (!records.Any())
            {
                return new QueryResults(dataTable.DefaultView, false);
            }

            dataTable.BeginLoadData();

            foreach (var record in records)
            {
                if (record.Value == null)
                {
                    continue;
                }

                UpdateDataTableColumns(dataTable, record.Value.bins);

                var dataRow = dataTable.NewRow();
                
                dataRow["InternalKey"] = record.Key;

                foreach (var bin in record.Value.bins)
                {
                    try
                    {
                        dataRow[bin.Key] = bin.Value;
                    }
                    catch (Exception ex)
                    {
                                 
                    }
                    
                }

                dataTable.Rows.Add(dataRow);
            }

            dataTable.EndLoadData();

            if (sortColumn != null)
            {
                dataTable.DefaultView.Sort = sortColumn;
            }

            return new QueryResults(dataTable.DefaultView, recordsetTruncated);
        }

        protected static QueryResults Get(AerospikeSet set, Key key)
        {
            var dataTable = new DataTable();
            var client = set.Parent.Parent.Client;

            dataTable.BeginLoadData();

            Record record = null;
            try
            {
                record = client.Get(null, key);
            }
            catch (AerospikeException ex)
            {      
#if DEBUG
                Debugger.Break();
#else
                MessageBox.Show("Failed to retrieve record: " + ex.Message);
#endif
            }

            if (record != null)
            {
                UpdateDataTableColumns(dataTable, record.bins);

                var dataRow = dataTable.NewRow();

                // Make a new key without the userKey so it shows up consistently in the table like normal queries.
                dataRow["InternalKey"] = new Key(key.ns, key.digest, key.setName, null);

                foreach (var bin in record.bins)
                {
                    try
                    {
                        dataRow[bin.Key] = bin.Value;
                    }
                    catch
                    {
                        
                    }

                    
                }

                dataTable.Rows.Add(dataRow);
            }

            dataTable.EndLoadData();
            return new QueryResults(dataTable.DefaultView, false);
        }
    }
}
