﻿using Aerospike.Client;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Data;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Threading;
using Curse.Aerospike.Tools.Configuration;
using Curse.Aerospike.Tools.Filtering;
using Microsoft.Win32;

namespace Curse.Aerospike.Tools
{
    /// <summary>
    /// Interaction logic for MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window
    {
        public MainWindow()
        {
            Environments = new ObservableCollection<AerospikeEnvironment>();
            InitializeComponent();


            _connectionInfo = ConfigurationHelper.LoadConfigurations<AerospikeConfigurationCollection>(@"C:\Projects\CloudServices\Source\Curse.Friends.Configuration\Configuration\");

            ClusterExplorer.DataContext = this;
            IPAddressListBox.ItemsSource = _connectionInfo;
            DataExplorerGrid.CellEditEnding += DataExplorerGrid_CellEditEnding;
            DataExplorerGrid.CurrentCellChanged += dataGrid_CurrentCellChanged;

            this.Closed += MainWindow_Closed;
            this.Loaded += OnLoaded;

        }

        private void OnLoaded(object sender, RoutedEventArgs routedEventArgs)
        {
            //ConnectToCluster(_connectionInfo.First());
            ConnectToCluster(_connectionInfo.First(x => x.Name == "Production"));
        }

        private DataRowView rowBeingEdited = null;
        private DataGridColumn colBeingEdited = null;

        private readonly AerospikeConfigurationCollection[] _connectionInfo;

        private void DeleteSelectedRow()
        {
            var selected = DataExplorerGrid.SelectedItems;

            if (selected == null || selected.Count == 0)
            {
                return;
            }

            if (MessageBox.Show("Are you sure you want to delete this row?", "Delete?", MessageBoxButton.YesNo) != MessageBoxResult.Yes)
            {
                return;
            }

            int deleteCount = 0;
            int failedCount = 0;
            var deletedRows = new List<DataRow>();

            foreach (var item in selected)
            {
                var dataRowView = item as DataRowView;
                var dataRow = dataRowView.Row;
                var set = ClusterExplorer.SelectedItem as AerospikeSet;
                var client = set.Parent.Parent.Client;
                var key = set.GetKey(dataRow);
                var deleted = client.Delete(null, key);
                if (deleted)
                {
                    deletedRows.Add(dataRow);
                    ++deleteCount;
                }
                else
                {
                    ++failedCount;
                }
            }

            foreach (var row in deletedRows)
            {
                row.Delete();
            }

            if (failedCount > 0)
            {
                MessageBox.Show(failedCount + " rows failed to be deleted!");
            }

        }

        private void dataGrid_CurrentCellChanged(object sender, EventArgs e)
        {
            if (rowBeingEdited != null)
            {
                rowBeingEdited.EndEdit();

                var dataRow = rowBeingEdited.Row;
                if (colBeingEdited.DisplayIndex < 0 || colBeingEdited.DisplayIndex > dataRow.ItemArray.Length)
                {
                    return;
                }

                var val = dataRow.ItemArray[colBeingEdited.DisplayIndex];


                var pk = dataRow.Table.PrimaryKey;
                var set = ClusterExplorer.SelectedItem as AerospikeSet;
                var client = set.Parent.Parent.Client;
                var changedBin = new Bin(colBeingEdited.SortMemberPath, val);

                var keyValues = new List<string>();
                foreach (var col in pk)
                {
                    keyValues.Add(dataRow.ItemArray[col.Ordinal].ToString());
                }

                var key = set.GetKey(dataRow);

                var existing = client.Get(null, key);
                if (existing == null)
                {
                    return;
                }

                client.Put(null, key, changedBin);
            }

            rowBeingEdited = null;
        }

        void DataExplorerGrid_CellEditEnding(object sender, DataGridCellEditEndingEventArgs e)
        {
            var x = e;
            colBeingEdited = e.Column;
            var dataRowView = e.Row.Item as DataRowView;
            rowBeingEdited = dataRowView;
        }

        void MainWindow_Closed(object sender, EventArgs e)
        {
            foreach (var environment in Environments)
            {
                environment.Dispose();
            }
        }

        public ObservableCollection<AerospikeEnvironment> Environments { get; set; } 

        private void RefreshButton_Click(object sender, RoutedEventArgs e)
        {
            ReloadCurrentSet(true);
        }

        private void Button_Click(object sender, RoutedEventArgs e)
        {
            var config = IPAddressListBox.SelectedValue as AerospikeConfigurationCollection;
            if (config == null)
            {
                return;
            }
            ConnectToCluster(config);

        }

        private void ConnectToCluster(AerospikeConfigurationCollection configs)
        {
            if (Environments.All(p => !p.Name.Equals(configs.Name)))
            {
                var clusters = new List<AerospikeCluster>();
                foreach (var config in configs.Configurations)
                {
                    

                    var cluster = CreateCluster(config, new HashSet<string>(configs.SetRouting.First(r => r.ID == config.SetRouteID).Sets));
                    clusters.Add(cluster);

                    if (config.MirrorAddresses != null && config.MirrorAddresses.Any())
                    {
                        var mirrorCluster = CreateMirrorCluster(config);
                        clusters.Add(mirrorCluster);
                    }
                }

                Environments.Add(new AerospikeEnvironment(configs.Name, clusters));
            }
        }

        private static AerospikeCluster CreateCluster(AerospikeConfiguration config, HashSet<string> setNames)
        {
            var cluster = new AerospikeCluster(config.RegionKey, setNames, config.Addresses);

            var clusterField = typeof(AerospikeClient).GetField("cluster", BindingFlags.NonPublic | BindingFlags.Instance);
            var client = (Cluster)clusterField.GetValue(cluster.Client);
            var partitionField = typeof(Cluster).GetField("partitionMap", BindingFlags.NonPublic | BindingFlags.Instance);
            var partitionMap = (Dictionary<string, Node[][]>)partitionField.GetValue(client);

            foreach (var k in partitionMap.Keys)
            {
                cluster.AddNamespace(k);
            }

            return cluster;
        }

        private static AerospikeCluster CreateMirrorCluster(AerospikeConfiguration config)
        {
            var cluster = new AerospikeCluster(config.RegionKey, config.MirrorSets, config.MirrorAddresses);

            var clusterField = typeof(AerospikeClient).GetField("cluster", BindingFlags.NonPublic | BindingFlags.Instance);
            var client = (Cluster)clusterField.GetValue(cluster.Client);
            var partitionField = typeof(Cluster).GetField("partitionMap", BindingFlags.NonPublic | BindingFlags.Instance);
            var partitionMap = (Dictionary<string, Node[][]>)partitionField.GetValue(client);

            foreach (var k in partitionMap.Keys)
            {
                cluster.AddNamespace(k);
            }

            return cluster;
        }

        private void ClusterExplorer_SelectedItemChanged(object sender, RoutedPropertyChangedEventArgs<object> e)
        {
            var newItem = e.NewValue;
            if (newItem is AerospikeSet)
            {
                var set = newItem as AerospikeSet;
                Dispatcher.Invoke((Action)(() =>
                {
                    DetailView.DataContext = set;
                }));

                ReloadCurrentSet();
            }
        }

        private void ReloadCurrentSet(bool forceRefresh = false)
        {
            var currentSet = ClusterExplorer.SelectedItem as AerospikeSet;
            ;
            if (currentSet != null)
            {
                System.Threading.Tasks.Task.Factory.StartNew(() => currentSet.LoadData(forceRefresh));
            }
        }

        private void DeleteButton_Click(object sender, RoutedEventArgs e)
        {
            DeleteSelectedRow();
        }

        private void ApplyButton_Click(object sender, RoutedEventArgs e)
        {
            FilterToggleButton.IsChecked = false;
            ReloadCurrentSet(true);
        }

        private const string AerospikeModelsFilter = "Aerospike Models File|*.asmodels";
        private const string ImportFilter = "Models File|*.asmodels;*.dll|" + AerospikeModelsFilter;

        private void ImportButton_OnClick(object sender, RoutedEventArgs e)
        {
            var dialog = new OpenFileDialog
            {
                InitialDirectory = AppDomain.CurrentDomain.BaseDirectory,
                Filter = ImportFilter
            };
            var result = dialog.ShowDialog();
            if (result.HasValue && result.Value)
            {
                foreach (var fileName in dialog.FileNames)
                {
                    AerospikeModel.ImportModels(fileName);
                }

                foreach (var environment in Environments)
                {
                    foreach (var cluster in environment.Clusters)
                    {
                        cluster.ReloadNamespaces();
                    }
                }
            }
        }

        private void ExportButton_OnClick(object sender, RoutedEventArgs e)
        {
            var dialog = new SaveFileDialog
            {
                InitialDirectory = AppDomain.CurrentDomain.BaseDirectory,
                Filter = AerospikeModelsFilter
            };
            var result = dialog.ShowDialog();
            if (result.HasValue && result.Value)
            {
                AerospikeModel.ExportModels(dialog.FileName);
            }
        }
    }
}
