﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Serialization;
using Aerospike.Client;
using Curse.Logging;

namespace Curse.Aerospike
{
    [XmlType("Cluster")]
    public class AerospikeConfiguration
    {
        public static AerospikeConfiguration[] Configurations;
        public static bool RegionalConnectionRouting = true;
        public static bool CreateIndexes = false;
        public static bool IsInitialized = false;

        public string UniqueKey
        {
            get { return RegionIdentifier + "-" + string.Join(";", Sets.ToArray()); }
        }

        [XmlElement]
        public int SetRouteID { get; set; }

        [XmlElement()]
        public int RegionIdentifier
        {
            get;
            set;
        }

        [XmlElement()]
        public string RegionKey
        {
            get;
            set;
        }

        [XmlElement()]
        public string RegionGroup
        {
            get;
            set;
        }

        [XmlArray()]
        [XmlArrayItem("Address")]
        public string[] Addresses
        {
            get;
            set;
        }

        [XmlArray()]
        [XmlArrayItem("Address")]
        public string[] MirrorAddresses
        {
            get;
            set;
        }

        [XmlElement("MirrorKeyspace")]
        public string MirrorKeyspace
        {
            get;
            set;
        }


        [XmlArray()]
        [XmlArrayItem("Set")]
        public HashSet<string> MirrorSets
        {
            get;
            set;
        }

        [XmlElement()]
        public int Port
        {
            get;
            set;
        }

        [XmlIgnore]
        public bool IsLocal
        {
            get;
            private set;
        }

        [XmlElement()]
        public string Username
        {
            get;
            set;
        }

        [XmlElement()]
        public string Password
        {
            get;
            set;
        }

        [XmlArray()]
        [XmlArrayItem("Set")]
        public HashSet<string> Sets
        {
            get;
            set;
        }
        
        public const int WriteTimeout = 20000;
        public const int ReadTimeout = 10000;
        public const int QueryTimeout = 30000;

        // Update Policies
        public static WritePolicy DefaultUpdatePolicy;
        public static WritePolicy DefaultFastWriteUpdatePolicy;

        // Insert Policies
        public static WritePolicy DefaultInsertPolicy;
        public static WritePolicy DefaultFastWriteInsertPolicy;
        public static WritePolicy DefaultConcurrentInsertPolicy;

        private readonly object _syncRoot = new object();

        public AerospikeClient Client
        {
            get
            {
                return _client;
            }
        }

        private AerospikeClient _client;

        private AerospikeClient _mirrorClient;

        public AerospikeClient MirrorClient
        {
            get
            {
                return _mirrorClient;
            }
        }

        public bool Connect()
        {

            Logger.Info("Connecting to Aerospike cluster: " + string.Join(", ", Addresses) + "...");


            lock (_syncRoot)
            {
                if (_client != null)
                {
                    _client.Dispose();
                    _client = null;
                }
                _client = CreateClient(Addresses);

                if (_client == null)
                {
                    return false;
                }

                if (MirrorAddresses != null && MirrorAddresses.Any())
                {
                    if (_mirrorClient != null)
                    {
                        _mirrorClient.Dispose();
                        _mirrorClient = null;
                    }
                    _mirrorClient = CreateClient(MirrorAddresses);
                    if (_mirrorClient == null)
                    {
                        return false;
                    }
                }

                return true;
            }

        }


        private AerospikeClient CreateClient(string[] addresses)
        {
            try
            {

                var hosts = addresses.Select(p => new Host(p, Port)).ToArray();
                var clientPolicy = new ClientPolicy();

                clientPolicy.queryPolicyDefault = new QueryPolicy { timeout = QueryTimeout };
                clientPolicy.readPolicyDefault = new Policy { timeout = ReadTimeout };
                clientPolicy.writePolicyDefault = new WritePolicy { timeout = WriteTimeout };
                clientPolicy.maxSocketIdle = 30;

                var client = new AerospikeClient(clientPolicy, hosts);
                Logger.Info("Successfully connected to cluster, '" + RegionKey + "': " + string.Join(", ", addresses));
                return client;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to connect to cluster, '" + RegionKey + "': " + string.Join(", ", addresses));
                return null;
            }
        }

        public static string DefaultKeySpace
        {
            get;
            private set;
        }

        private static readonly LogCategory Logger = new LogCategory("Aerospike");

        private static void LogCallback(Log.Level level, string msg)
        {
            switch (level)
            {
                case Log.Level.DEBUG:
                    Logger.Debug(msg);
                    break;
                case Log.Level.INFO:
                    Logger.Info(msg);
                    break;

                case Log.Level.WARN:
                    Logger.Warn(msg);
                    break;

                case Log.Level.ERROR:
                    Logger.Error(msg);
                    break;
            }
        }
        
         public bool ShouldMirror(string setName)
        {
            return MirrorSets != null && MirrorSets.Contains(setName);
        }

        public static WritePolicy GetWritePolicy(UpdateMode mode, int ttlSeconds = 0)
        {
            if (ttlSeconds == 0)
            {
                switch (mode)
                {
                    case UpdateMode.Concurrent:
                        return DefaultConcurrentInsertPolicy;

                    case UpdateMode.Fast:
                        return DefaultFastWriteInsertPolicy;

                    default:
                        return DefaultInsertPolicy;
                }
            }
           
            switch (mode)
            {
                case UpdateMode.Concurrent:
                    return new WritePolicy { timeout = WriteTimeout, recordExistsAction = RecordExistsAction.CREATE_ONLY, expiration = ttlSeconds };

                case UpdateMode.Fast:
                    return new WritePolicy { timeout = WriteTimeout, commitLevel = CommitLevel.COMMIT_MASTER, expiration = ttlSeconds };

                default:
                    return new WritePolicy { timeout = WriteTimeout, expiration = ttlSeconds};
            }
            
        }

        public static void Initialize(string localRegionIdentifier, string defaultKeySpace, AerospikeConfigurationCollection configurations, bool enableRegionalConnectionRouting = true, bool createIndexes = false, bool localOnly = false)
        {

            CreateIndexes = createIndexes;

            // Update policy to ensure that an update statement doesn't insert a partial record into the DB
            DefaultUpdatePolicy = new WritePolicy { timeout = WriteTimeout, recordExistsAction = RecordExistsAction.UPDATE_ONLY };
            DefaultFastWriteUpdatePolicy = new WritePolicy { timeout = WriteTimeout, recordExistsAction = RecordExistsAction.UPDATE_ONLY, commitLevel = CommitLevel.COMMIT_MASTER };

            DefaultInsertPolicy = new WritePolicy { timeout = WriteTimeout };
            DefaultFastWriteInsertPolicy = new WritePolicy { timeout = WriteTimeout, commitLevel = CommitLevel.COMMIT_MASTER };
            DefaultConcurrentInsertPolicy = new WritePolicy { timeout = WriteTimeout, recordExistsAction = RecordExistsAction.CREATE_ONLY };

            var setRoutes = configurations.SetRouting.ToDictionary(c => c.ID);

            RegionalConnectionRouting = enableRegionalConnectionRouting;
            Log.SetLevel(Log.Level.WARN);
            Log.SetCallback(LogCallback);
            DefaultKeySpace = defaultKeySpace;
            Configurations = configurations.Configurations;

            foreach (var config in Configurations)
            {
                AerospikeSetRoute route;
                if (!setRoutes.TryGetValue(config.SetRouteID, out route))
                {
                    throw new InvalidOperationException("Set route could not be found for Aerospike configuration!");
                }

                config.Sets = new HashSet<string>(route.Sets);

                if (config.RegionKey.Equals(localRegionIdentifier, StringComparison.InvariantCultureIgnoreCase))
                {
                    config.IsLocal = true;
                }
                else if (localOnly)
                {
                    continue;
                }

                config.Connect();
            }

            IsInitialized = true;
        }

        public static void Shutdown()
        {
            if (Configurations == null)
            {
                return;
            }

            foreach (var config in Configurations)
            {
                if (config.Client != null)
                {
                    config.Client.Dispose();
                }
            }
        }
    }
}
