﻿using Aerospike.Client;
using Curse.Logging;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Reflection;
using System.Security.Cryptography;
using Curse.Extensions;

namespace Curse.Aerospike
{
    public abstract class BaseTable<TEntity>
        where TEntity : BaseTable<TEntity>, new()
    {

        static readonly ReplicationMode ReplicationMode;
        static readonly string BaseKeySpace;
        static readonly string TableName;
                
        private static readonly Dictionary<int, AerospikeConfiguration> ConfigByRegion = new Dictionary<int, AerospikeConfiguration>();
        private static readonly Dictionary<int, string> RegionKeys = new Dictionary<int, string>();
        public static readonly AerospikeConfiguration[] RemoteConfigurations;
        public static readonly AerospikeConfiguration[] AllConfigurations;
        public static AerospikeConfiguration LocalConfiguration;
        public static readonly int LocalConfigID;

        protected static readonly LogCategory Logger = new LogCategory(typeof(TEntity).Name);
        protected static readonly LogCategory DiagLogger = new LogCategory(typeof(TEntity).Name) { Throttle = TimeSpan.FromMinutes(5) };

        public static string GetRegionKey(int regionIdentifier)
        {
            string key;
            if (RegionKeys.TryGetValue(regionIdentifier, out key))
            {
                return key;
            }

            return "na";
        }

        public static bool IsValidRegion(int regionIdentifier)
        {
            return ConfigByRegion.ContainsKey(regionIdentifier);
        }

        public static AerospikeConfiguration GetConfiguration(int regionIdentifier)
        {
            if (!AerospikeConfiguration.RegionalConnectionRouting)
            {
                return LocalConfiguration;
            }

            if (!ConfigByRegion.ContainsKey(regionIdentifier))
            {             
                throw new InvalidOperationException("Attempt to use an unknown Aerospike configuration region: " + regionIdentifier);
            }

            return ConfigByRegion[regionIdentifier];
        }

        public static TEntity GetLocal(string keyValue)
        {
            return Get(LocalConfiguration, keyValue);
        }

        public static TEntity Get(int regionID, params object[] keyValues)
        {
            return Get(GetConfiguration(regionID), keyValues);
        }

        protected static string GetKeySpace(string region)
        {
            if (ReplicationMode == ReplicationMode.None)
            {
                return BaseKeySpace + "-" + region.ToLower();
            }
            else
            {
                return BaseKeySpace;
            }
        }

        public static TEntity GetWritable(params object[] keyValues)
        {
            var model = Get(LocalConfiguration, keyValues);
            if (model == null)
            {
                return null;
            }

            if (ReplicationMode == ReplicationMode.HomeRegion)
            {
                var homeRegionID = (model as IModelRegion).RegionID;
                if (model.SourceConfiguration.RegionIdentifier != homeRegionID)
                {
                    return Get(homeRegionID, keyValues);
                }
            }

            return model;
        }

        public TEntity EnsureWritable()
        {
            if (SourceConfiguration == null)
            {
                return this as TEntity;
            }

            var modelRegion = (this as IModelRegion);
            if (modelRegion == null || SourceConfiguration.RegionIdentifier == modelRegion.RegionID)
            {
                return this as TEntity;
            }

            TEntity homeModel = null;

            var key = GetKey(SourceConfiguration);

            try
            {
                homeModel = GetByKey(GetConfiguration(modelRegion.RegionID), key);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex);
            }
            

            if (homeModel == null)
            {
                Logger.Warn("Failed to retrieve model from home region. The current model will be used instead.",
                    new { modelRegion.RegionID, key = key.ToString() });

                return this as TEntity;
            }

            return homeModel;

        }


        public static TEntity GetLocal(params object[] keyValues)
        {
            return Get(LocalConfiguration, keyValues);
        }

        protected const string KeyDelimeter = "-";

        public static TEntity GetLocalByKey(Key key)
        {
            return GetByKey(LocalConfiguration, key);
        }

        public static IReadOnlyCollection<TEntity> MultiGetByKeys(AerospikeConfiguration configuration, Key[] keys, bool returnNulls = false)
        {
            var models = new List<TEntity>(keys.Length);

            try
            {
                foreach (var batch in keys.InSetsOf(5000))
                {
                    var records = configuration.Client.Get(null, batch.ToArray());
                    foreach (var record in records)
                    {
                        if (record != null)
                        {
                            var model = new TEntity();
                            model.Hydrate(configuration, null, record);
                            models.Add(model);
                        }
                        else if (returnNulls)
                        {
                            models.Add(null);
                        }
                    }
                }
            }
            catch (AerospikeException ex)
            {
                if (LocalConfigID != configuration.RegionIdentifier) // Remote access, probably packet loss
                {
                    Logger.Error(ex, "Get operation from region " + LocalConfiguration.RegionKey + " failed to access remote region " + configuration.RegionKey, new
                    {
                        LocalRegion = LocalConfiguration.RegionKey,
                        AccessedRegion = configuration.RegionKey,
                    });
                }
                else // Local Access, other issue
                {
                    Logger.Error(ex, "Get operation failed to retrieve data from local region " + configuration.RegionKey, new
                    {
                        LocalRegion = LocalConfiguration.RegionKey,
                        AccessedRegion = configuration.RegionKey,
                    });
                }

                throw;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Get operation failed due to an unhandled error", new
                {
                    LocalRegion = LocalConfiguration.RegionKey,
                    AccessedRegion = configuration.RegionKey,
                });

                throw;
            }

            return models;
        }

        protected virtual void OnRefresh() { }

        public void Refresh()
        {
            var config = SourceConfiguration;
            var key = GetKey(config);
            
            var record = config.Client.Get(null, key);
            if (record == null)
            {
                throw new InvalidOperationException("Unable to refresh model! It could not be retrieved from the database.");

            }

            Hydrate(config, key, record);
            OnRefresh();
        }

        public static TEntity GetByKey(AerospikeConfiguration configuration, Key key)
        {
            Record record = null;
            var model = new TEntity();

            try
            {
                record = configuration.Client.Get(null, key);

                if (record == null)
                {
                    return null;
                }

                model.Hydrate(configuration, key, record);
            }
            catch (AerospikeException ex)
            {
                var bins = record != null ? record.bins : null;
                if (LocalConfigID != configuration.RegionIdentifier) // Remote access, probably packet loss
                {
                    Logger.Error(ex, "Get operation from region " + LocalConfiguration.RegionKey + " failed to access remote region " + configuration.RegionKey, new
                    {
                        LocalRegion = LocalConfiguration.RegionKey,
                        AccessedRegion = configuration.RegionKey,
                        Bins = bins
                    });
                }
                else // Local Access, other issue
                {
                    Logger.Error(ex, "Get operation failed to retrieve data from local region " + configuration.RegionKey, new
                    {
                        LocalRegion = LocalConfiguration.RegionKey,
                        AccessedRegion = configuration.RegionKey,
                        Bins = bins
                    });
                }

                throw;
            }
            catch (Exception ex)
            {
                var bins = record != null ? record.bins : null;

                Logger.Error(ex, "Get operation failed due to an unhandled error", new
                {
                    LocalRegion = LocalConfiguration.RegionKey,
                    AccessedRegion = configuration.RegionKey,
                    Bins = bins
                });

                throw;
            }

            return model;
        }

        public static TEntity Get(AerospikeConfiguration configuration, params object[] keyValues)
        {
            var keyVals = string.Join(KeyDelimeter, keyValues);
            var key = new Key(GetKeySpace(configuration.RegionGroup), TableName, keyVals);
            return GetByKey(configuration, key);
        }

        public static IReadOnlyCollection<TEntity> MultiGet(AerospikeConfiguration configuration, IEnumerable<KeyInfo> keySet)
        {
            var keys = keySet.Select(keyInfo => GetKey(configuration, keyInfo.Values)).ToArray();
            return MultiGetByKeys(configuration, keys);
        }

        public static IReadOnlyCollection<TEntity> MultiGet(int regionID, IEnumerable<KeyInfo> keySet)
        {
            var configuration = ConfigByRegion[regionID];
            var keys = keySet.Select(keyInfo => GetKey(configuration, keyInfo.Values)).ToArray();
            return MultiGetByKeys(configuration, keys);
        }

        public static IReadOnlyCollection<TEntity> MultiGetLocal(IEnumerable<KeyInfo> keySet)
        {
            var keys = keySet.Select(keyInfo => GetKey(LocalConfiguration, keyInfo.Values)).ToArray();
            return MultiGetByKeys(LocalConfiguration, keys);
        }

        public static TEntity[] GetAllLocal<TProperty>(Expression<Func<TEntity, TProperty>> expr, TProperty keyValue)
        {
            return GetAll(LocalConfigID, expr, keyValue);
        }

        public static TEntity[] GetAll<TProperty>(int regionID, Expression<Func<TEntity, TProperty>> expr, TProperty keyValue)
        {
            return GetAll(GetConfiguration(regionID), expr, keyValue);
        }

        private static Statement GetIndexStatement<TProperty>(AerospikeConfiguration configuration, Expression<Func<TEntity, TProperty>> expr, TProperty keyValue)
        {

            var propertyName = GetPropertyNameFromExpression(expr);
            var binName = GetBinNameFromExpression(expr);

            var stmt = new Statement();
            stmt.SetNamespace(GetKeySpace(configuration.RegionGroup));
            stmt.SetSetName(TableName);
            stmt.SetIndexName(GetIndexName(propertyName));

            if (keyValue.GetType() == typeof(Int32) || keyValue.GetType() == typeof(Int64))
            {
                stmt.SetFilters(Filter.Equal(binName, Convert.ToInt64(keyValue)));
            }
            else if (keyValue.GetType().IsEnum)
            {
                stmt.SetFilters(Filter.Equal(binName, Convert.ToInt64(keyValue)));
            }
            else if (keyValue.GetType() == typeof(Guid))
            {
                var guid = new Guid(keyValue.ToString());
                stmt.SetFilters(Filter.Equal(binName, guid.ToString()));
            }
            else
            {
                stmt.SetFilters(Filter.Equal(binName, Convert.ToString(keyValue)));
            }

            return stmt;
        }

        /// <summary>
        /// An efficient way to just return a single bin from a series of records.
        /// </summary>
        /// <typeparam name="TProperty"></typeparam>
        /// <typeparam name="TBin"></typeparam>
        /// <param name="regionID"></param>
        /// <param name="expr"></param>
        /// <param name="keyValue"></param>
        /// <param name="binExpression"></param>
        /// <returns></returns>
        public static TBin[] GetAllBins<TProperty, TBin>(int regionID, Expression<Func<TEntity, TProperty>> expr, TProperty keyValue, Expression<Func<TEntity, TBin>> binExpression)
        {
            var configuration = ConfigByRegion[regionID];

            try
            {
                var stmt = GetIndexStatement(configuration, expr, keyValue);
                var propertyName = GetPropertyNameFromExpression(binExpression);
                var binName = GetBinNameFromExpression(binExpression);
                var columnInfo = Columns.FirstOrDefault(p => p.PropertyInfo.Name == propertyName);

                stmt.SetBinNames(binName);

                if (columnInfo == null)
                {
                    throw new Exception("Unknown column: " + propertyName);
                }

                using (var rs = configuration.Client.Query(null, stmt))
                {
                    var bins = new List<TBin>();

                    while (rs.Next())
                    {
                        object binValue;

                        if (!rs.Record.bins.TryGetValue(binName, out binValue))
                        {
                            continue;
                        }

                        var converted = (TBin)columnInfo.ConvertValue(binValue);
                        bins.Add(converted);
                    }
                    return bins.ToArray();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "GetAllBins operation failed for '" + typeof(TEntity).Name + "'", new
                {
                    LocalRegion = LocalConfiguration.RegionKey,
                    AccessedRegion = configuration.RegionKey,
                });
                throw;
            }
        }

        public static TEntity[] GetAll<TProperty>(AerospikeConfiguration configuration, Expression<Func<TEntity, TProperty>> expr, TProperty keyValue)
        {

            try
            {
                var stmt = GetIndexStatement(configuration, expr, keyValue);

                using (var rs = configuration.Client.Query(null, stmt))
                {
                    var entities = new List<TEntity>();

                    while (rs.Next())
                    {
                        var model = new TEntity();
                        try
                        {
                            model.Hydrate(configuration, rs.Key, rs.Record);
                            entities.Add(model);
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex,
                                "GetAll operation failed to hydrate '" + typeof(TEntity).Name + "' model from record.",
                                rs.Record.bins);
                        }
                    }
                    return entities.ToArray();
                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "GetAll operation failed for '" + typeof(TEntity).Name + "'", new
                {
                    LocalRegion = LocalConfiguration.RegionKey,
                    AccessedRegion = configuration.RegionKey,
                });
                throw;
            }
        }

        public static TEntity[] GetAllLocal()
        {
            return GetAll(LocalConfiguration);
        }

        public static TEntity[] GetAll(AerospikeConfiguration configuration)
        {
            var policy = new ScanPolicy { concurrentNodes = true, priority = Priority.LOW, includeBinData = true };
            var entities = new List<TEntity>();

            configuration.Client.ScanAll(policy, GetKeySpace(configuration.RegionGroup), TableName, (key, record) =>
            {
                var model = new TEntity();
                model.Hydrate(configuration, key, record);
                
                lock (entities)
                {
                    entities.Add(model);
                }
            });

            return entities.ToArray();
        }

        public static void BatchOperateLocal(int batchSize, Action<IEnumerable<TEntity>> action)
        {
            BatchOperate(LocalConfiguration, batchSize, action);
        }

        public static void BatchOperate(int regionID, int batchSize, Action<IEnumerable<TEntity>> action)
        {
            BatchOperate(GetConfiguration(regionID), batchSize, action);
        }

        public static void BatchOperate(AerospikeConfiguration configuration, int batchSize, Action<IEnumerable<TEntity>> action, bool runConcurrently = false)
        {
            var policy = new ScanPolicy()
            {
                concurrentNodes = runConcurrently,
                priority = Priority.LOW,
                includeBinData = true,
                timeout = 0,
                maxRetries = 1,
            };

            var entities = new List<TEntity>();
            configuration.Client.ScanAll(policy, GetKeySpace(configuration.RegionGroup), TableName, (key, record) =>
            {
                var model = new TEntity();
                model.Hydrate(configuration, key, record);
                lock (entities)
                {
                    entities.Add(model);
                    if (entities.Count >= batchSize)
                    {
                        action(entities);
                        entities.Clear();
                    }
                }
            });

            action(entities);

        }

        public static void UpdateAllRecords(AerospikeConfiguration configuration, Action<Key, Record> action, bool runConcurrently = true)
        {
            var policy = new ScanPolicy()
            {
                concurrentNodes = runConcurrently,
                priority = Priority.LOW,
                includeBinData = true,
            };

            configuration.Client.ScanAll(policy, GetKeySpace(configuration.RegionGroup), TableName, (key, record) => action(key, record));
        }

        public static Key[] ScanAllKeys(AerospikeConfiguration configuration)
        {

            var keys = new ConcurrentBag<Key>();
            
            configuration.Client.ScanAll(null, GetKeySpace(configuration.RegionGroup), TableName, (key, record) =>
            {
                keys.Add(key);
            });

            return keys.ToArray();
        }

        public static void BatchOperateOnIndexLocal<TProperty>(Expression<Func<TEntity, TProperty>> expr, TProperty keyValue, int batchSize, Action<IEnumerable<TEntity>> action)
        {
            BatchOperateOnIndex(LocalConfiguration, expr, keyValue, batchSize, action);
        }

        public static void BatchOperateOnIndex<TProperty>(int regionID, Expression<Func<TEntity, TProperty>> expr, TProperty keyValue, int batchSize, Action<IEnumerable<TEntity>> action)
        {
            BatchOperateOnIndex(GetConfiguration(regionID), expr, keyValue, batchSize, action);
        }

        public static void BatchOperateOnIndexRecordsLocal<TProperty>(Expression<Func<TEntity, TProperty>> expr, TProperty keyValue, int batchSize, Action<Record[]> action)
        {
            BatchOperateOnIndexRecords(LocalConfiguration, expr, keyValue, batchSize, action);
        }

        public static void BatchOperateOnIndexRecords<TProperty>(AerospikeConfiguration configuration, Expression<Func<TEntity, TProperty>> expr, TProperty keyValue, int batchSize, Action<Record[]> action)
        {

            try
            {
                var entities = new List<Record>();
                var stmt = GetIndexStatement(configuration, expr, keyValue);
                using (var rs = configuration.Client.Query(new QueryPolicy { timeout = 0, maxConcurrentNodes = 0, }, stmt))
                {
                    while (rs.Next())
                    {
                        entities.Add(rs.Record);

                        if (entities.Count >= batchSize)
                        {
                            action(entities.ToArray());
                            entities.Clear();
                        }
                    }
                }

                action(entities.ToArray());

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "BatchOperateOnIndexRecords operation failed for '" + typeof(TEntity).Name + "'", new
                {
                    LocalRegion = LocalConfiguration.RegionKey,
                    AccessedRegion = configuration.RegionKey,
                });

                throw;
            }
        }

        public static void BatchOperateOnIndex<TProperty>(AerospikeConfiguration configuration, Expression<Func<TEntity, TProperty>> expr, TProperty keyValue, int batchSize, Action<IEnumerable<TEntity>> action)
        {

            try
            {
                var stmt = GetIndexStatement(configuration, expr, keyValue);
                var entities = new List<TEntity>();
                using (var rs = configuration.Client.Query(new QueryPolicy { timeout = 0, maxConcurrentNodes = 0 }, stmt))
                {
                    while (rs.Next())
                    {
                        var model = new TEntity();
                        try
                        {
                            model.Hydrate(configuration, rs.Key, rs.Record);
                            entities.Add(model);
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex,
                                "GetAll operation failed to hydrate '" + typeof(TEntity).Name + "' model from record.",
                                rs.Record.bins);
                        }

                        if (entities.Count >= batchSize)
                        {
                            action(entities);
                            entities.Clear();
                        }
                    }
                }

                action(entities);

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "GetAll operation failed for '" + typeof(TEntity).Name + "'", new
                {
                    LocalRegion = LocalConfiguration.RegionKey,
                    AccessedRegion = configuration.RegionKey,
                });
                throw;
            }

        }
        
        public Bin[] GetBins(params Expression<Func<TEntity, object>>[] expressions)
        {
            Bin[] bins;

            if (expressions.Any())
            {
                var binNames = new HashSet<string>();
                foreach (var expr in expressions)
                {
                    string name = GetPropertyNameFromExpression(expr);
                    binNames.Add(name);
                }
                bins = GetBins(binNames);
            }
            else
            {
                bins = GetBins();
            }

            return bins;
        }


        public Bin[] GetBins(HashSet<string> limit = null)
        {
            var bins = new List<Bin>();

            foreach (var col in Columns)
            {
                if (col.ColumnAttribute.IsLargeSet)
                {
                    continue;
                }

                if (limit == null || limit.Contains(col.PropertyInfo.Name))
                {
                    bins.Add(col.GetBin(this));
                }
            }

            return bins.ToArray();
        }

        public static string GetBinNameFromPropertyName(string propertyName)
        {
            string binName;

            if (!PropertyNameToColumnName.TryGetValue(propertyName, out binName))
            {
                throw new ArgumentException("Unknown column: " + propertyName);
            }

            return binName;
        }

        /// <summary>
        /// The aerospike Configuration which was used to retrieve this record
        /// </summary>
        private AerospikeConfiguration _sourceConfiguration = null;

        protected AerospikeConfiguration SourceConfiguration
        {
            get { return _sourceConfiguration ?? LocalConfiguration; }
        }

        protected int CurrentGeneration;
        protected Key Key;

        public void Hydrate(AerospikeConfiguration configuration, Key key, Record row)
        {
            _sourceConfiguration = configuration;
            CurrentGeneration = row.generation;
            Key = key;

            foreach (var col in Columns)
            {
                if (col.ColumnAttribute.IsLargeSet) // Special case for large sets
                {
                    col.PropertyInfo.SetValue(this, col.ConvertLargeSet(configuration.Client, key, col.ColumnAttribute.Name));
                    continue;
                }

                if (!row.bins.ContainsKey(col.ColumnAttribute.Name))
                {
                    continue;
                }

                var value = row.bins[col.ColumnAttribute.Name];

                try
                {
                    var convertedValue = col.ConvertValue(value);
                    col.PropertyInfo.SetValue(this, convertedValue);
                }
                catch (Exception ex)
                {                    
                    Logger.Error(ex, "Failed to convert value during hydration!", new { Table = TableName, Column = col.ColumnAttribute.Name, Key = key, Record = row, Value = value });
                    continue;
                }
            }
        }

        protected Key GetKey(AerospikeConfiguration configuration)
        {
            return new Key(GetKeySpace(configuration.RegionGroup), TableName, GenerateKey(this));
        }

        protected Key GetMirrorKey(AerospikeConfiguration configuration)
        {
            if (!string.IsNullOrEmpty(configuration.MirrorKeyspace))
            {
                return new Key(configuration.MirrorKeyspace, TableName, GenerateKey(this));    
            }

            return GetKey(configuration);
            
        }

        protected static Key GetKey(AerospikeConfiguration configuration, params object[] keyVals)
        {
            return new Key(GetKeySpace(configuration.RegionGroup), TableName, string.Join(KeyDelimeter, keyVals));
        }

        public static void DeleteByKeyLocal(params object[] keyVals)
        {
            DeleteByKey(LocalConfiguration, keyVals);
        }

        public static void DeleteByKey(int regionID, params object[] keyVals)
        {
            DeleteByKey(ConfigByRegion[regionID], keyVals);
        }

        public static void DeleteByKey(AerospikeConfiguration configuration, params object[] keyVals)
        {
            var key = new Key(GetKeySpace(configuration.RegionGroup), TableName, string.Join(KeyDelimeter, keyVals));
            configuration.Client.Delete(null, key);
        }
        
        public static void DurableDeleteByKey(AerospikeConfiguration configuration, Key key)
        {         
            var writePolicy = new WritePolicy { timeout = 5000, commitLevel = CommitLevel.COMMIT_MASTER, expiration = 1 };
            configuration.Client.Touch(writePolicy, key);
            configuration.Client.Delete(null, key);
        }

        public void InsertLocal(UpdateMode updateMode = UpdateMode.Default, int ttlSeconds = 0)
        {
            if (ReplicationMode == ReplicationMode.HomeRegion)
            {
                Insert(((IModelRegion)this).RegionID, updateMode, ttlSeconds);
            }
            else
            {
                Insert(LocalConfiguration, updateMode);
            }                     
        }

        protected virtual void OnInserted()
        {
            
        }

        protected virtual void OnBeforeInsertOrUpdate()
        {

        }

        public void Insert(int regionID, UpdateMode updateMode = UpdateMode.Default, int ttlSeconds = 0)
        {
            Insert(ConfigByRegion[regionID], updateMode, ttlSeconds);
        }

        public void Insert(AerospikeConfiguration config, UpdateMode updateMode = UpdateMode.Default, int ttlSeconds = 0)
        {
            _sourceConfiguration = config;

            try
            {
                OnBeforeInsertOrUpdate();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process pre-inseration event!");
            }  
            
            // Convert each property into a bin
            var bins = GetBins();

            var key = GetKey(config);

            if (!TryValidate())
            {
                return;
            }
            

            // Write record
            config.Client.Put(GetInsertPolicy(updateMode, ttlSeconds), key, bins);

            // If any of this record has any large set columns, initialize them
            foreach (var col in Columns)
            {                
                if (col.ColumnAttribute.IsLargeSet) // Special case for large sets
                {
                    col.PropertyInfo.SetValue(this, col.ConvertLargeSet(config.Client, key, col.ColumnAttribute.Name));
                }
            }

            // Run any post-insert code
            try
            {
                OnInserted();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process post-inseration event!");
            }

            Mirror(config, bins, false);            
        }

        private void Mirror(AerospikeConfiguration config, Bin[] bins, bool isPartialUpdate)
        {
            // If configured for mirroring, save this via the mirror client
            if (!config.ShouldMirror(TableName))
            {
                return;                
            }

            try
            {
                var mirrorBins = isPartialUpdate ? GetBins() : bins;
                config.MirrorClient.Put(null, GetMirrorKey(config), mirrorBins);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to mirror data in " + TableName);
            }
            
        }

       

        public void Update(UpdateMode updateMode, params Expression<Func<TEntity, object>>[] expressions)
        {
            if (SourceConfiguration == null)
            {
                throw new InvalidOperationException("SourceConfiguration must be set, before a model can be updated.");
            }

            var config = ReplicationMode == ReplicationMode.HomeRegion
                ? ConfigByRegion[((IModelRegion) this).RegionID]
                : SourceConfiguration;

            Update(updateMode, config, expressions);
        }

        public void Update(params Expression<Func<TEntity, object>>[] expressions)
        {
            if (SourceConfiguration == null)
            {
                throw new InvalidOperationException("SourceConfiguration must be set, before a model can be updated.");
            }

            Update(UpdateMode.Default, SourceConfiguration, expressions);
        }

        public void AddOrUpdate(int regionID)
        {
            if (SourceConfiguration != null)
            {
                Update();
            }
            else
            {
                Insert(ConfigByRegion[regionID]);
            }
        }

        public void AddOrUpdateLocal()
        {
            if (_sourceConfiguration != null)
            {
                Update();
            }
            else
            {
                Insert(LocalConfiguration);
            }
        }

        public void CopyToRegion(int regionID)
        {
            CopyToRegion(ConfigByRegion[regionID], GetBins());
        }

        private void CopyToRegion(AerospikeConfiguration config, Bin[] bins)
        {         
            config.Client.Put(null, GetKey(config), bins);         
        }

        private static string GetPropertyNameFromExpression<TProperty>(Expression<Func<TEntity, TProperty>> expr)
        {
            MemberExpression body;
            if (expr.Body is MemberExpression)
            {
                body = (MemberExpression)expr.Body;
            }
            else
            {
                var unary = (UnaryExpression)expr.Body;
                body = (MemberExpression)unary.Operand;
            }

            return body.Member.Name;
        }

        public static string GetBinNameFromExpression<TProperty>(Expression<Func<TEntity, TProperty>> expr)
        {
            var propertyName = GetPropertyNameFromExpression(expr);
            return GetBinNameFromPropertyName(propertyName);
        }

        private static bool HasLoggedReplicationWriteError = false;

        public void Update(UpdateMode updateMode, AerospikeConfiguration config, params Expression<Func<TEntity, object>>[] expressions)
        {

            if (ReplicationMode == ReplicationMode.HomeRegion)
            {
                var homeRegion = (this as IModelRegion).RegionID;
                if (SourceConfiguration.RegionIdentifier != homeRegion)
                {
                    // Throw exception or log warning
#if false && DEBUG
                    throw new InvalidOperationException("You must retrieve this model from its home region to update it.");
#else
                    if (!HasLoggedReplicationWriteError)
                    {
                        HasLoggedReplicationWriteError = true;
                        Logger.Fatal("You should retrieve this model from its home region to update it.", new { Environment.StackTrace });    
                    }                    
#endif
                }
            }

            try
            {
                OnBeforeInsertOrUpdate();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process pre-update event!");
            }  

            Bin[] bins;

            if (expressions.Any())
            {
                var binNames = new HashSet<string>();
                foreach (var expr in expressions)
                {
                    var name = GetPropertyNameFromExpression(expr);
                    binNames.Add(name);
                }
                bins = GetBins(binNames);
            }
            else
            {
                bins = GetBins();
            }


            if (!TryValidate())
            {
                return;
            }

            var policy = GetUpdatePolicy(updateMode, CurrentGeneration);

            var key = GetKey(config);

            // Write record
            config.Client.Put(policy, key, bins);

            Mirror(config, bins, expressions.Any());            
        }

        public static void UpdateByLocalKey(UpdateMode updateMode, Dictionary<string, object> binValues, params object[] keyVals)
        {
            UpdateByKey(updateMode, LocalConfiguration, binValues, keyVals);
        }

        public static void UpdateByKey(UpdateMode updateMode, int regionID, Dictionary<string, object> binValues, params object[] keyVals)
        {
            UpdateByKey(updateMode, ConfigByRegion[regionID], binValues, keyVals);
        }

        public static void UpdateByKey(UpdateMode updateMode, AerospikeConfiguration configuration, Dictionary<string, object> binValues, params object[] keyVals)
        {
            var key = new Key(GetKeySpace(configuration.RegionGroup), TableName, string.Join(KeyDelimeter, keyVals));
            var bins = binValues.Select(p => new Bin(p.Key, p.Value)).ToArray();

            var policy = GetUpdatePolicy(updateMode);

            configuration.Client.Put(policy, key, bins);
        }

        private static WritePolicy GetInsertPolicy(UpdateMode mode, int ttlSeconds)
        {
            return AerospikeConfiguration.GetWritePolicy(mode, ttlSeconds);
        }


        private static WritePolicy GetUpdatePolicy(UpdateMode mode, int generation = 0)
        {
            switch (mode)
            {
                case UpdateMode.Concurrent:
                    return new WritePolicy
                    {
                        recordExistsAction = RecordExistsAction.UPDATE_ONLY,
                        generation = generation,
                        generationPolicy = GenerationPolicy.EXPECT_GEN_EQUAL,
                        consistencyLevel = ConsistencyLevel.CONSISTENCY_ALL,
                        commitLevel = CommitLevel.COMMIT_ALL
                    };
                    
                case UpdateMode.Fast:
                    return AerospikeConfiguration.DefaultFastWriteUpdatePolicy;
           
                default:
                    return AerospikeConfiguration.DefaultUpdatePolicy;                    

            }
        }
        
        public void DeleteRemote(int regionID)
        {
            Delete(ConfigByRegion[regionID]);
        }

        public void Delete()
        {
            if (SourceConfiguration == null)
            {
                throw new InvalidOperationException("Source region ID must be set, before a model can be updated.");
            }

            Delete(SourceConfiguration);
        }

        public void Delete(AerospikeConfiguration config)
        {
            config.Client.Delete(null, GetKey(config));
        }

        public void DurableDelete()
        {
            var writePolicy = new WritePolicy {timeout = 5000, commitLevel = CommitLevel.COMMIT_MASTER, expiration = 1};
            SourceConfiguration.Client.Touch(writePolicy, GetKey(SourceConfiguration));
            SourceConfiguration.Client.Delete(null, GetKey(SourceConfiguration));
        }

        /// <summary>
        /// Method which can provide business logic for determining if a model is in a valid state before saving.
        /// If invalid, it should throw an exception with a useful message.
        /// </summary>
        protected virtual void Validate()
        {

        }

        private bool TryValidate()
        {
            try
            {
                Validate();
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed validation", new { Model = this, Environment.StackTrace });
                return false;
            }

            return true;
        }

        private static string GetIndexName(string binName)
        {
            return "IDX_" + TableName + "_" + binName;
        }

        private static IndexType GetIndexTypeFromClrType(Type t)
        {

            if (t.IsEnum || t == typeof(Int32) || t == typeof(Int64) || t == typeof(float) || t == typeof(double))
            {
                return IndexType.NUMERIC;
            }


            return IndexType.STRING;


        }

        private static void CreateIndex(AerospikeConfiguration config, string binName, Type type)
        {
            if (AerospikeConfiguration.CreateIndexes)
            {
                var indexType = GetIndexTypeFromClrType(type);                
                var task = config.Client.CreateIndex(null, GetKeySpace(config.RegionGroup), TableName, GetIndexName(binName), binName, indexType);
                task.Wait();

                if (config.ShouldMirror(TableName))
                {
                    try
                    {
                        task = config.MirrorClient.CreateIndex(null, config.MirrorKeyspace, TableName, GetIndexName(binName), binName, indexType);
                        task.Wait();
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Failed to create index on mirrored cluster for table: " + TableName);
                    }
                    
                }
            }
        }

        private static readonly ColumnDefinition[] Columns;

        private static readonly Dictionary<string, string> PropertyNameToColumnName;

        private static readonly Func<object, string> GenerateKey;

        public string GetGeneratedKey()
        {
            return GenerateKey(this);
        }

        static BaseTable()
        {

            try
            {
                if (!AerospikeConfiguration.IsInitialized)
                {
                    throw new InvalidOperationException("AerospikeConfiguration must be initialized before any models are accessed.");
                }

                var tableDefinition = typeof(TEntity).GetCustomAttribute<TableDefinitionAttribute>();

                if (tableDefinition == null)
                {
                    Logger.Error("Table definition is missing for type: " + typeof(TEntity).FullName);
                    throw new Exception("All entities must have a TableDefinition attribute.");
                }


                var clusterConfigs = AerospikeConfiguration.Configurations.Where(p => p.Sets.Any(s => s.Equals(tableDefinition.TableName, StringComparison.InvariantCultureIgnoreCase))).ToArray();

                if (!clusterConfigs.Any())
                {
                    Logger.Error("No configs found for this type: " + typeof(TEntity).FullName);
                    throw new Exception("No configs found for this set: " + tableDefinition.TableName);
                }

                RegionKeys = clusterConfigs.ToDictionary(p => p.RegionIdentifier, p => p.RegionGroup.ToLowerInvariant());
                AllConfigurations = clusterConfigs;
                var remoteSessions = new List<AerospikeConfiguration>();

                // Iterate over each configuration, and create a session for that region
                foreach (var config in clusterConfigs)
                {
                    if (config.IsLocal && config.Client == null)
                    {
                        Logger.Error("Unable to open a connection to the local cluster!");
                        throw new Exception("Unable to establish a session to the local cluster.");
                    }

                    ConfigByRegion[config.RegionIdentifier] = config;

                    if (config.IsLocal)
                    {
                        LocalConfigID = config.RegionIdentifier;
                        LocalConfiguration = config;
                    }
                    else
                    {
                        remoteSessions.Add(config);
                    }

                }

                RemoteConfigurations = remoteSessions.ToArray();


                // Set our keyspace
                ReplicationMode = tableDefinition.ReplicationMode;

                if (tableDefinition.KeySpace == null)
                {
                    BaseKeySpace = AerospikeConfiguration.DefaultKeySpace.ToLower();
                }
                else
                {
                    BaseKeySpace = tableDefinition.KeySpace.ToLower();
                }

                TableName = tableDefinition.TableName;

                var columnDefinitions = typeof(TEntity).GetProperties();

                var columns = new List<ColumnDefinition>();
                var propertyNameToColumName = new Dictionary<string, string>();
                var keyColumns = new List<ColumnDefinition>();

                foreach (var col in columnDefinitions)
                {
                    var colAttribute = col.GetCustomAttribute<ColumnAttribute>();
                    if (colAttribute == null)
                    {
                        continue;
                    }

                    if (colAttribute.IsIndexed)
                    {
                        try
                        {
                            CreateIndex(LocalConfiguration, colAttribute.Name, col.PropertyType);
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex, "Failed to create index '" + colAttribute.Name + "' in table '" + TableName + "'");
                        }

                    }

                    var columnDefinition = new ColumnDefinition(col, colAttribute);

                    if (colAttribute.KeyOrdinal > 0)
                    {
                        if (keyColumns.Any(p => p.ColumnAttribute.KeyOrdinal == colAttribute.KeyOrdinal))
                        {
                            throw new InvalidOperationException("Duplicate key ordinal " + colAttribute.KeyOrdinal + " found on column: " + colAttribute.Name);
                        }

                        keyColumns.Add(columnDefinition);

                    }

                    if (propertyNameToColumName.ContainsKey(columnDefinition.PropertyInfo.Name))
                    {
                        throw new InvalidOperationException("Duplicate property name found: " + columnDefinition.PropertyInfo.Name);
                    }

                    propertyNameToColumName.Add(columnDefinition.PropertyInfo.Name, columnDefinition.ColumnAttribute.Name);
                    columns.Add(columnDefinition);

                }

                if (keyColumns.Count == 0)
                {
                    throw new InvalidOperationException("At least one column must be defined as a key column.");
                }

                var orderedKeyCols = keyColumns.OrderBy(p => p.ColumnAttribute.KeyOrdinal).ToArray();

                GenerateKey = (model) =>
                {
                    var keyParts = new List<string>();
                    foreach (var p in orderedKeyCols)
                    {
                        var value = p.PropertyInfo.GetValue(model);

                        if (IsDefault(value))
                        {
                            if (!p.ColumnAttribute.IsOptional)
                            {
                                keyParts.Add(value == null ? "<null>" : value.ToString());
                            }
                        }
                        else
                        {
                            keyParts.Add(value.ToString());
                        }

                    }
                    return string.Join(KeyDelimeter, keyParts.ToArray());
                };

                Columns = columns.ToArray();
                PropertyNameToColumnName = propertyNameToColumName;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to initialize BaseTable for type: " + typeof(TEntity).Name);
                throw;
            }            
        }

        private static bool IsDefault(object value)
        {
            if (value == null)
            {
                return true;
            }

            if (value is int && (int)value == 0)
            {
                return true;
            }

            if (value.GetType().IsEnum && (int)value == 0)
            {
                return true;
            }

            return false;
        }

        public int IncrementCounterLocal(UpdateMode updateMode, Expression<Func<TEntity, object>> expr, int amount)
        {
            return IncrementCounter(LocalConfiguration, updateMode, expr, amount);
        }

        public int IncrementCounter(int regionID, UpdateMode updateMode, Expression<Func<TEntity, object>> expr, int amount)
        {
            return IncrementCounter(ConfigByRegion[regionID], updateMode, expr, amount);
        }

        public int IncrementCounter(AerospikeConfiguration config, UpdateMode updateMode, Expression<Func<TEntity, object>> expr, int amount)
        {            
            var name = GetBinNameFromExpression(expr);
            var bin = new Bin(name, amount);            
            var addOperation = Operation.Add(bin);            
            var getOperation = Operation.Get(name);
            var key = GetKey(config);
            var record = config.Client.Operate(GetUpdatePolicy(updateMode), key, addOperation, getOperation);

            if (record == null)
            {
                Logger.Warn("Increment operation failed. A record was not returned.", new { key });
                throw new AerospikeOperationException("Increment operation failed. A record was not returned.");
            }

            return Convert.ToInt32(record.bins[bin.name]);
        }

        public int IncrementCounterAndSetValue(UpdateMode updateMode, Expression<Func<TEntity, object>> incrementExpr, int amount, Expression<Func<TEntity, object>> setExpr, object value)
        {
            return IncrementCounterAndSetValue(SourceConfiguration, updateMode, incrementExpr, amount, setExpr, value);
        }

        public int IncrementCounterAndSetValue(int regionID, UpdateMode updateMode, Expression<Func<TEntity, object>> incrementExpr, int amount, Expression<Func<TEntity, object>> setExpr, object value)
        {
            return IncrementCounterAndSetValue(ConfigByRegion[regionID], updateMode, incrementExpr, amount, setExpr, value);
        }

        public int IncrementCounterAndSetValue(AerospikeConfiguration config, UpdateMode updateMode, Expression<Func<TEntity, object>> incrementExpr, int amount, Expression<Func<TEntity, object>> setExpr, object value)
        {
            var incrementName = GetBinNameFromExpression(incrementExpr);
            var incrementBin = new Bin(incrementName, amount);
            var addOperation = Operation.Add(incrementBin);
            var getOperation = Operation.Get(incrementName);

            var setName = GetBinNameFromExpression(setExpr);
            var setBin = new Bin(setName, value);
            var setOperation = Operation.Put(setBin);
            var key = GetKey(config);
            var record = config.Client.Operate(GetUpdatePolicy(updateMode), key, addOperation, setOperation, getOperation);
            
            if (record == null)
            {
                Logger.Warn("Increment and set operation failed. A record was not returned.", new { key });
                throw new AerospikeOperationException("Increment operation failed. A record was not returned.");
            }

            return Convert.ToInt32(record.bins[incrementName]);
        }

        public static int IncrementCounterAndSetValue(int regionID, UpdateMode updateMode, Expression<Func<TEntity, object>> incrementExpr, int amount, Expression<Func<TEntity, object>> setExpr, object value, params object[] keyVals)
        {
            return IncrementCounterAndSetValue(ConfigByRegion[regionID], updateMode, incrementExpr, amount, setExpr, value, keyVals);
        }

        public static int IncrementCounterAndSetValue(AerospikeConfiguration config, UpdateMode updateMode, Expression<Func<TEntity, object>> incrementExpr, int amount, Expression<Func<TEntity, object>> setExpr, object value, params object[] keyVals)
        {
            var key = new Key(GetKeySpace(config.RegionGroup), TableName, string.Join(KeyDelimeter, keyVals));

            var incrementName = GetBinNameFromExpression(incrementExpr);
            var incrementBin = new Bin(incrementName, amount);
            var addOperation = Operation.Add(incrementBin);
            var getOperation = Operation.Get(incrementName);

            var setName = GetBinNameFromExpression(setExpr);
            var setBin = new Bin(setName, value);            
            var setOperation = Operation.Put(setBin);

            var record = config.Client.Operate(GetUpdatePolicy(updateMode), key, addOperation, setOperation, getOperation);
            if (record == null)
            {
                Logger.Warn("Increment operation failed. A record was not returned.", new { keys = keyVals });
                throw new AerospikeOperationException("Increment operation failed. A record was not returned.");
            }

            if (record.bins == null)
            {
                Logger.Warn("Increment operation failed. Bins were not included in the result.", new { keys = keyVals });
                throw new AerospikeOperationException("Increment operation failed. Bins were not included in the result.");
            }

            if (!record.bins.ContainsKey(incrementName))
            {
                Logger.Warn("Increment operation failed. The result did not include the bins requsted", new { keys = keyVals, incrementName });
                throw new AerospikeOperationException("Increment operation failed. The result did not include the bins requsted: " + incrementName);
            }

            return Convert.ToInt32(record.bins[incrementName]);
        }

        public static void ResetCounterAndSetValues(int regionID, UpdateMode updateMode, KeyInfo keyInfo, Expression<Func<TEntity, object>> resetExpr, params Tuple<Expression<Func<TEntity, object>>, object>[] setExpressions)
        {
            ResetCounterAndSetValues(ConfigByRegion[regionID], updateMode, keyInfo, resetExpr, setExpressions);

        }

        public static void ResetCounterAndSetValues(AerospikeConfiguration config, UpdateMode updateMode, KeyInfo keyInfo, Expression<Func<TEntity, object>> resetExpr, params Tuple<Expression<Func<TEntity, object>>, object>[] setExpressions)
        {
            var key = new Key(GetKeySpace(config.RegionGroup), TableName, string.Join(KeyDelimeter, keyInfo.Values));
            var operations = new List<Operation>();

            var resetName = GetBinNameFromExpression(resetExpr);
            var resetBin = new Bin(resetName, 0);
            operations.Add(Operation.Put(resetBin));

            foreach (var setExpr in setExpressions)
            {
                var setName = GetBinNameFromExpression(setExpr.Item1);
                var setBin = new Bin(setName, setExpr.Item2);
                operations.Add(Operation.Put(setBin));
            }

            config.Client.Operate(GetUpdatePolicy(updateMode), key, operations.ToArray());
        }

        public void ResetCounterAndSetValueLocal(UpdateMode updateMode, Expression<Func<TEntity, object>> resetExpr, Expression<Func<TEntity, object>> setExpr, object value)
        {
            ResetCounterAndSetValue(LocalConfiguration, updateMode, resetExpr, setExpr, value);
        }


        public void ResetCounterAndSetValue(int regionID, UpdateMode updateMode, Expression<Func<TEntity, object>> resetExpr, Expression<Func<TEntity, object>> setExpr, object value)
        {
            ResetCounterAndSetValue(ConfigByRegion[regionID], updateMode, resetExpr, setExpr, value);
        }

        public void ResetCounterAndSetValue(AerospikeConfiguration config, UpdateMode updateMode, Expression<Func<TEntity, object>> resetExpr, Expression<Func<TEntity, object>> setExpr, object value)
        {
            var resetName = GetBinNameFromExpression(resetExpr);
            var resetBin = new Bin(resetName, 0);
            var addOperation = Operation.Put(resetBin);

            var setName = GetBinNameFromExpression(setExpr);
            var setBin = new Bin(setName, value);
            var setOperation = Operation.Put(setBin);

            config.Client.Operate(GetUpdatePolicy(updateMode), GetKey(config), addOperation, setOperation);
        }

        public void ResetCounterAndSetValue(UpdateMode updateMode, Expression<Func<TEntity, object>> resetExpr, Expression<Func<TEntity, object>> setExpr, object value)
        {
            ResetCounterAndSetValue(SourceConfiguration, updateMode, resetExpr, setExpr, value);
        }        

        public object ShallowClone()
        {
            return MemberwiseClone();
        }

        #region Range Query

        public static TEntity[] GetAllInRangeLocal<TProperty>(Expression<Func<TEntity, TProperty>> expr, long min = long.MinValue, long max = long.MaxValue)
        {
            return GetAllInRange(LocalConfiguration, expr, min, max);
        }

        public static TEntity[] GetAllInRange<TProperty>(int regionID, Expression<Func<TEntity, TProperty>> expr, long min = long.MinValue, long max = long.MaxValue)
        {
            return GetAllInRange(GetConfiguration(regionID), expr, min, max);
        }

        public static TEntity[] GetAllInRange<TProperty>(AerospikeConfiguration configuration, Expression<Func<TEntity, TProperty>> expr, long min = long.MinValue, long max = long.MaxValue)
        {
            try
            {
                var stmt = GetRangeIndexStatement(configuration, expr, min, max);

                using (var rs = configuration.Client.Query(null, stmt))
                {
                    var entities = new List<TEntity>();

                    while (rs.Next())
                    {
                        var model = new TEntity();
                        try
                        {
                            model.Hydrate(configuration, rs.Key, rs.Record);
                            entities.Add(model);
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex,
                                "GetAll operation failed to hydrate '" + typeof(TEntity).Name + "' model from record.",
                                rs.Record.bins);
                        }
                    }
                    return entities.ToArray();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "GetAll operation failed for '" + typeof(TEntity).Name + "'", new
                {
                    LocalRegion = LocalConfiguration.RegionKey,
                    AccessedRegion = configuration.RegionKey,
                });
                throw;
            }
        }

        private static Statement GetRangeIndexStatement<TProperty>(AerospikeConfiguration configuration, Expression<Func<TEntity, TProperty>> expr, long min = long.MinValue, long max = long.MaxValue)
        {
            var propertyName = GetPropertyNameFromExpression(expr);
            var binName = GetBinNameFromExpression(expr);

            var stmt = new Statement();
            stmt.SetNamespace(GetKeySpace(configuration.RegionGroup));
            stmt.SetSetName(TableName);
            stmt.SetIndexName(GetIndexName(propertyName));
            stmt.SetFilters(Filter.Range(binName, min, max));
            return stmt;
        }

        #endregion    
    }
}
