﻿using Aerospike.Client;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using Curse.Extensions;

namespace Curse.Aerospike
{
    public class ColumnDefinition
    {
        public ColumnDefinition(PropertyInfo propertyInfo, ColumnAttribute attribute)
        {
            PropertyInfo = propertyInfo;
            ColumnAttribute = attribute;

            var type = propertyInfo.PropertyType;

            if (attribute.IsLargeSet)
            {
                InitializeLargeSetConverter(type);
            }
            else
            {
                InitializeConverter(propertyInfo, type);
            }
           
            if (type.IsGenericType && type.GetGenericTypeDefinition() == typeof(Dictionary<,>))
            {
                GetBin = (v) => new Bin(ColumnAttribute.Name, (IDictionary) PropertyInfo.GetValue(v));
            }

            else if (type == typeof(HashSet<int>))
            {
                GetBin = (v) =>
                {
                    var rawValue = PropertyInfo.GetValue(v) as HashSet<int>;
                    var value = rawValue == null ? new int[0] : rawValue.ToArray();
                    return new Bin(ColumnAttribute.Name, value);
                };
            }
            else if (type == typeof(HashSet<Guid>))
            {
                GetBin = (v) =>
                {
                    var rawValue = PropertyInfo.GetValue(v) as HashSet<Guid>;
                    var value = rawValue == null ? new string[0] : rawValue.Select(p => p.ToString()).ToArray();
                    return new Bin(ColumnAttribute.Name, value);
                };
            }
            else if (type == typeof(HashSet<string>))
            {
                GetBin = (v) =>
                {
                    var rawValue = PropertyInfo.GetValue(v) as HashSet<string>;
                    var value = rawValue == null ? new string[0] : rawValue.ToArray();
                    return new Bin(ColumnAttribute.Name, value);
                };
            }
            else if (type == typeof(Guid))
            {
                GetBin = (v) => new Bin(ColumnAttribute.Name, PropertyInfo.GetValue(v).ToString());
            }
  
            else
            {
                GetBin = (v) => new Bin(ColumnAttribute.Name, PropertyInfo.GetValue(v));
            }
        }

        private void InitializeLargeSetConverter(Type type)
        {
            if (type == typeof(LargeDictionary<int, string>))
            {
                ConvertLargeSet = (client, key, binName) =>
                {

                    var largeMap = client.GetLargeMap(null, key, binName, null);
                    return new LargeDictionary<int, string>(largeMap, Convert.ToInt32, Convert.ToString);
                };

                return;
            }

            if (type == typeof (LargeList<>) ||
                (type.IsGenericType && type.GetGenericTypeDefinition() == typeof (LargeList<>)))
            {
                ConvertLargeSet = (client, key, binName) =>
                {

                    var largeList = client.GetLargeList(null, key, binName, null);                    
                    return Activator.CreateInstance(type, largeList);
                };

                return;
            }


            throw new InvalidOperationException("Unsupported large set type: " + type.FullName);

        }


        private void InitializeConverter(PropertyInfo propertyInfo, Type type)
        {
            if (type == typeof(Int32))
            {
                ConvertValue = (v) => Convert.ToInt32(v);
            }
            else if (type == typeof(Int64))
            {
                ConvertValue = (v) => Convert.ToInt64(v);
            }
            else if (type.IsEnum)
            {
                ConvertValue = (v) => Enum.ToObject(propertyInfo.PropertyType, v);
            }
            else if (type == typeof(bool))
            {
                ConvertValue = (v) => Convert.ToBoolean(v);
            }
            else if (type == typeof(bool?))
            {
                ConvertValue = (v) =>
                {
                    if (v == DBNull.Value)
                    {
                        return null;
                    }
                    return Convert.ToBoolean(v);
                };
            }
            else if (type == typeof(Guid))
            {
                ConvertValue = (v) => new Guid(v.ToString());
            }
            else if (type.IsGenericType && type.GetGenericTypeDefinition() == typeof(Dictionary<,>) && 
                type.GetGenericArguments().First() == typeof(Int32) && type.GetGenericArguments().Skip(1).First() !=typeof(Int32)) 
                // This is a special case, due to Aerospike's storage of Int32 as Int64
            {
                ConvertValue = (v) =>
                {
                    var dict = v as Dictionary<object, object>;
                    var value = Activator.CreateInstance(type) as IDictionary;

                    foreach (var item in dict)
                    {
                        value.Add(Convert.ToInt32(item.Key), item.Value);
                    }

                    return value;
                };
            }
            else if (type.IsGenericType && type.GetGenericTypeDefinition() == typeof(Dictionary<,>) && 
                type.GetGenericArguments().First() == typeof(Int32) &&
                type.GetGenericArguments().Skip(1).First() == typeof(Int32)) // This is a special case, due to Aerospike's storage of Int32 as Int64
            {
                ConvertValue = (v) =>
                {
                    var dict = v as Dictionary<object, object>;
                    var value = Activator.CreateInstance(type) as IDictionary;

                    foreach (var item in dict)
                    {
                        value.Add(Convert.ToInt32(item.Key), Convert.ToInt32(item.Value));
                    }

                    return value;
                };
            }
            else if (type.IsGenericType && type.GetGenericTypeDefinition() == typeof (Dictionary<,>) &&
                     type.GetGenericArguments().First() != typeof (Int32) &&
                     type.GetGenericArguments().Skip(1).First() == typeof (Int32))
            {
                ConvertValue = (v) =>
                {
                    var dict = v as Dictionary<object, object>;
                    var value = Activator.CreateInstance(type) as IDictionary;

                    foreach (var item in dict)
                    {
                        value.Add(item.Key, Convert.ToInt32(item.Value));
                    }

                    return value;
                };
            }
            else if (type.IsGenericType && type.GetGenericTypeDefinition() == typeof(Dictionary<,>))
            {
                ConvertValue = (v) =>
                {
                    var dict = v as Dictionary<object, object>;
                    var value = Activator.CreateInstance(type) as IDictionary;

                    foreach (var item in dict)
                    {
                        value.Add(item.Key, item.Value);
                    }

                    return value;
                };
            }
            else if (type == typeof(HashSet<int>))
            {
                ConvertValue = (v) =>
                {
                    var list = new HashSet<int>(((List<object>)v).Select(Convert.ToInt32));
                    return list;
                };
            }
            else if (type == typeof(HashSet<string>))
            {
                ConvertValue = (v) =>
                {
                    var list = new HashSet<string>(((List<object>)v).Select(Convert.ToString));
                    return list;
                };
            }
            else if (type == typeof(HashSet<Guid>))
            {
                ConvertValue = (v) =>
                {
                    if (v is HashSet<Guid>)
                    {
                        return v;
                    }

                    var list = new HashSet<Guid>(((List<object>)v).Select(ConvertObjectToGuid));
                    return list;
                };
            }
            else if (type.IsClass)
            {
                ConvertValue = (v) =>
                {
                    if (v == DBNull.Value)
                    {
                        return null;
                    }
                    return v;
                };
            }
            else if (type == typeof(DateTime))
            {
                ConvertValue = (v) => ((DateTime) v).NormalizeToUtc();
            }
            else
            {
                ConvertValue = (v) => v;
            }
        }

        protected Guid ConvertObjectToGuid(object v)
        {
            var rawValue = Convert.ToString(v);
            return new Guid(rawValue);
        }

        public PropertyInfo PropertyInfo
        {
            get;
            private set;
        }

        public ColumnAttribute ColumnAttribute
        {
            get;
            private set;
        }

        public Func<object, object> ConvertValue;

        public Func<object, Bin> GetBin;

        public Func<AerospikeClient, Key, string, object> ConvertLargeSet;

    }
}
