﻿using System;
using System.Collections.Generic;
using System.Linq;
using Aerospike.Client;
using System.Collections;
using System.Runtime.InteropServices.WindowsRuntime;

namespace Curse.Aerospike
{
    public class LargeList<V> where V : class, ILargeListElement, new()
    {
        private readonly LargeList _map;

        public LargeList(LargeList map)
        {
            if (map == null)
            {
                throw new ArgumentNullException("map");
            }

            _map = map;            
        }

        public void Add(V element)
        {
            var hash = element.ToHash();
            var key = element.GetKey();
            hash["key"] = key;

            try
            {
                _map.Add(Value.Get(hash));
            }
            catch (AerospikeException ex)
            {
                if (ex.Result == 1402)
                {
                    throw new AerospikeUniqueKeyViolation(key, "The key value supplied '" + key + "' violated a unique key constraint.", ex);
                }

                throw;
            }
            
        }

        public void AddRange(IEnumerable<V> elements)
        {
            _map.Add(elements.Select(e =>
            {
                var hash = e.ToHash();
                hash["key"] = e.GetKey();
                return Value.Get(hash);
            }).ToList());
        }

        public void Update(V element)
        {
            var hash = element.ToHash();
            hash["key"] = element.GetKey();
            _map.Update(Value.Get(hash));
        }

        public void Remove(long key)
        {
            var dict = new Dictionary<object, object> { { "key", key } };
            _map.Remove(Value.Get(dict));
        }        

        public void Clear()
        {
            try
            {
                _map.Destroy();
            }
            catch (AerospikeException ex) // Catch LDT does not exist errors
            {
                if (ex.Result != 1417)
                {
                    throw;
                }
            }

        }

        public IEnumerable<V> Range(long startRange, long endRange)
        {
            IEnumerable<V> result = null;

            try
            {
                result = _map.Range(Value.Get(startRange), Value.Get(endRange))
                             .Cast<Dictionary<object, object>>()
                             .Select(HydrateFromHash)
                             .ToArray();
            }
            catch (AerospikeException ex) // Catch LDT does not exist errors
            {
                if (ex.Result != 1417)
                {
                    throw;
                }
            }

            return result;

        }

        public V[] FindAll(HashSet<int> values)
        {
            return values.Select(Find).Where(item => item != null).ToArray();
        }

        public V[] FindAll(HashSet<long> values)
        {
            return values.Select(Find).Where(item => item != null).ToArray();
        }

        public V Find(int value)
        {
            return Find((long) value);
        }

        public V Find(long value)
        {
         
            try
            {
                var dict = new Dictionary<object, object> { {"key", value} };

                var result = _map.Find(Value.Get(dict));

                if (result != null)
                {
                
                    return result.Cast<Dictionary<object, object>>()
                        .Select(HydrateFromHash)
                        .FirstOrDefault();
                }

                return null;
            }
            catch (AerospikeException ex) // Catch LDT does not exist errors
            {
                if (ex.Result != 1417)
                {
                    throw;
                }

                return null;
            }             
        }
       
        public bool Exists(long value)
        {
            IList result = null;

            try
            {
                var dict = new Dictionary<object, object> { { "key", value } };
                return _map.Find(Value.Get(dict)) != null;
            }
            catch (AerospikeException ex) // Catch LDT does not exist errors
            {
                if (ex.Result != 1417)
                {
                    throw;
                }                
            }

            return false;
        }

        public IEnumerable<V> GetValues()
        {

            try
            {                
                return _map.Scan().Cast<Dictionary<object, object>>().Select(HydrateFromHash).ToArray();
            }
            catch
            {

                return new V[0];
            }

        }

        public int Count
        {   
            get
            {   
                try
                {
                    return _map.Size();
                }
                catch (AerospikeException ex)
                {
                    if (ex.Result != 1417)
                    {
                        throw;
                    }
                }
                return 0;
            }
        }

        private V HydrateFromHash(Dictionary<object, object> hash)
        {
            var element = new V();
            element.Hydrate(hash);
            return element;
        }
    }
}
