﻿using Curse.AzureCaching;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.Storage.Table;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Reflection;
using System.Runtime.Serialization;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace Curse.AzureData
{
    [Serializable]
    public class BaseTableEntity : TableEntity
    {        
        public BaseTableEntity() { }
        public BaseTableEntity(string partitionKey, string rowKey)
        {
            PartitionKey = partitionKey;
            RowKey = rowKey;
        }        
        
    }
    
    [Serializable]
    public abstract class BaseTableEntity<T, V> : BaseTableEntity 
        where T : BaseTableEntity, new()
        where V : class
    {

        protected static bool IsCached = false;

        static BaseTableEntity()
        {
            var table = GetTable(AzureClientHelper.LocalStorageID);
            table.CreateIfNotExists();

            var entityDefinition = typeof(T).GetCustomAttribute<EntityDefinitionAttribute>();
            if(entityDefinition != null)
            {
                IsCached = entityDefinition.IsCached;
            }

        }

        protected static Regex InvalidKeyCharacters = new Regex(@"[\\\/\?\#]", RegexOptions.Compiled);

        protected abstract string GenerateRowKey();

        protected string CreateSanitizedRowKey()
        {
            var baseKey = GenerateRowKey();
            return SanitizeKey(baseKey);
        }

        private string SanitizeKey(string keyValue)
        {
            return InvalidKeyCharacters.Replace(keyValue, string.Empty);
        }

        public abstract V ToContract();
        
        public abstract void ParseContract(V value);
       
        public BaseTableEntity(string partitionKey, string rowKey) : base(partitionKey, rowKey) { }

        public BaseTableEntity(int partitionKey, int rowKey) : base(partitionKey.ToString(), rowKey.ToString()) { }

        public BaseTableEntity(int partitionKey, Guid rowKey) : base(partitionKey.ToString(), rowKey.ToString()) { }

        public static T Get(int storageAccountID, int partitionKey, int rowKey)
        {
            return Get(storageAccountID, partitionKey.ToString(), rowKey.ToString());
        }

        public static T Get(CloudTable table, int partitionKey, int rowKey)
        {
            return Get(table, partitionKey.ToString(), rowKey.ToString());
        }

        public static T Get(CloudTable table, string partitionKey, string rowKey)
        {
            if (string.IsNullOrEmpty(partitionKey))
            {
                throw new ArgumentNullException("partitionKey");
            }

            if (string.IsNullOrEmpty(rowKey))
            {
                throw new ArgumentNullException("rowKey");
            }

            var retrieveOperation = TableOperation.Retrieve<T>(partitionKey, rowKey);
            var query = table.Execute(retrieveOperation);
            T result = (T)query.Result;
            return result;
        }

        public static T Get(int storageAccountID, string partitionKey, string rowKey)
        {                       
           var table = GetTable(storageAccountID);            
           return Get(table, partitionKey, rowKey);
        }

        public static T GetCached(int storageAccountID, string partitionKey, string rowKey)
        {
            var cacheKey = GenerateCacheKey(partitionKey);
           
            return AzureCacheManager.GetOrAdd<T>(partitionKey, () => Get(storageAccountID, partitionKey, rowKey));
        }
        

        public static CloudTable GetTable(int storageAccountID)
        {
            var client = AzureClientHelper.GetClient(storageAccountID);
            var table = client.GetTableReference(typeof(T).Name);
            return table;
        }

        public void Delete(int storageAccountID)
        {
            TableOperation deleteOperation = TableOperation.Delete(this);
            GetTable(storageAccountID).Execute(deleteOperation);
            ExpireCache();
        }

        public void Insert(int storageAccountID)
        {
            TableOperation insertOperation = TableOperation.Insert(this);
            var result = GetTable(storageAccountID).Execute(insertOperation);
            this.ETag = result.Etag;
            ExpireCache();
        }

        public void InsertOrReplace(int storageAccountID)
        {
            TableOperation insertOperation = TableOperation.InsertOrReplace(this);
            GetTable(storageAccountID).Execute(insertOperation);
            ExpireCache();
        }

        public void Update(CloudTable table)
        {
            TableOperation updateOperation = TableOperation.Merge(this);
            table.Execute(updateOperation);

            ExpireCache();
        }
        
        protected void ExpireCache()
        {
            if (!IsCached)
            {
                return;
            }

            AzureCacheManager.DefaultCache.Remove(CacheKey);            
        }

        public void Update(int storageAccountID)
        {            
            var table = GetTable(storageAccountID);
            Update(table);
            ExpireCache();
        }

        public static T[] GetAll(int storageAccountID)
        {
            var query = new TableQuery<T>();
            return GetTable(storageAccountID).ExecuteQuery(query).ToArray();
        }

        public static T[] GetAllByPartitionKey(int storageAccountID, string partitionKey)
        {
            TableQuery<T> query = new TableQuery<T>().Where(TableQuery.GenerateFilterCondition("PartitionKey", QueryComparisons.Equal, partitionKey));
            return GetTable(storageAccountID).ExecuteQuery(query).ToArray();
        }

        public static void BulkInsert(int storageAccountID, IEnumerable<T> values)
        {

            var table = GetTable(storageAccountID);
            var batch = new TableBatchOperation();
            foreach (var value in values)
            {
                batch.Insert(value);
            }
            table.ExecuteBatch(batch);
        }

        public static void BulkMerge(int storageAccountID, IEnumerable<T> values)
        {
            var table = GetTable(storageAccountID);
            var batch = new TableBatchOperation();
            foreach (var value in values)
            {
                batch.Merge(value);
            }
            table.ExecuteBatch(batch);
        }

        public static void BulkInsertOrMerge(int storageAccountID, IEnumerable<T> values)
        {
            var table = GetTable(storageAccountID);
            var batch = new TableBatchOperation();
            foreach (var value in values)
            {
                batch.InsertOrMerge(value);
            }
            table.ExecuteBatch(batch);
        }

        public static int BulkDeleteByPartionKey(int storageAccountID, string partitionKey)
        {
            var table = GetTable(storageAccountID);
            var values = GetAllByPartitionKey(storageAccountID, partitionKey);
            var count = values.Count();

            var partitions = values.Batch(100);
            foreach (var part in partitions)
            {
                var batch = new TableBatchOperation();

                foreach (var value in part)
                {
                    batch.Delete(value);
                }

                if (batch.Any())
                {
                    table.ExecuteBatch(batch);
                }
            }

            return count;

        }       

        public static string GenerateCacheKey(string partitionKey)
        {
            return typeof(T).FullName + ":" + partitionKey;
        }

        public string CacheKey
        {
            get
            {
                return GenerateCacheKey(PartitionKey);
            }
        }

    }
}
