﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.WindowsAzure.Storage.Queue;
using Newtonsoft.Json;
using System.Diagnostics;

namespace Curse.AzureData
{

    public enum QueueManagerMode
    {
        Local,
        Mirror,
        Replicate
    }

    /// <summary>
    /// Helper class for managing a local queue. Supports replication and mirroring.
    /// </summary>
    /// <typeparam name="K"></typeparam> 
    public abstract class QueueManager<K>
    {
        protected int[] _storageAccountIDs;
        protected string _queueName;       

        protected CloudQueue GetLocalQueue(int storageID)
        {
            var client = AzureClientHelper.GetQueueClient(AzureClientHelper.LocalStorageID);
            string queueName = _queueName.ToLowerInvariant();
            if(storageID != AzureClientHelper.LocalStorageID)
            {
                queueName += "-" + storageID.ToString();
            }
            var queue = client.GetQueueReference(queueName);
            return queue;
        }

        protected CloudQueue GetRemoteQueue(int storageID)
        {
            var client = AzureClientHelper.GetQueueClient(storageID);
            var queue = client.GetQueueReference(_queueName.ToLowerInvariant());
            return queue;
        }

        protected QueueManager(QueueManagerMode mode)
        {
            _queueName = typeof(K).Name;        
                
            // Based on the storage mode, determine which queues to place data in
            if (mode == QueueManagerMode.Mirror || mode == QueueManagerMode.Replicate)
            {
                _storageAccountIDs = AzureClientHelper.AllStorageIDs;
            }
            else
            {
                _storageAccountIDs = new[] { AzureClientHelper.LocalStorageID };
            }

            // Ensure the local queues exist
            foreach (var id in _storageAccountIDs)
            {
                GetLocalQueue(id).CreateIfNotExists();
            }
        }

        protected string Serialize(K value)
        {
            var serializedValue = JsonConvert.SerializeObject(value);

            if (serializedValue == null || !AzureUtility.IsAllowedQueueMessageSize(serializedValue))
            {

                throw new Exception("Serialized message size exceeds limit.");
            }

            return serializedValue;
        }

        protected K Deserialize(string value)
        {
            return JsonConvert.DeserializeObject<K>(value);
        }

        public void EnqueueToAll(K value)
        {
            var serializedValue = Serialize(value);
            var message = new CloudQueueMessage(serializedValue);

            foreach (var id in _storageAccountIDs)
            {
                var queue = GetLocalQueue(id);
                queue.AddMessage(message);
            }
        }

        public void Enqueue(K value, int storageAccountID)
        {
            var serializedValue = Serialize(value);
            var message = new CloudQueueMessage(serializedValue);
            var queue = GetLocalQueue(storageAccountID);
            queue.AddMessage(message);
        }

        public void EnqueueToLocal(K value)
        {
            var serializedValue = Serialize(value);
            var message = new CloudQueueMessage(serializedValue);
            var queue = GetLocalQueue(AzureClientHelper.LocalStorageID);
            queue.AddMessage(message);
        }

        public K Dequeue(int storageAccountID)
        {
            var queue = GetLocalQueue(storageAccountID);
            var message = queue.GetMessage();
            return Deserialize(message.AsString);
        }

        protected virtual void Initialize(EntityStorageMode mode)
        {
                  
        }
    }

    public class DistributedQueueManager<K> : QueueManager<K>        
    {
        protected int[] _remoteStorageAccountIDs;

        protected DistributedQueueManager() : base(QueueManagerMode.Replicate)
        {
            // Ensure the remote queue exists
            foreach (var id in _storageAccountIDs)
            {
                GetRemoteQueue(id).CreateIfNotExists();
            }
            
            _remoteStorageAccountIDs = _storageAccountIDs.Where(p => p != AzureClientHelper.LocalStorageID).ToArray();
        }

        public void ReplicateQueue()
        {
            // For each storage account, copy from the local version of the queue to the remote version
            foreach (var id in _remoteStorageAccountIDs)
            {
                var localQueue = GetLocalQueue(id);
                var remoteQueue = GetRemoteQueue(id);

                while (true)
                {
                    CloudQueueMessage localMessage = null;
                    try
                    {
                        localMessage = localQueue.GetMessage();
                    }
                    catch (Exception ex)
                    {
                        Trace.TraceError("Failed to get message from queue for storage account " + id + ":" + ex.Message);
                        break;
                    }

                    if (localMessage == null)
                    {
                        break;
                    }
                                       
                    // Try to persist this to table storage of the remote account
                    try
                    {
                        var remoteMessage = new CloudQueueMessage(localMessage.AsBytes);

                        // Add the message to the remote queue
                        remoteQueue.AddMessage(remoteMessage);

                        // Remove it from the local queue
                        localQueue.DeleteMessage(localMessage);
                    }
                    catch (Exception ex)
                    {
                        Trace.TraceError("Failed to insert or replace model for storage account " + id + ": " + ex.Message);
#if !DEBUG
                        if (localMessage.DequeueCount >= 5)
                        {
                            localQueue.DeleteMessage(localMessage);                            
                        }
#endif
                    }

                }
            }

        }
    }
    
    public class QueueManager<K, V> : QueueManager<K>
        where K : BaseTableEntity<K, V>, new()
        where V : class
    {
              
        
        protected QueueManager(QueueManagerMode mode) : base(mode)
        {
            // Ensure remote tables exist
            if (mode == QueueManagerMode.Mirror)
            {
                foreach (var id in _storageAccountIDs)
                {
                    try
                    {
                        BaseTableEntity<K, V>.GetTable(id).CreateIfNotExists();
                    }
                    catch (Exception ex)
                    {

                    }
                }
            }                        
        }              

        public void ProcessQueue()
        {
            // For each storage account, try to dequeue a message, then persist it to the appropriate remote storage account
            foreach (var id in _storageAccountIDs)
            {
                var queue = GetLocalQueue(id);
                
                while (true)
                {
                    CloudQueueMessage message = null;
                    try
                    {
                        message = queue.GetMessage();
                    }
                    catch (Exception ex)
                    {
                        Trace.TraceError("Failed to get message from queue for storage account " + id + ":" + ex.Message);
                        break;
                    }
                    
                    if(message == null)
                    {
                        break;
                    }


                    K model = null;

                    try
                    {
                        model = Deserialize(message.AsString);
                    }
                    catch(Exception ex)
                    {
                        Trace.TraceError("Failed to deserialize message from queue for storage account " + id + ":" + ex.Message);
                        continue;
                    }
                    
                    // Try to persist this to table storage of the remote account
                    try
                    {
                        
                        model.InsertOrReplace(id);
                        queue.DeleteMessage(message);
                    }
                    catch (Exception ex)
                    {
                        Trace.TraceError("Failed to insert or replace model for storage account " + id + ": " + ex.Message);
#if !DEBUG
                        if (message.DequeueCount >= 5)
                        {
                            queue.DeleteMessage(message);                            
                        }
#endif
                    }
                    
                }
            }

        }


    }
}
