﻿using Curse.CloudServices.Authentication;
using Curse.CloudServices.AuthenticationModels;
using Curse.Cassandra;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Requests;
using Curse.Friends.Responses;
using Microsoft.WindowsAzure.ServiceRuntime;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Runtime.Serialization;
using System.ServiceModel;
using System.ServiceModel.Web;
using System.Text;
using System.Threading.Tasks;

namespace Curse.Friends
{
    
    public class FriendsService : IFriendsService
    {

        static FriendsService()
        {
            StorageConfiguration.Initialize();            
        }

        public AddFriendHintResponse AddFriendHint(FriendHintRequest request)
        {
            var response = new AddFriendHintResponse();

            try
            {
                var authToken = AzureAuthenticationContext.Current;

                if(request.FriendHint.Type != FriendHintType.Game)
                {
                    response.Status = AddFriendHintResponseStatus.Unauthorized;
                    return response;
                }

                if (!request.Validate())
                {
                    response.Status = AddFriendHintResponseStatus.Invalid;
                    return response;
                }


                //var entity = new FriendHintEntity(authToken.UserID);        
                ////entity.Insert(FriendHintEntity.Lo)
                //entity.ParseContract(request.FriendHint);                                                
                //// Add this to all queues
                //FriendHintQueue.Instance.EnqueueToAll(entity);
                response.Status = AddFriendHintResponseStatus.Successful;

            }
            catch (Exception ex)
            {
                response.Status = AddFriendHintResponseStatus.Error;                                
            }           

            return response;
        }

        public GetSelfHintsResponse GetSelfHints()
        {
            var response = new GetSelfHintsResponse();

            try
            {
                var authToken = AzureAuthenticationContext.Current;

                //var hints = FriendHintEntity.GetAllByPartitionKey(AzureClientHelper.LocalStorageID, authToken.UserID.ToString());                
                //response.Hints = hints.Select(h => h.ToContract()).ToArray();

            }
            catch (Exception ex)
            {
                response.Status = GetSelfHintsResponseStatus.Error;
            }

            return response;
        }

        public GetMyFriendsResponse GetMyFriends()
        {
            var response = new GetMyFriendsResponse();

            try
            {
                var authToken = AzureAuthenticationContext.Current;

                // See if the user has registered before
                var userRegion = UserRegion.Find(authToken.UserID);

                if (userRegion == null)
                {
                    throw new Exception("User has not registered!");
                }
                
                response.Friends = Friendship.GetAll(userRegion.RegionID, authToken.UserID);
                response.Status = GetMyFriendsStatus.Successful;                

            }
            catch (Exception ex)
            {
                response.Status = GetMyFriendsStatus.Successful;
            }

            return response;
        }

        public ConfirmFriendshipResponse ConfirmFriendship(ConfirmFriendshipRequest request)
        {
            try
            {
                var authToken = AzureAuthenticationContext.Current;
                var friendshipContext = new FriendshipContext(authToken.UserID, request.FriendID);
             
                // Try to get the friendship requested                
                var myFriendship = friendshipContext.GetMyFriendship();
                var theirFriendship = friendshipContext.GetTheirFriendship();

                if (myFriendship == null || theirFriendship == null)
                {
                    return new ConfirmFriendshipResponse(ConfirmFriendshipStatus.NotFound);
                }

                // Ensure that the far end friendship is in the 'AwaitingConfirmation' state.
                if (theirFriendship.Status != FriendshipStatus.AwaitingThem)
                {
                    return new ConfirmFriendshipResponse(ConfirmFriendshipStatus.NotFound);
                }

                // Finalize the friendship
                Friendship.Confirm(friendshipContext.Me, friendshipContext.MyRegion, myFriendship, friendshipContext.Them, friendshipContext.TheirRegion, theirFriendship);

                // Queue off a notification for each user
                FriendshipResolver.Enqueue(new FriendshipResolver { UserID = friendshipContext.Me.UserID, FriendID = friendshipContext.Them.UserID,  Status = FriendshipStatus.Confirmed });
                FriendshipResolver.Enqueue(new FriendshipResolver { UserID = theirFriendship.UserID, FriendID = theirFriendship.OtherUserID, Status = theirFriendship.Status });

                return new ConfirmFriendshipResponse(ConfirmFriendshipStatus.Successful);
            }
            catch (Exception ex)
            {
                return new ConfirmFriendshipResponse(ConfirmFriendshipStatus.Error);
            }
        }

        private UserRegion GetCurrentUserRegion(AzureAuthenticationToken authToken)
        {            
            // See if the user has registered before
            var myRegion = UserRegion.Find(authToken.UserID);

            if (myRegion == null)
            {
                throw new Exception("User has not registered!");
            }

            return myRegion;
        }

        public RequestFriendshipResponse RequestFriendship(FriendshipRequest request)
        {            
            try
            {
                
                if(!request.Validate())
                {
                    return new RequestFriendshipResponse { Status = RequestFriendshipStatus.FailedValidation };
                }

                var authToken = AzureAuthenticationContext.Current;
                var friendshipContext = new FriendshipContext(authToken.UserID, request.FriendID);

                var myFriendship = friendshipContext.GetMyFriendship();
                var theirFriendship = friendshipContext.GetTheirFriendship();
                
                                
                // Already requested
                if (myFriendship != null && theirFriendship != null)
                {
                    if (myFriendship.Status == FriendshipStatus.AwaitingThem)
                    {
                        return new RequestFriendshipResponse(RequestFriendshipStatus.AlreadyRequested);
                    }
                    else if(myFriendship.Status != FriendshipStatus.AwaitingMe)
                    {
                        return new RequestFriendshipResponse(RequestFriendshipStatus.AlreadyRequested);
                    }

                    Friendship.Confirm(friendshipContext.Me, friendshipContext.MyRegion, myFriendship, friendshipContext.Them, friendshipContext.TheirRegion, theirFriendship);

                    return new RequestFriendshipResponse(RequestFriendshipStatus.Successful);                    
                }

                // Ensure this user doesn't have a ton of friendship records already
                var myFriendCount = Friendship.GetCountByUserID(friendshipContext.MyRegion.RegionID, friendshipContext.Me.UserID);

                if (myFriendCount > Friendship.MaxFriendCount)
                {
                    return new RequestFriendshipResponse { Status = RequestFriendshipStatus.MaximumFriends };
                }

                // Create the request on both sides
                Friendship.CreateRequest(friendshipContext.Me, friendshipContext.MyRegion, friendshipContext.Them, friendshipContext.TheirRegion, request.KnownIdentity);

                // Queue off a notification for each user
                FriendshipResolver.Enqueue(new FriendshipResolver { UserID = friendshipContext.Me.UserID, FriendID = friendshipContext.Them.UserID, FriendName = request.KnownIdentity, Status = FriendshipStatus.AwaitingThem});
                FriendshipResolver.Enqueue(new FriendshipResolver { UserID = friendshipContext.Them.UserID, FriendID = friendshipContext.Me.UserID, FriendName = friendshipContext.Me.Username, Status = FriendshipStatus.AwaitingMe });
                
                return new RequestFriendshipResponse(RequestFriendshipStatus.Successful);
            }
            catch (Exception ex)
            {
                return new RequestFriendshipResponse(RequestFriendshipStatus.Error);
            }            
        }

        public RegisterSelfResponse RegisterSelf(RegisterSelfRequest request)
        {            
            var authToken = AzureAuthenticationContext.Current;
            
            if(string.IsNullOrEmpty(authToken.Username))
            {
                return new RegisterSelfResponse { Status = RegisterSelfStatus.Error };
            }

            // See if the user has registered before
            var userRegion = UserRegion.Find(authToken.UserID);
            
            // If not, register them to the local storage node
            if (userRegion == null)
            {
                userRegion = new UserRegion { UserID = authToken.UserID, RegionID = UserRegion.LocalConfigID };
                userRegion.Insert(UserRegion.LocalSession);
            }
            
            // From their storage region, get their record and update their status            
            var user = User.Get(userRegion.RegionID, authToken.UserID);

            if (user == null) // If it doesn't exist, insert or replace
            {
                user = new User
                {
                    UserID = authToken.UserID,
                    Username = authToken.Username,
                    ConnectionStatus = request.Status
                };

                user.Insert(userRegion.RegionID);
            }
            else // Otherwise, update the status
            {
                user.ConnectionStatus = request.Status;
                user.Update(userRegion.RegionID);
            }
                       
            var sessionID = Guid.NewGuid().ToString();

            // Register the client endpoint

            var entity = ClientEndpoint.GetLocal(authToken.UserID, request.MachineKey.ToString());
            if (entity == null)
            {
                entity = new ClientEndpoint
                {
                    UserID = authToken.UserID,
                    MachineKey = request.MachineKey.ToString(),
                    SessionDate = DateTime.UtcNow,   
                    IsConnected = false,
                    RegionID = ClientEndpoint.LocalConfigID,
                    SessionID = sessionID
                };

                entity.InsertLocal();
            }
            else
            {
                ClientEndpoint.UpdateSession(sessionID, authToken.UserID, request.MachineKey.ToString());
            }

            // Now, queue off a job to let their friends know that they are online
            UserStatusResolver.Enqueue(new UserStatusResolver { UserID = user.UserID  });

            return new RegisterSelfResponse { Status = RegisterSelfStatus.Successful, DisplayName = user.Username, SessionID = sessionID };
        }

        public void DebugAddFriend(int otherUserID)
        {
            var authToken = AzureAuthenticationContext.Current;
            
            // See if the user has registered before
            var userRegion = UserRegion.Find(authToken.UserID);

            if(userRegion== null)
            {
                throw new Exception("No region!");
            }

            var otherUserRegion = UserRegion.Find(otherUserID);

            if (otherUserRegion == null)
            {
                throw new Exception("No region!");
            }

            var sessionManager = User.LocalSession;            
            var myUser = User.Get(sessionManager, authToken.UserID);
            var otherUser = User.Get(sessionManager, otherUserID);

            var session = Friendship.LocalSession;

            // Create a friendship record in both 'sides'
            Friendship mySide = new Friendship
            {
                UserID = myUser.UserID,
                OtherUserID = otherUser.UserID,
                OtherUsername = otherUser.Username,
                OtherUserNickname = null,
                OtherUserConnectionStatus = otherUser.ConnectionStatus,
                OtherUserRegionID = otherUserRegion.RegionID

            };
            mySide.Insert(session);

            var otherUserSession = Friendship.GetSession(otherUserRegion.RegionID);
            // Create a friendship record in both 'sides'
            Friendship theirSide = new Friendship
            {
                UserID = otherUser.UserID,
                OtherUserID = myUser.UserID,
                OtherUsername = myUser.Username,
                OtherUserNickname = null,
                OtherUserConnectionStatus = myUser.ConnectionStatus,
                OtherUserRegionID = userRegion.RegionID

            };

            theirSide.Insert(otherUserSession);
            
        }


        public IEnumerable<string> DebugCreateEntities(int seedID, int numberToCreate)
        {

            var results = new List<string>();

            Stopwatch sw = Stopwatch.StartNew();
            
            var upperBound = seedID + numberToCreate;            
            var ids = new List<int>();

            for (int i = seedID; i < upperBound; i++)
            {
                ids.Add(i);
            }

            var sessionManager = User.LocalSession;

            var userIDToTest = seedID;

            Parallel.ForEach(ids, new ParallelOptions { MaxDegreeOfParallelism = 8 }, id =>
            {

                var user = new User
                {
                    UserID = id,
                    Username = "User-" + id,
                    ConnectionStatus = UserConnectionStatus.Offline,
                    CustomStatusMessage = "None"
                };

                user.Insert(sessionManager);

            });
            
           
            sw.Stop();            

            results.Add("Created " + numberToCreate + " entities in " + sw.ElapsedMilliseconds.ToString("###,##0.00") + "ms");

            // Try to read this from the other data center
            var session = User.GetSession(1);
            var queryPlan = session.Session.Cluster.Configuration.Policies.LoadBalancingPolicy.NewQueryPlan(session.GetSelectStatement(userIDToTest));
            var allUSers = User.Get(session, userIDToTest);


            sw.Reset();
            sw.Start();
            
            for (int i = seedID; i < upperBound; i++)
            {
                var user = User.Get(sessionManager, i);
            }
            
            sw.Stop();

            results.Add("Retrieved " + numberToCreate + " entities in " + sw.ElapsedMilliseconds.ToString("###,##0.00") + "ms");

            sw.Reset();
            sw.Start();

            
            for (int i = seedID; i < upperBound; i++)
            {
                var user = User.Get(sessionManager, i);
                user.ConnectionStatus = UserConnectionStatus.Online;
                user.Update(sessionManager);
            }
            
            sw.Stop();

            results.Add("Updated " + numberToCreate + " entities in " + sw.ElapsedMilliseconds.ToString("###,##0.00") + "ms");

            return results;
        }
              
    
    }
}
