﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.SocketServer;
using Curse.SocketInterface;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Data;
using Curse.CloudQueue;
using Curse.SocketMessages;
using Curse.Logging;
using Curse.Friends.Enums;
using Curse.Friends.Data.Queues;

namespace Curse.NotificationWorker
{
    public class NotificationServer : SocketServer.SocketServer
    {
        public static readonly NotificationServer Instance = new NotificationServer(50000);
        private static readonly NotificationInstance NotificationInstance = new NotificationInstance();

        public NotificationServer(int maxConnections)
            : base(maxConnections, 1, false)
        {
            AddContractDispatcher<JoinRequest>(OnJoinRequest);
            AddContractDispatcher<VoiceInvitationRequest>(OnVoiceInvitationRequest);

        }

        protected override void OnHandshake(BaseSocketInterface baseSocketInterface, Handshake handshake)
        {
            baseSocketInterface.IsHandshaken = true;
            baseSocketInterface.DateLastHandshake = DateTime.UtcNow;
            baseSocketInterface.SendContract(handshake);
        }

        private void OnJoinRequest(BaseSocketInterface socketInterface, JoinRequest joinRequest)
        {
            var resp = new JoinResponse()
            {
                Status = JoinStatus.FailedUnhandledException
            };

            try
            {
                var session = NotificationInstance.CreateSession(socketInterface, joinRequest.UserID, joinRequest.MachineKey, joinRequest.SessionID);
                if (session == null)
                {

                    resp.Status = JoinStatus.InvalidSessionID;
                    return;
                }
                socketInterface.Session = session;
                socketInterface.IsAuthenticated = true;
                resp.Status = JoinStatus.Successful;
            }
            catch (Exception ex)
            {
                //TODO: Log!
            }
            finally
            {
                socketInterface.SendContract(resp);
            }

        }

        private void OnVoiceInvitationRequest(BaseSocketInterface socketInterface, VoiceInvitationRequest request)
        {
            var session = (NotificationSession)socketInterface.Session;

            if (session == null)
            {
                Logger.Warn("Attempt to send without a session");
                return;
            }

            var resp = new VoiceInvitationResponse()
            {
                Status = VoiceInvitationStatus.FailedUnhandledException
            };


            try
            {
                // Ensure that the intended recipient is a confirmed friend
                var userRegion = UserRegion.Find(session.UserID);

                if (userRegion == null)
                {
                    Logger.Warn("Unable to find region for user.");
                    resp.Status = VoiceInvitationStatus.UnknownUser;
                    return;
                }

                var friendship = Friendship.Get(userRegion.RegionID, session.UserID, request.FriendID);
                if (friendship == null || friendship.Status != FriendshipStatus.Confirmed)
                {
                    resp.Status = VoiceInvitationStatus.InvalidFriend;
                    return;
                }

                if(friendship.OtherUserConnectionStatus != UserConnectionStatus.Online)
                {
                    resp.Status = VoiceInvitationStatus.FriendOffline;
                    return;
                }

                VoiceInvitationResolver.Enqueue(new VoiceInvitationResolver { FriendID = request.FriendID, InviteUrl = request.InviteUrl, UserID = session.UserID });
                resp.Status = VoiceInvitationStatus.Successful;
                
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in OnVoiceInvitationRequest");
            }
            finally
            {
                socketInterface.SendContract(resp);
            }
                        
        }

        public override void Start(System.Net.IPEndPoint[] localEndPoints)
        {
            base.Start(localEndPoints);

            UserStatusNotifier.QueueProcessor.ProcessMessage += QueueProcessor_ProcessMessage;
            UserStatusNotifier.QueueProcessor.Start();

            FriendshipNotifier.QueueProcessor.ProcessMessage += FriendshipNotifier_ProcessMessage;
            FriendshipNotifier.QueueProcessor.Start();

            VoiceInvitationResolver.QueueProcessor.Start();
            VoiceInvitationNotifier.QueueProcessor.ProcessMessage += VoiceInvitationNotifier_ProcessMessage;
            VoiceInvitationNotifier.QueueProcessor.Start();
        }

        void VoiceInvitationNotifier_ProcessMessage(object sender, VoiceInvitationNotifier e)
        {
            var session = NotificationInstance.GetSession(e.SessionID);

            if (session == null)
            {
                return;
            }

            session.NotifyVoiceInvitation(new VoiceInvitationNotification
            {
                SenderID = e.SenderID, 
                InviteUrl = e.InviteUrl
            });
        }

        void FriendshipNotifier_ProcessMessage(object sender, FriendshipNotifier e)
        {
            var session = NotificationInstance.GetSession(e.SessionID);

            if (session == null)
            {
                return;
            }

            session.NotifyFriendship(new FriendshipNotification
            {
                FriendID = e.FriendID,
                FriendName = e.FriendName,
                Status = e.Status
            });
        }

        void QueueProcessor_ProcessMessage(object sender, UserStatusNotifier e)
        {
            var session = NotificationInstance.GetSession(e.SessionID);

            if (session == null)
            {
                return;
            }

            session.NotifyUserStatus(new UserStatusNotification
            {
                UserID = e.UserID,
                Status = e.Status,
            });
        }




    }
}
