﻿using System.Collections.Concurrent;
using System.Security.Policy;
using System.Text.RegularExpressions;
using Curse.CloudServices;
using Curse.Friends.Enums;
using Curse.Friends.Client;
using Curse.Friends.NotificationContracts;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Text;
using System.Threading.Tasks;
using Curse.Friends.Client.FriendsService;
using System.Security;
using Curse.Backend.BehaviorTest.ClientService;
using Curse.SocketInterface;
using Curse.SocketMessages;

namespace Curse.Backend.BehaviorTest
{
    class Program
    {
        private const string ApiKey = "***REMOVED***";
        private static Guid MachineKey = new Guid("79e8d2b3-f7af-49c8-aa3b-476a5eac46d2");
        private const string ClientServiceUrl = "http://clientservice-v6-beta.curse.com/CClientService.svc/Binary";
        private static Guid CurrentGroupID = Guid.Empty;

        private static readonly ConcurrentDictionary<Guid, GroupCache> _groupCache = new ConcurrentDictionary<Guid, GroupCache>();

        private static string ServiceUrl = "http://friends.cursevoice.dev/FriendsService.svc";

        private const int TestUserStartID = 20000;
        private const int TestUserAmount = 1000;

        static int UserID;
        static string Username;
        static string Token;

        static FriendsNotificationClient _client;
        private static CClientServiceClient _clientServiceClient;

        static bool IsConnected
        {
            get
            {
                return _client != null && _client.IsConnected;
            }
        }

        static SecureString GetPassword()
        {
            SecureString pwd = new SecureString();
            while (true)
            {
                ConsoleKeyInfo i = Console.ReadKey(true);
                if (i.Key == ConsoleKey.Enter)
                {
                    break;
                }
                else if (i.Key == ConsoleKey.Backspace)
                {
                    if (pwd.Length > 0)
                    {
                        pwd.RemoveAt(pwd.Length - 1);
                        Console.Write("\b \b");
                    }
                }
                else
                {
                    pwd.AppendChar(i.KeyChar);
                    Console.Write("*");
                }
            }
            return pwd;
        }

        static HashSet<int> PromptUserList(int limit = 0)
        {
            var recipientIDs = new HashSet<int>();

            while (true)
            {
                Console.WriteLine("Enter a friend ID to invite (or blank to continue): ");
                var input = Console.ReadLine();
                if (string.IsNullOrEmpty(input))
                {
                    break;
                }
                recipientIDs.Add(int.Parse(input));

                if (recipientIDs.Count == limit)
                {
                    break;
                }
            }

            return recipientIDs;
        }

        static void Divider()
        {
            Console.WriteLine();
            Console.WriteLine("------------------------------------------");
            Console.WriteLine();
        }

        static void Main(string[] args)
        {
            if (args.Length != 4)
            {
                Console.WriteLine("Usage: <Test.exe> {login} {password} {GUI} <LocalhostAddr>");
                return;
            }
            try
            {

            }
            catch (Exception)
            {
                
                throw;
            }
            Console.Title = "Friends Tester";

            FriendsNotificationClient.InstantMessageNotification += _notificationClient_InstantMessageNotification;
            FriendsNotificationClient.InstantMessageResponse += _notificationClient_InstantMessageResponse;
            FriendsNotificationClient.FriendshipChangeNotification += _notificationClient_FriendshipChangeNotification;
            FriendsNotificationClient.FriendshipRemovedNotification += _notificationClient_FriendshipRemovedNotification;
            FriendsNotificationClient.FriendshipRemovedNotification += _notificationClient_FriendshipRemovedNotification;
            FriendsNotificationClient.UserChangeNotification += Instance_UserChangeNotification;
            FriendsNotificationClient.VoiceInvitationNotification += FriendsNotificationClient_VoiceInvitationNotification;
            FriendsNotificationClient.VoiceDeclineNotification += FriendsNotificationClient_VoiceDeclineNotification;
            FriendsNotificationClient.GroupMessageNotification += FriendsNotificationClient_GroupMessageNotification;
            FriendsNotificationClient.GroupChangeNotification += FriendsNotificationClientOnGroupChangeNotification;
            FriendsNotificationClient.FavoriteGroupNotification += FriendsNotificationClientOnFavoriteGroupNotification;
            FriendsNotificationClient.OfflineMessageNotification += FriendsNotificationClientOnOfflineMessageNotification;

            var options = new Dictionary<string, Action>();
            options.Add("login", Login);

            // Group commands
            options.Add("new-group", CreateGroup);
            options.Add("leave-group", LeaveGroup);
            options.Add("delete-group", DeleteGroup);
            options.Add("remove-group-member", RemoveGroupMember);
            options.Add("add-group-member", AddGroupMembers);
            options.Add("change-member-role", ChangeGroupMemberRole);
            options.Add("new-group-invite", CreateGroupInvite);
            options.Add("claim-group-invite", ProcessGroupInvite);
            options.Add("delete-group-invite", DeleteGroupInvite);

            options.Add("message-group", GroupMessage);
            options.Add("call-group", CallGroup);
            options.Add("rename-group", RenameGroup);
            options.Add("mute-group", MuteGroup);
            options.Add("unmute-group", UnmuteGroup);
            options.Add("favorite-group", ToggleGroupFavorite);
            options.Add("enter-group", EnterGroup);
            options.Add("offlinemessage-group", SyncOfflineMessages);

            // Self commands
            options.Add("change-status", ChangeStatus);
            options.Add("change-game-status", ChangeGameStatus);
            options.Add("instant-message", InstantMessage);
            options.Add("send-friend-request", SendFriendRequest);
            options.Add("rename-friend", RenameFriend);
            options.Add("confirm-friend-request", ConfirmFriendRequest);
            options.Add("decline-friend-request", DeclineFriendRequest);
            options.Add("favorite-friend", FavoriteFriend);
            options.Add("remove-friend", RemoveFriend);
            options.Add("unblock-friend", UnblockFriend);
            options.Add("get-profile", GetProfile);
            options.Add("search", Search);

            options.Add("call-friend", SendVoiceInvite);
            options.Add("decline-call", DeclineVoiceInvite);

            while (true)
            {

                if (UserID != 0)
                {
                    Console.WriteLine("You are logged in as " + Username + "#" + UserID);

                    GroupCache currentGroup;

                    if (_groupCache.TryGetValue(CurrentGroupID, out currentGroup))
                    {
                        Console.WriteLine("You are currently in " + currentGroup.RootGroup.GroupTitle);

                        if (currentGroup != null && !currentGroup.HasLoadedDetails)
                        {
                            GetGroupDetails(currentGroup.RootGroup.GroupID);
                            currentGroup.HasLoadedDetails = true;
                        }
                    }

                    var resp = GetMyFriends();

                    if (resp.Status != GetMyFriendsStatus.Successful)
                    {
                        Console.WriteLine("Unable to retrieve your friends list!");
                        Console.ReadKey();

                        return;
                    }

                    if (_groupCache.Count == 0)
                    {
                        var groupResp = GetMyGroups();

                        if (groupResp.Status != BasicServiceResponseStatus.Successful)
                        {
                            Console.WriteLine("Unable to retrieve your group list!");
                            Console.ReadKey();

                            return;
                        }

                        foreach (var group in groupResp.Groups.Where(p => p.GroupID == p.RootGroupID).OrderByDescending(p => p.DateMessaged))
                        {
                            if (!_groupCache.ContainsKey(group.RootGroupID))
                            {
                                _groupCache.TryAdd(group.GroupID, new GroupCache(group));
                            }
                        }

                        if (CurrentGroupID == Guid.Empty)
                        {
                            // Determine the best group
                            var bestGroup = groupResp.Groups.FirstOrDefault(p => p.GroupID == p.RootGroupID);
                            if (bestGroup != null)
                            {
                                CurrentGroupID = bestGroup.GroupID;
                            }
                        }
                    }

                    Console.WriteLine();
                    Console.WriteLine("GROUPS");


                    foreach (var groupCache in _groupCache.Values)
                    {
                        Console.WriteLine();
                        Console.WriteLine(groupCache.RootGroup.GroupTitle + " (" + groupCache.RootGroup.GroupAccessLevel + "+)");

                        Console.WriteLine("Members");
                        foreach (var member in groupCache.Members.Values)
                        {
                            Console.WriteLine(" -" + member.Username + " (" + member.Role + ")");
                        }

                        if (groupCache.Children != null && groupCache.Children.Any())
                        {
                            Console.WriteLine("Sub Groups");

                            foreach (var child in groupCache.Children.Values.Where(p => p.ParentGroupID == groupCache.RootGroup.GroupID).OrderBy(p => p.DisplayOrder))
                            {
                                Console.WriteLine(" -" + child.GroupTitle + " (" + child.GroupAccessLevel + "+)");
                                foreach (var subChild in groupCache.Children.Values.Where(p => p.ParentGroupID == child.GroupID).OrderBy(p => p.DisplayOrder))
                                {
                                    Console.WriteLine("     -" + subChild.GroupTitle + " (" + subChild.GroupAccessLevel + "+)");
                                }
                            }
                        }


                        Console.WriteLine("-------------------------------------------");
                    }

                    Console.WriteLine();
                    if (resp.Friends.Length == 0)
                    {
                        Console.WriteLine("You have no friends :(");
                    }
                    else
                    {
                        Console.WriteLine("FRIENDS");
                    }

                    foreach (var group in resp.Friends.GroupBy(p => p.Status))
                    {
                        Console.WriteLine();
                        Console.WriteLine(group.Key.ToString());

                        var ordered = group.OrderByDescending(p => p.IsFavorite);

                        foreach (var friend in ordered.Take(10))
                        {
                            Console.Write(" ");
                            if (friend.IsFavorite)
                            {
                                Console.Write("* ");
                            }
                            Console.WriteLine((friend.OtherUserNickname ?? friend.OtherUsername) + "#"
                                + friend.OtherUserID
                                + " (" + friend.OtherUserConnectionStatus
                                + (!string.IsNullOrEmpty(friend.OtherUserStatusMessage) ? " - " + friend.OtherUserStatusMessage : "") + ")");
                        }
                        if (group.Count() > 10)
                        {
                            Console.WriteLine("Showing first 10 friends...");
                        }
                    }

                }

                Divider();
                foreach (var option in options.Keys)
                {
                    Console.WriteLine(" " + option);
                }
                Divider();
                Console.WriteLine("Enter an option from the list above...");

                var entered = Console.ReadLine().ToLower().Trim();
                Console.Clear();

                try
                {
                    if (entered == string.Empty)
                    {
                        Console.Clear();
                        continue;
                    }
                    Action action = null;
                    if (options.TryGetValue(entered, out action))
                    {
                        action();
                        Console.WriteLine("Press any key to continue...");
                    }
                    else
                    {
                        Console.WriteLine("Unknown option: " + entered);
                    }
                    Console.ReadKey(true);
                }
                catch (Exception ex)
                {
                    Console.WriteLine("Invalid entry!");
                }




                Console.Clear();

            }
        }

        static void CreateGroup()
        {
            var client = GetClient();
            Console.Write("Enter a title: ");
            var title = Console.ReadLine();


            Console.Write("Choose a type (Large, normal, temp): ");
            var typeString = Console.ReadLine();
            var type = GroupType.Normal;
            Enum.TryParse(typeString, true, out type);


            Guid parentGroupID = Guid.Empty;
            var accessLevel = GroupRole.Member;

            if (type == GroupType.Large || type == GroupType.Temporary)
            {
                Console.Write("Enter an access level (Member, Officer, Owner leave blank for default): ");
                var accessLevelString = Console.ReadLine();
                if (!string.IsNullOrEmpty(accessLevelString))
                {
                    Enum.TryParse(accessLevelString, true, out accessLevel);
                }

                Console.WriteLine("Is this a sub group? (Y|N)");
                var isSubGroup = Console.ReadKey(true).Key == ConsoleKey.Y;

                if (isSubGroup)
                {
                    parentGroupID = PromptGroup(false);
                }
            }

            var recipientIDs = PromptUserList();

            var resp = client.CreateGroup(new CreateGroupRequest { Title = title, RecipientsUserIDs = recipientIDs.ToArray(), AccessLevel = accessLevel, ParentGroupID = parentGroupID, Type = type });

            Console.WriteLine("Group creation status: " + resp.Status);

            if (resp.Status == CreateGroupStatus.Successful)
            {
                Console.WriteLine("Group ID: " + resp.GroupID);
            }
        }

        private static void DeleteGroup()
        {
            var client = GetClient();
            var groupID = PromptGroup(false);
            var resp = client.DeleteGroup(new DeleteGroupRequest { GroupID = groupID });
            if (resp.Status != BasicServiceResponseStatus.Successful)
            {
                Console.WriteLine("Failed to delete group: " + resp.Status);
            }
        }

        private static Guid PromptGroup(bool rootOnly = true)
        {
            var counter = 0;
            var choices = new Dictionary<int, Guid>();
            foreach (var group in _groupCache.Values)
            {
                choices[++counter] = group.RootGroup.GroupID;
                Console.WriteLine(counter + ". " + group.RootGroup.GroupTitle);
                if (!rootOnly)
                {
                    foreach (var child in group.Children)
                    {
                        choices[++counter] = child.Value.GroupID;
                        Console.WriteLine(counter + ". " + child.Value.GroupTitle);
                    }
                }
            }

            Console.Write("Enter the number of the group: ");

            try
            {
                var choice = int.Parse(Console.ReadLine());
                return choices[choice];
            }
            catch (Exception)
            {

                Console.WriteLine("Invalid selection! Press any key to continue...");
                Console.ReadKey(true);
                return Guid.Empty;
            }

        }

        static void LeaveGroup()
        {
            var client = GetClient();
            var resp = client.LeaveGroup(new LeaveGroupRequest { GroupID = CurrentGroupID });
            Console.WriteLine("Group leave status: " + resp.Status);
        }

        static void RemoveGroupMember()
        {
            var recipientIDs = PromptUserList();

            var client = GetClient();
            var resp = client.RemoveUsersFromGroup(new RemoveUsersFromGroupRequest { GroupID = CurrentGroupID, UserIDs = recipientIDs.ToArray() });
            Console.WriteLine("Remove users status: " + resp.Status);
        }

        static void GroupMessage()
        {

            Console.Write("Message: ");
            var message = Console.ReadLine();
            _client.SendGroupMessage(CurrentGroupID, message, Guid.NewGuid());
        }

        static void RenameGroup()
        {
            Console.Write("New Title: ");
            var title = Console.ReadLine();
            var client = GetClient();
            var resp = client.RenameGroup(new ChangeGroupRequest() { GroupID = CurrentGroupID, Title = title });
            Console.WriteLine("Change Group's status: " + resp.Status);
        }

        static void ToggleGroupFavorite()
        {
            Console.Write("Enter True/False:");
            var favorite = bool.Parse(Console.ReadLine());
            var client = GetClient();
            var resp = client.ToggleFavoriteGroup(new FavoriteGroupRequest() { GroupID = CurrentGroupID, IsFavorite = favorite });
            Console.WriteLine("Change Group's status: " + resp.Status);
        }

        static void MuteGroup()
        {

            var client = GetClient();
            var resp =
                client.ChangeGroupNotificationPreferences(new ChangeGroupNotificationPreferencesRequest
                {
                    GroupID = CurrentGroupID,
                    Preference = NotificationPreference.Disabled
                });

            Console.WriteLine("Mute group status: " + resp.Status);
        }

        static void UnmuteGroup()
        {

            var client = GetClient();
            var resp =
                client.ChangeGroupNotificationPreferences(new ChangeGroupNotificationPreferencesRequest
                {
                    GroupID = CurrentGroupID,
                    Preference = NotificationPreference.Enabled
                });

            Console.WriteLine("Mute group status: " + resp.Status);
        }

        static void AddGroupMembers()
        {
            var recipientIDs = PromptUserList();

            var client = GetClient();
            var resp = client.AddUsersToGroup(new AddUserToGroupRequest { GroupID = CurrentGroupID, UserIDs = recipientIDs.ToArray() });
            Console.WriteLine("Add users status: " + resp.Status);
        }

        static void CreateGroupInvite()
        {
            Console.Write("How long would you like this invite to be valid (Enter a number of hours or 0 for no expiration)? ");
            var hours = int.Parse(Console.ReadLine());

            bool autoRemoveGuests = false;
            if (hours > 0)
            {
                Console.Write("When this invite expires, should we automatically remove any guests that used it (Y|N)? ");
                autoRemoveGuests = Console.ReadLine().ToLower() == "y";
            }

            TimeSpan? lifespan = null;

            if (hours > 0)
            {
                lifespan = TimeSpan.FromHours(hours);
            }

            var client = GetClient();
            var resp = client.CreateGroupInvite(new CreateGroupInviteRequest { GroupID = CurrentGroupID, AutoRemoveMembers = autoRemoveGuests, Lifespan = lifespan });

            Console.WriteLine("Crate group invite: " + resp.Status);

            if (resp.Status == BasicServiceResponseStatus.Successful)
            {
                Console.WriteLine("Invite code: " + resp.InviteCode);
            }
        }

        static void ProcessGroupInvite()
        {
            Console.Write("Enter an invite code: ");
            var code = Console.ReadLine();

            var client = GetClient();
            var resp = client.ProcessGroupInvite(code);
            Console.WriteLine("Process code result: " + resp.Status);
        }

        static void DeleteGroupInvite()
        {
            Console.Write("Enter an invite code: ");
            var code = Console.ReadLine();

            Console.Write("Would you like to also remove any guests that used this code (Y|N)? ");
            var removeGuests = Console.ReadLine().ToLower() == "y";

            var client = GetClient();
            var resp = client.DeleteGroupInvite(new DeleteGroupInviteRequest { InviteCode = code, RemoveGuests = removeGuests });

            Console.WriteLine("Process code result: " + resp.Status);
        }

        static void ChangeGroupMemberRole()
        {
            var recipientIDs = PromptUserList(1);

            var role = PromptRole("Enter a new group role for this user.");

            var client = GetClient();
            var resp = client.ChangeGroupUserRole(new ChangeGroupUserRoleRequest
                {
                    GroupID = CurrentGroupID,
                    UserID = recipientIDs.First(),
                    Role = role
                });
            Console.WriteLine("Change member role: " + resp.Status);
        }

        static GroupRole PromptRole(string promptMessage)
        {
            Console.WriteLine(promptMessage);

            var roles = Enum.GetValues(typeof(GroupRole));

            foreach (var r in roles)
            {
                Console.WriteLine("-" + (GroupRole)r);
            }

            GroupRole role;
            while (true)
            {
                var accessLevelString = Console.ReadLine();
                if (!string.IsNullOrEmpty(accessLevelString))
                {
                    if (Enum.TryParse(accessLevelString, true, out role))
                    {
                        break;
                    }
                }
            }

            return role;
        }

        static void CallGroup()
        {
            var client = GetClient();

            Console.WriteLine("Creating or joining a group call...");

            var resp = client.GroupVoiceSession(new GroupVoiceSessionRequest
            {
                ClientVersion = "6.1.0.0",
                GroupID = CurrentGroupID
            });

            if (resp.Status == GroupVoiceSessionStatus.Successful)
            {
                Console.WriteLine("Group voice call succeeded! " + resp.InviteUrl);
            }
            else
            {
                Console.WriteLine("Group voice call failed :(");
            }
            Console.ReadKey();
        }

        static void EnterGroup()
        {
            var groupID = PromptGroup();

            // Try to get this from the group cache
            GroupCache found;
            if (!_groupCache.TryGetValue(groupID, out found))
            {
                Console.WriteLine("Unknown group!");
                return;
            }

            GetGroupDetails(found.RootGroup.GroupID);
            CurrentGroupID = groupID;
        }

        static void ChangeStatus()
        {

            var client = GetClient();

            var enumOptions = Enum.GetValues(typeof(UserConnectionStatus));
            foreach (var option in enumOptions)
            {
                Console.WriteLine((int)option + ". " + option.ToString());
            }


            var newStatus = (UserConnectionStatus)int.Parse(Console.ReadLine());

            Console.Write("Enter an optional status message: ");
            var statusMessage = Console.ReadLine();

            var resp = client.ChangeStatus(new ChangeStatusRequest { Status = newStatus, CustomStatusMessage = statusMessage, MachineKey = MachineKey });
        }


        static void ChangeGameStatus()
        {

            var client = GetClient();

            Console.Write("Enter a game ID: ");
            var gameID = int.Parse(Console.ReadLine());

            Console.Write("Enter a game status message: ");
            string gameStatus = Console.ReadLine();

            var resp = client.ChangeGameStatus(new ChangeGameStatusRequest { GameID = gameID, GameStatusMessage = gameStatus, GameState = 1, IsRunning = gameID > 0 });

        }

        static void UpdateProfile()
        {

            var client = GetClient();

            Console.Write("Enter an avatar URL: ");
            string avatarUrl = Console.ReadLine();

            var resp = client.ChangeProfile(new ChangeProfileRequest { AvatarUrl = avatarUrl });

        }

        static void SendFriendRequest()
        {

            Console.Write("Enter a friend ID: ");
            var friendID = int.Parse(Console.ReadLine());
            Console.Write("Enter the name of your friend: ");
            string knownIdentity = Console.ReadLine();

            var result = MakeFriendRequest(friendID, knownIdentity);
            Console.WriteLine(result.Status);
        }

        static void Disconnect()
        {
            CurrentGroupID = Guid.Empty;
            _client.Disconnect(SocketDisconnectReason.UserInitiated);
            _groupCache.Clear();
            CurrentGroupID = Guid.Empty;
        }

        static void Login()
        {

            // Need to disconnect here!
            if (_client != null && _client.IsConnected)
            {
                Disconnect();
            }

            Console.Write("Enter your username: ");
            string username = Console.ReadLine();
            Console.Write("Enter your password: ");
            var password = GetPassword();

            var credentials = new NetworkCredential(username, password);

            Console.WriteLine();
            Console.Write("Select a Device Platform (0=Windows, 1=iOS, 2=Android): ");
            string platform = Console.ReadLine();

            Device device;
            int platformOrdinal;
            if (!int.TryParse(platform, out platformOrdinal) || platformOrdinal < 1 || platformOrdinal > 2)
            {
                platformOrdinal = 0;
            }

            if (platformOrdinal == 1)
            {
                device = Device.iOS;
            }
            else if (platformOrdinal == 2)
            {
                device = Device.Android;
            }
            else
            {
                device = Device.Windows;
            }

            Console.Clear();
            Console.Write("Signing into central service...");
            var result = TestSignOn(credentials, device.Platform, device.DeviceID);
            Console.WriteLine(result.Status);
            if (result.Status != RegisterSelfStatus.Successful)
            {
                Console.WriteLine("Login failed!");
                Console.ReadKey(true);
                return;
            }

            var hostList = result.NotificationHostList;

            if (ServiceUrl.Contains(".dev"))
            {
                Console.WriteLine("Using local notification server.");
                hostList = new[] { "127.0.0.1" };
            }

            Console.WriteLine(result.Status);


            var connectionInfo = new NotificationConnectionInfo
            {
                Ports = result.NotificationHostPorts,
                UserID = UserID,
                MachineKey = MachineKey.ToString(),
                NotificationHosts = hostList,
                SessionID = result.SessionID,
                ClientVersion = "6.0.5000.0",

            };
            JoinResponse resp;
            connectionInfo.DesiredStatus = UserConnectionStatus.Online;

            Console.Write("Connecting to notification server....");
            _client = FriendsNotificationClient.Connect(connectionInfo, out resp);
            Console.WriteLine(resp.Status);

            if (resp.Status != JoinStatus.Successful)
            {
                UserID = 0;
                CurrentGroupID = Guid.Empty;
                Console.WriteLine("Connection failed: " + resp.Status);
                Console.WriteLine("Press any key to continue...");
                Console.ReadKey();
                return;
            }

            _client.Disconnected += _client_Disconnected;
        }

        static void UnregisterEndpoint()
        {
            // Need to disconnect here!
            if (_client != null && _client.IsConnected)
            {
                Disconnect();
            }

            var session = _clientServiceClient.v2GetCurrentSession();
            var client = ServiceClientHelper.CreateFriendsClient(session.UserID, session.Token, ServiceUrl);
            client.UnregisterEndpoint(new UnregisterEndpointRequest { MachineKey = MachineKey });
            CurrentGroupID = Guid.Empty;
            _groupCache.Clear();
        }

        static void ConfirmFriendRequest()
        {
            Console.Write("Enter a friend ID: ");
            var userID = int.Parse(Console.ReadLine());
            var result = ConfirmFriendRequest(userID);
            Console.WriteLine(result.Status);
        }

        static void SendVoiceInvite()
        {
            if (!IsConnected)
            {
                Console.WriteLine("You must be connected to send a voice invitation.");
                return;
            }

            Console.Write("Enter a friend ID: ");
            var userID = int.Parse(Console.ReadLine());
            _client.SendVoiceInvitation(userID, "http://www.cursevoice.com/join/1234");
        }

        static void DeclineVoiceInvite()
        {
            if (_currentVoiceInviteUserID > 0)
            {
                _client.SendVoiceDecline(_currentVoiceInviteUserID, _currentVoiceInviteUrl);
                _currentVoiceInviteUserID = 0;
                _currentVoiceInviteUrl = null;
            }
        }

        static void FavoriteFriend()
        {
            Console.Write("Enter a friend ID to favorite: ");
            var userID = int.Parse(Console.ReadLine());
            var resp = ToggleFavorite(userID, true);
            if (resp.Status != ToggleFavoriteStatus.Successful)
            {
                Console.WriteLine(resp.Status);
            }
        }

        static void RenameFriend()
        {
            Console.Write("Enter a friend ID to rename: ");
            var userID = int.Parse(Console.ReadLine());
            Console.Write("Enter the new name: ");
            var nickname = Console.ReadLine();
            var client = GetClient();
            var resp = client.RenameFriend(new RenameFriendRequest { FriendID = userID, Nickname = nickname });
            if (resp.Status != RenameFriendStatus.Successful)
            {
                Console.WriteLine(resp.Status);
            }
        }

        static void DeclineFriendRequest()
        {
            Console.Write("Enter a friend ID to decline: ");
            var userID = int.Parse(Console.ReadLine());

            Console.Write("Do you want to block this user from sending you requests in the future [Y|N]: ");
            var blockFuture = Console.ReadKey().Key == ConsoleKey.Y;

            var client = GetClient();
            var resp = client.DeclineFriendship(new DeclineFriendshipRequest { BlockFutureRequests = blockFuture, FriendID = userID });
            if (resp.Status != DeclineFriendshipStatus.Successful)
            {
                Console.WriteLine(resp.Status);
            }
        }

        static void RemoveFriend()
        {
            Console.Write("Enter a friend ID to remove: ");
            var userID = int.Parse(Console.ReadLine());
            var client = GetClient();
            var resp = client.RemoveFriendship(new RemoveFriendshipRequest { FriendID = userID });
        }


        static void UnblockFriend()
        {
            Console.Write("Enter a friend ID to unblock: ");
            var userID = int.Parse(Console.ReadLine());
            var client = GetClient();
            var resp = client.UnblockFriendship(new UnblockFriendshipRequest { FriendID = userID });
        }

        static void Search()
        {
            while (true)
            {
                Console.Write("Username, character or complete email address (X to exit): ");

                var query = Console.ReadLine();
                if (query.ToLower().Equals("x"))
                {
                    return;
                }

                Console.Clear();

                var client = GetClient();
                var resp = client.FindFriends(new FriendSearchRequest { QueryString = query });

                if (resp.Status != FriendSearchStatus.Successful)
                {
                    Console.WriteLine("Failed to search: " + resp.Status);
                    Console.WriteLine("Press any key to continue...");
                    Console.ReadKey(true);
                }

                Console.WriteLine("Search completed in " + resp.ElapsedMilliseconds + " ms");
                Console.WriteLine();

                if (resp.EmailMatches.Any())
                {
                    Console.WriteLine("Email Matches");
                    Console.WriteLine();
                    foreach (var m in resp.EmailMatches)
                    {
                        Console.WriteLine("- " + m.Value.EmailAddress);
                    }
                }


                if (resp.UserMatches.Any())
                {
                    Console.WriteLine("User Matches");
                    Console.WriteLine();

                    foreach (var m in resp.UserMatches)
                    {
                        Console.WriteLine("- " + m.Value.Username + " (" + m.Value.FriendCount + " friends)");
                    }
                }

                if (resp.CharacterMatches.Any())
                {
                    Console.WriteLine("Characters");
                    Console.WriteLine();
                    foreach (var m in resp.CharacterMatches)
                    {
                        Console.WriteLine("- " + m.Value.CharacterName + "@" + m.Value.ServerName + " - " + m.Value.ServerRegion);
                    }


                }

                Console.WriteLine();
            }
        }


        static void InstantMessage()
        {
            Console.Write("Enter a friend ID: ");
            var userID = int.Parse(Console.ReadLine());
            Console.Write("Message: ");
            var message = Console.ReadLine();

            _client.SendInstantMessage(userID, message, Guid.NewGuid());
        }

        static void SyncOfflineMessages()
        {
            Console.WriteLine("How many days do you want to sync?");
            int days;
            if (int.TryParse(Console.ReadLine(), out days))
            {
                Console.Clear();
                Console.WriteLine("Sending offline message request. Press any key to continue...");
                Console.WriteLine();
                _client.SendOfflineMessageRequest(Guid.NewGuid(), DateTime.UtcNow.AddDays(-days));
                Console.ReadKey();
            }
            else
            {
                Console.WriteLine("Invalid number of days, Press any key to continue.");
                Console.ReadKey();
            }

        }

        static void GetProfile()
        {
            Console.Write("Enter a user ID: ");
            var userID = int.Parse(Console.ReadLine());

            var client = GetClient();
            var resp = client.GetUserProfile(userID);

            if (resp.Status == BasicServiceResponseStatus.Successful)
            {
                Console.WriteLine("Found user profile:");
                Console.Write(resp.Username);
                Console.WriteLine(resp.Name);

                foreach (var f in resp.MutualFriends)
                {
                    Console.WriteLine("Mutual Friend: " + f.UserID + " (" + f.MutualUserIDs.Length + ")");
                }

                Console.ReadLine();
            }
        }

        private static void GetGroupDetails(Guid groupID)
        {
            var client = GetClient();
            var resp = client.GetGroupDetails(new GroupDetailsRequest() { GroupID = groupID, ShowDeletedChannels = true });

            if (resp.Status == BasicServiceResponseStatus.Successful)
            {
                GroupCache cache;
                if (_groupCache.TryGetValue(groupID, out cache))
                {
                    if (resp.Groups != null)
                    {
                        //for normal groups resp.Groups will be null
                        foreach (var group in resp.Groups)
                        {
                            cache.AddOrUpdateGroup(group);
                        }
                    }
                    cache.AddMembers(resp.Members);
                }
            }
            else
            {
                Console.WriteLine("Failed to enter group: " + resp.Status);
                Console.ReadLine();
            }
        }

        private static void FriendsNotificationClientOnOfflineMessageNotification(object o, EventArgs<OfflineMessageNotification> e)
        {
            foreach (var groupConvo in e.Value.GroupMessages)
            {
                Console.WriteLine("Received offline messages for group " + groupConvo.GroupID + ". Total Messages: " + groupConvo.Count);
                foreach (var message in groupConvo.Messages)
                {
                    ProcessGroupMessage(message);
                }
                Console.WriteLine("-----------------------------------------------------------------------------------");
                Console.WriteLine();
            }

            foreach (var friendConvo in e.Value.InstantMessages)
            {
                Console.WriteLine("Received offline messages from friend " + friendConvo.FriendID + ". Total Messages: " + friendConvo.Count);
                foreach (var message in friendConvo.Messages)
                {
                    ProcessInstantMessage(message);
                }
                Console.WriteLine("-----------------------------------------------------------------------------------");
                Console.WriteLine();

            }
        }

        private static void FriendsNotificationClientOnGroupChangeNotification(object o, EventArgs<GroupChangeNotification> eventArgs)
        {
            var notification = eventArgs.Value;
            var notificationGroup = notification.Group;

            GroupCache cache;

            if (!_groupCache.TryGetValue(notificationGroup.RootGroupID, out cache))
            {
                // Add it!
                if (notificationGroup.GroupID == notificationGroup.RootGroupID)
                {
                    cache = new GroupCache(notificationGroup);
                    cache.AddMembers(notification.Members);
                    _groupCache.TryAdd(notificationGroup.RootGroupID, cache);
                }
                else
                {
                    Console.WriteLine("Received a notification for an unknown parented group!");
                    Console.ReadKey(true);
                    return;
                }
            }

            if (cache == null)
            {
                return;
            }

            var sender = cache.GetMember(notification.SenderID);
            var senderName = sender != null ? sender.Username : "User#" + notification.SenderID;

            switch (notification.ChangeType)
            {
                case GroupChangeType.CreateGroup:
                    Console.WriteLine(senderName + " created group " + notificationGroup.GroupTitle);
                    cache.AddOrUpdateGroup(notificationGroup);
                    break;

                case GroupChangeType.RemoveGroup:
                    Console.WriteLine(senderName + " removed group " + notificationGroup.GroupTitle);
                    cache.RemoveChildGroup(notificationGroup.GroupID);
                    break;

                case GroupChangeType.AddUsers:

                    foreach (var newUser in notification.Members)
                    {
                        cache.AddMember(newUser);
                        Console.WriteLine(newUser.Username + " has joined " + cache.RootGroup.GroupTitle);
                    }

                    break;

                case GroupChangeType.RemoveUsers:

                    if (notification.Members.Any(p => p.UserID == UserID))
                    {
                        _groupCache.TryRemove(cache.RootGroup.GroupID, out cache);
                        Console.WriteLine("You have been removed from " + cache.RootGroup.GroupTitle);
                    }
                    else
                    {
                        foreach (var removedUser in notification.Members)
                        {
                            var removed = cache.RemoverMember(removedUser.UserID);

                            if (removed != null)
                            {
                                Console.WriteLine(removed.Username + " has left " + cache.RootGroup.GroupTitle);
                            }
                        }
                    }

                    break;

                case GroupChangeType.UpdateUsers:

                    foreach (var u in notification.Members)
                    {
                        var updatedMember = cache.UpdateMember(u);
                        Console.WriteLine(senderName + " has updated " + updatedMember.Username + " (" + updatedMember.Role + ") in " + cache.RootGroup.GroupTitle);
                    }

                    break;

                case GroupChangeType.ChangeInfo:
                    {
                        var updatedGroup = cache.AddOrUpdateGroup(notificationGroup);
                        Console.WriteLine("{0} changed group title to {1} and avatar url to {2}", senderName, updatedGroup.GroupTitle, updatedGroup.GroupAvatar);
                        break;
                    }

                case GroupChangeType.GroupReorganized:
                    {
                        var updatedGroup = cache.AddOrUpdateGroup(notificationGroup);
                        Console.WriteLine("Group '" + updatedGroup.GroupTitle + "' has been reorganized!");
                        foreach (var group in notification.ChildGroups)
                        {
                            Console.WriteLine("Child group '" + group.GroupTitle + "' has been reorganized!");
                        }

                        break;
                    }
            }

        }

        private static void FriendsNotificationClientOnFavoriteGroupNotification(object sender, EventArgs<FavoriteGroupNotification> eventArgs)
        {
            var favorite = eventArgs.Value;
            GroupCache cache;
            if (_groupCache.TryGetValue(favorite.GroupID, out cache))
            {
                if (favorite.IsFavorite)
                {
                    Console.WriteLine("Group {0} added to favorites", cache.RootGroup.GroupTitle);
                }
                else
                {
                    Console.WriteLine("Group {0} removed from favorites", cache.RootGroup.GroupTitle);
                }
            }
        }

        static void FriendsNotificationClient_GroupMessageNotification(object o, SocketMessages.EventArgs<GroupMessageNotification> e)
        {
            ProcessGroupMessage(e.Value);
        }

        static void ProcessGroupMessage(GroupMessageNotification messageNotification)
        {
            // Try to get the group from the group membership dictionary
            GroupCache cache;
            if (_groupCache.TryGetValue(messageNotification.GroupID, out cache))
            {
                GroupMemberNotification sender;
                var senderName = "User " + messageNotification.SenderID;
                if (cache.Members.TryGetValue(messageNotification.SenderID, out sender))
                {
                    senderName = sender.Username;
                }

                Console.WriteLine(senderName + " says to " + cache.RootGroup.GroupTitle + ": " + messageNotification.Message);

                // Notify that the group message has been read
                _client.SendConversationRead(messageNotification.GroupID, DateTime.UtcNow);
            }
        }

        static void FriendsNotificationClient_VoiceDeclineNotification(object sender, SocketMessages.EventArgs<VoiceDeclineNotification> e)
        {
            Console.WriteLine("User " + e.Value.SenderID + " has declined your voice invitation");
        }

        static void FriendsNotificationClient_VoiceInvitationNotification(object sender, SocketMessages.EventArgs<VoiceInvitationNotification> e)
        {
            _currentVoiceInviteUserID = e.Value.SenderID;
            _currentVoiceInviteUrl = e.Value.InviteUrl;
            Console.WriteLine("Received a voice invite from user " + _currentVoiceInviteUserID + ": " + _currentVoiceInviteUrl);
        }

        private static int _currentVoiceInviteUserID = 0;
        private static string _currentVoiceInviteUrl = null;

        static FriendHint[] GetTestHints()
        {
            var hints = new List<FriendHint>();

            hints.Add(new FriendHint
            {
                UserID = 817369,
                SearchTerm = "Crs Acenth",
                GameID = 341,
                Region = "NA",
                Type = FriendHintType.Game,
                Verification = FriendHintVerification.ClientObserved,
                AvatarUrl = "http://avatar.leagueoflegends.com/NA/Crs Acenth.png"
            });

            hints.Add(new FriendHint
                {
                    UserID = 817369,
                    SearchTerm = "Ace#1476",
                    Type = FriendHintType.Platform,
                    Platform = FriendPlatform.BattleNet,
                    Verification = FriendHintVerification.ClientObserved,
                    AvatarUrl = null
                });


            return hints.ToArray();
        }

        static void _client_Disconnected(object sender, SocketInterface.SocketDisconnectEventArgs e)
        {
            Console.WriteLine("Client disconnected!");
        }

        static void Instance_UserChangeNotification(object sender, SocketMessages.EventArgs<UserChangeNotification> e)
        {
            Console.WriteLine("Your user info changed! Avatar is: " + e.Value.User.AvatarUrl);

        }

        static void Instance_ConnectionFailed(object sender, EventArgs e)
        {
            Console.WriteLine("Connection failed!");
        }

        static void _notificationClient_FriendshipRemovedNotification(object sender, SocketMessages.EventArgs<FriendshipRemovedNotification> e)
        {

            Console.WriteLine("Your friendship with " + e.Value.FriendID + " has beeen dissolved!");
        }

        static void _notificationClient_FriendshipChangeNotification(object sender, SocketMessages.EventArgs<FriendshipChangeNotification> e)
        {
            var friendship = e.Value.Friendship;
            Console.WriteLine("Your friendship with " + (friendship.OtherUsername ?? friendship.OtherUserNickname) + " has updated.");
            Console.WriteLine("Status: " + e.Value.Friendship.Status
                + Environment.NewLine
                + "Avatar: " + friendship.OtherUserAvatarUrl
                + Environment.NewLine
                + "Game: " + friendship.OtherUserGameID + ", " + friendship.OtherUserGameState);
        }

        static void _notificationClient_InstantMessageResponse(object sender, SocketMessages.EventArgs<InstantMessageResponse> e)
        {
            if (e.Value.Status != DeliveryStatus.Successful)
            {
                Console.WriteLine("Unable to send message: " + e.Value.Status);
            }
        }

        static void _notificationClient_InstantMessageNotification(object sender, SocketMessages.EventArgs<InstantMessageNotification> e)
        {
            ProcessInstantMessage(e.Value);
        }

        static void ProcessInstantMessage(InstantMessageNotification instantMessage)
        {
            Console.WriteLine(instantMessage.Timestamp.ToLocalTime().ToShortTimeString() + " - " + instantMessage.SenderID + ": " + instantMessage.Message);

            // Notify that the IM has been read
            _client.SendConversationRead(
                UserID == instantMessage.SenderID ? instantMessage.RecipientID : instantMessage.SenderID,
                DateTime.UtcNow);
        }

        static void TestEncryption()
        {

            // Login to the client service            
            var clientServiceClient = TestServiceClientHelper.CreateClientServiceClient("zusername", "password", ClientServiceUrl);
            var session = clientServiceClient.v2GetCurrentSession();

            // Login to the friends service, using the token
            var friendsServiceClient = ServiceClientHelper.CreateFriendsClient(session.UserID, session.Token, ServiceUrl);
            friendsServiceClient.RegisterSelf(new RegisterSelfRequest { MachineKey = MachineKey, Status = UserConnectionStatus.Online });

        }

        static ToggleFavoriteResponse ToggleFavorite(int userID, bool isFavorite)
        {
            var client = GetClient();
            return client.ToggleFavorite(new ToggleFavoriteRequest { FriendID = userID, IsFavorite = isFavorite });
        }

        private static FriendsServiceClient GetClient()
        {
            return ServiceClientHelper.CreateFriendsClient(UserID, Token, ServiceUrl, ApiKey);
        }

        private static void TestInserts()
        {
            var client = ServiceClientHelper.CreateFriendsClient(1, "Adamar", ServiceUrl);
            //var result = client.DebugCreateEntities(TestUserStartID, 1000);
            //var result = client.DebugCreateEntitiesAzure(6000, 1000);
            //var result = client.DebugCreateEntitiesSqlAzure(8000, 1000);
        }

        private static RegisterSelfResponse TestSignOn(NetworkCredential credentials, DevicePlatform platform = 0, string deviceID = null)
        {
            _clientServiceClient = TestServiceClientHelper.CreateClientServiceClient(credentials.UserName, credentials.Password, ClientServiceUrl);
            var session = _clientServiceClient.v2GetCurrentSession();
            var client = ServiceClientHelper.CreateFriendsClient(session.UserID, session.Token, ServiceUrl);
            var resp = client.RegisterSelf(
                new RegisterSelfRequest
                {
                    MachineKey = MachineKey,
                    Status = UserConnectionStatus.Online,
                    Platform = platform,
                    DeviceID = deviceID
                });

            if (resp.Status == RegisterSelfStatus.Successful)
            {
                UserID = session.UserID;
                Username = session.Username;
                Token = session.Token;
            }

            return resp;
        }

        private static RequestFriendshipResponse MakeFriendRequest(int friendID, string knownIdentity)
        {
            var client = GetClient();
            return client.RequestFriendship(new FriendshipRequest() { FriendID = friendID, KnownIdentity = knownIdentity });
        }


        private static ConfirmFriendshipResponse ConfirmFriendRequest(int friendID)
        {
            var client = GetClient();
            return client.ConfirmFriendship(new ConfirmFriendshipRequest() { FriendID = friendID });
        }

        private static GetMyFriendsResponse GetMyFriends()
        {
            var client = GetClient();
            return client.GetMyFriends();
        }

        private static GetMyGroupsResponse GetMyGroups()
        {
            var client = GetClient();
            return client.GetMyGroups();
        }

        private static void TestAddingFriends()
        {
            var client = ServiceClientHelper.CreateFriendsClient(1, "Adamar", ServiceUrl);
            var endRange = TestUserStartID + TestUserAmount;
            for (int i = TestUserStartID; i < endRange; i++)
            {
                //client.DebugAddFriend(i);
            }
        }

        private static void CreateFakeUsers()
        {
            var endRange = TestUserStartID + TestUserAmount;
            for (int i = TestUserStartID; i < endRange; i++)
            {
                var client = ServiceClientHelper.CreateFriendsClient(i, "RandomUser-" + i.ToString(), ServiceUrl);
                client.RegisterSelf(new RegisterSelfRequest { MachineKey = MachineKey, Status = UserConnectionStatus.Online });
            }
        }

    }
}
