﻿using Cassandra;
using Cassandra.Data.Linq;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Curse.Cassandra
{
    public abstract class BaseSessionManager<TEntity, TManager>
        where TEntity : BaseTable<TEntity, TManager>, new()
        where TManager : BaseSessionManager<TEntity, TManager>, new()
    {
        protected string KeySpace;
        protected string TableName;
        protected static bool IsRegional;

        public static TManager Create(Cluster cluster, string keySpace, string tableName, bool isRegional)
        {                        
            var sessionManager = new TManager();
            IsRegional = isRegional;

            sessionManager.Session = cluster.Connect(keySpace);

            if (CreateTable(sessionManager.Session, keySpace, tableName))
            {
                CreateSecondaryIndexes(sessionManager.Session, keySpace, tableName);
            }            
#if DEBUG           
            Thread.Sleep(100); // TODO: Figure out why this is necessary!
            cluster.RefreshSchema();
#endif

            sessionManager.TableName = tableName;
            Stopwatch sw = Stopwatch.StartNew();
            sessionManager.PrepareStatements();
            sessionManager.SelectAllStatement = sessionManager.GetPreparedStatement(string.Format("SELECT * FROM \"{0}\"", tableName));

            Curse.Logging.Logger.Info("[Cassandra] Finished creating a session manager for " + keySpace + "." + tableName + " in " + sw.ElapsedMilliseconds.ToString("###,##0.00") + "ms");

            return sessionManager;
        }

        private static bool CreateTable(ISession session, string keySpace, string tableName)
        {
            var sql = string.Format("CREATE TABLE IF NOT EXISTS \"{0}\".\"{1}\"", keySpace, tableName);


            var properties = typeof(TEntity).GetProperties(BindingFlags.Instance | BindingFlags.Public | BindingFlags.SetProperty);

            List<string> columns = new List<string>();
            List<string> keys = new List<string>();
            foreach (var property in properties)
            {
                var columnAttribute = property.GetCustomAttribute<ColumnAttribute>();

                if (columnAttribute == null)
                {
                    continue;
                }

                var colSql = string.Format("\"{0}\" {1}", columnAttribute.Name, CassandraUtilities.GetCsqlType(property.PropertyType));

                if (property.GetCustomAttribute<PartitionKeyAttribute>() != null)
                {
                    keys.Add(property.Name);
                }

                columns.Add(colSql);
            }
            columns.Add("PRIMARY KEY(\"" + string.Join("\", \"", keys.ToArray()) + "\")");
            sql += "(" + string.Join(",", columns.ToArray()) + ")";
                        
            try
            {
                session.Execute(sql);
                return true;
            }
            catch (Exception ex)
            {
                Curse.Logging.Logger.Error(ex, "Failed to create table: " + tableName);
                return false;
            }             
        }

        private static void CreateSecondaryIndexes(ISession session, string keySpace, string tableName)
        {
            var properties = typeof(TEntity).GetProperties(BindingFlags.Instance | BindingFlags.Public | BindingFlags.SetProperty);

            foreach (var property in properties)
            {
                var columnAttribute = property.GetCustomAttribute<ColumnAttribute>();
                var secondaryIndexAttribute = property.GetCustomAttribute<SecondaryIndexAttribute>();

                if (secondaryIndexAttribute == null || columnAttribute == null)
                {
                    continue;
                }

                var indexSql = string.Format("CREATE INDEX IF NOT EXISTS ON \"{2}\".\"{0}\" (\"{1}\")", tableName, columnAttribute.Name, keySpace);
                try
                {
                    session.Execute(indexSql, ConsistencyLevel.LocalSerial);
                }
                catch (Exception ex)
                {
                    Curse.Logging.Logger.Error(ex, "Failed to create index: " + columnAttribute.Name + " on table " + tableName);
                }
                
            }
        }
           
        public ISession Session;
                       
        protected PreparedStatement GetPreparedStatement(string statement)
        {
            return Session.Prepare(string.Format(statement, TableName));
        }

        public virtual BoundStatement GetSelectStatement(params object[] keys)
        {
            return SelectStatement.Bind(keys);
        }

        public virtual BoundStatement GetSelectManyStatement(params object[] keys)
        {
            return SelectManyStatement.Bind(keys);
        }

        public virtual BoundStatement GetSelectAllStatement()
        {
            return SelectAllStatement.Bind();
        }

        public abstract BoundStatement GetUpdateStatement(TEntity model);
        public abstract BoundStatement GetInsertStatement(TEntity model);        
        public abstract BoundStatement GetDeleteStatement(TEntity model);

        protected abstract void PrepareStatements();

        protected PreparedStatement InsertStatement
        {
            get;
            set;
        }

        protected PreparedStatement UpdateStatement
        {
            get;
            set;            
        }

        protected PreparedStatement SelectStatement
        {
            get;
            set;    
        }

        protected PreparedStatement SelectManyStatement
        {
            get;
            set;
        }

        protected PreparedStatement DeleteStatement
        {
            get;
            set;    
        }

        protected PreparedStatement SelectAllStatement
        {
            get;
            set;
        }
    }
}
