﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Cassandra.Data.Linq;
using Cassandra;
using System.Collections.Concurrent;
using System.Reflection;

namespace Curse.Cassandra
{
    [Serializable]
    public abstract class BaseTable<TEntity, TManager>
        where TEntity : BaseTable<TEntity, TManager>, new()
        where TManager : BaseSessionManager<TEntity, TManager>, new()
    {
        static readonly bool IsRegional;
        static readonly string BaseKeySpace;
        static readonly string TableName;

        protected static string GetKeySpace(string region)
        {
            if(IsRegional)
            {
                return BaseKeySpace + region;
            }
            else
            {
                return BaseKeySpace;
            }
        }

        static readonly ConcurrentDictionary<int, CassandraConfiguration> ConfigByRegion = new ConcurrentDictionary<int, CassandraConfiguration>();
        static readonly ConcurrentDictionary<int, TManager> SessionsByRegion = new ConcurrentDictionary<int, TManager>();
        public static readonly TManager[] RemoteSessions;
        public static TManager LocalSession;                 
        
        public static readonly int LocalConfigID;

        public static TManager CreateSession(CassandraConfiguration config)
        {            
            var keySpace = GetKeySpace(config.RegionGroup);       
            return BaseSessionManager<TEntity, TManager>.Create(config.Cluster, keySpace, TableName, IsRegional);
        }

        

        public static TManager GetSession(int regionIdentifier)
        {
            return SessionsByRegion[regionIdentifier];
        }
           
        public abstract void Hydrate(Row row);

        public static TEntity GetLocal(params object[] keyValues)
        {
            return Get(LocalSession, keyValues);
        }

        public static TEntity Get(int regionID, params object[] keyValues)
        {
            return Get(GetSession(regionID), keyValues);
        }

        public static TEntity Get(TManager sessionManager, params object[] keyValues)
        {

            var boundStatement = sessionManager.GetSelectStatement(keyValues);
            var rowSet = sessionManager.Session.Execute(boundStatement);
            var rows = rowSet.GetRows().ToArray();

            if (rows.Length == 0)
            {
                return null;
            }

            var row = rows[0];

            var model = new TEntity();
            model.Hydrate(row);            
            return model;
        }

        public static TEntity[] GetAllLocal(params object[] keyValues)
        {
            return GetAll(LocalSession, keyValues);
        }

        public static TEntity[] GetAll(int regionID,  params object[] keyValues)
        {
            return GetAll(GetSession(regionID), keyValues);
        }

        public static TEntity[] GetAll(TManager sessionManager, params object[] keyValues)
        {
            var boundStatement = sessionManager.GetSelectManyStatement(keyValues);
            var rowSet = sessionManager.Session.Execute(boundStatement);
            var rows = rowSet.GetRows().ToArray();

            List<TEntity> entities = new List<TEntity>();
           

            foreach(var row in rows)
            {
                var model = new TEntity();
                model.Hydrate(row);
                entities.Add(model);

            }
            return entities.ToArray();
        }

        public static TEntity[] SelectAllLocal()
        {
            return SelectAll(LocalSession);
        }

        public static TEntity[] SelectAll(TManager sessionManager)
        {
            var boundStatement = sessionManager.GetSelectAllStatement();
            var rowSet = sessionManager.Session.Execute(boundStatement);
            var rows = rowSet.GetRows().ToArray();

            List<TEntity> entities = new List<TEntity>();


            foreach (var row in rows)
            {
                var model = new TEntity();
                model.Hydrate(row);
                entities.Add(model);

            }
            return entities.ToArray();
        }

        public void InsertLocal()
        {
            Insert(LocalSession);

        }
        public void Insert(int regionID)
        {
            Insert(GetSession(regionID));
        }

        public void Insert(TManager sessionManager)
        {
            
            var boundStatement = sessionManager.GetInsertStatement((TEntity)this);
            sessionManager.Session.Execute(boundStatement);
        }

        public void UpdateLocal()
        {
            Update(LocalSession);
        }

        public void Update(int regionID)
        {
            Update(GetSession(regionID));
        }

        public void Update(TManager sessionManager)
        {
            var boundStatement = sessionManager.GetUpdateStatement((TEntity)this);
            sessionManager.Session.Execute(boundStatement);            
        }
        
        public void Delete(int regionID)
        {
            var sessionManager = GetSession(regionID);
            Delete(sessionManager);
        }

        public void Delete(TManager sessionManager)
        {
            var boundStatement = sessionManager.GetDeleteStatement((TEntity)this);
            sessionManager.Session.Execute(boundStatement);
        }

        static BaseTable()
        {
            var tableDefinition = typeof(TEntity).GetCustomAttribute<TableDefinitionAttribute>();

            if(tableDefinition == null)
            {
                throw new Exception("All Cassandra entities must have a TableDefinition attribute.");
            }

            // Set our keyspace
            IsRegional = tableDefinition.IsRegional;
            BaseKeySpace = tableDefinition.KeySpace;
            TableName = tableDefinition.TableName;

            var remoteSessions = new List<TManager>();

            // Iterate over each configuration, and create a session for that region
            foreach (var config in CassandraConfiguration.Configurations)
            {
                var isValid = false;

                try
                {
                    var regionalKeySpace = GetKeySpace(config.RegionGroup);
                    using (var session = config.Cluster.Connect(regionalKeySpace))
                    {                       
                        isValid = true;
                    }                    
                
                }
                catch (Exception ex)
                {
                    Curse.Logging.Logger.Error(ex, "[Cassandra] Failed to open a database connection!", new { config.Addresses, config.RegionKey });
                }

                if (config.IsLocal && !isValid)
                {
                    throw new Exception("Unable to establish a session to the local cluster.");
                }

                ConfigByRegion.TryAdd(config.RegionIdentifier, config);
                var sessionManager = CreateSession(config);

                SessionsByRegion.TryAdd(config.RegionIdentifier, sessionManager);
                
                if(config.IsLocal)
                {
                    LocalConfigID = config.RegionIdentifier;
                    LocalSession = sessionManager;
                }
                else
                {
                    remoteSessions.Add(sessionManager);
                }
                                                             
            }

            RemoteSessions = remoteSessions.ToArray();
        }                           
    }
}
