﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace Curse.CloudFlare
{
    public static class CloudFlareApi
    {
        private static string _apiKey;
        private static string _apiEmailAddress;
        private const string BaseUrl = "https://api.cloudflare.com/client/v4";
        private static bool _isInitialized;

        private static readonly Dictionary<string, string> DomainToIDMap = new Dictionary<string, string>(); 

        public static void Initialize(string apiKey, string apiEmailAddress)
        {
            _apiKey = apiKey;
            _apiEmailAddress = apiEmailAddress;
            _isInitialized = true;
        }

        public static CloudFlareDnsRecord GetDnsRecord(string domain, string host, string domainID = null)
        {
            if (!_isInitialized)
            {
                throw new Exception("CloudFlareApi must be initialized!");

            }

            domainID = domainID ?? GetIDForDomain(domain);
            var json = SendRequest("GET", string.Format("zones/{0}/dns_records?type=A&name={1}.{2}", domainID, host, domain));
            var result = JsonConvert.DeserializeObject<CloudFlareResult<CloudFlareDnsRecord>>(json);            
            if (!result.success)
            {
                throw new Exception("Failed to retrieve domain status");
            }
            int totalFound = result.result_info.total_count;
            if (totalFound == 0)
            {
                return null;
            }

            return result.result.FirstOrDefault();

        }

        public static bool UpdateDnsHost(string domain, string host, string ipAddress)
        {
            if (!_isInitialized)
            {
                throw new Exception("CloudFlareApi must be initialized!");

            }

            var domainID = GetIDForDomain(domain);

            var existing = GetDnsRecord(domain, host, domainID);
            var fqdn = host + "." + domain;
            var record = new CloudflareDnsRecordPost { content = ipAddress, name = fqdn, type = "A", ttl = 120 };
            
            if (existing == null)
            {                
                var json = SendRequest("POST", string.Format("zones/{0}/dns_records", GetIDForDomain(domain)), JsonConvert.SerializeObject(record));
                
            }
            else if (existing.content != ipAddress)
            {
                record.ttl = existing.ttl;
                var json = SendRequest("PUT", string.Format("zones/{0}/dns_records/{1}", GetIDForDomain(domain), existing.id), JsonConvert.SerializeObject(record));
            }

          
            return true;
        }

        public static bool InvalidateUrl(string domain, string url)
        {
            if (!_isInitialized)
            {
                throw new Exception("CloudFlareApi must be initialized!");
            }

            var json = SendRequest("DELETE", string.Format("zones/{0}/purge_cache", GetIDForDomain(domain)), string.Format(@"{{""files"":[""{0}""]}}", url));
            dynamic d = JObject.Parse(json);
            return (bool) d.success;
        }

        private static string GetIDForDomain(string domain)
        {
            string id;
            if (DomainToIDMap.TryGetValue(domain, out id))
            {
                return id;
            }

            var json = SendRequest("GET", string.Format(@"zones?name={0}", domain));
            var response = JObject.Parse(json);

            var success = response["success"];
            if (success == null || success.Type != JTokenType.Boolean || !(bool) success)
            {
                throw new ArgumentException(string.Format("Error requesting details for domain " + domain));
            }

            var result = response["result"] as JArray;
            if (result == null || result.Count < 1)
            {
                throw new ArgumentException(string.Format("Domain not found: " + domain));
            }

            id = (string) result[0]["id"];
            DomainToIDMap[domain] = id;

            return id;
        }

        private static string SendRequest(string method, string relativeUrl, string requestJson = null)
        {
            var request = WebRequest.CreateHttp(string.Format(@"{0}/{1}", BaseUrl, relativeUrl));
            request.Method = method;
            request.Accept = "application/json";
            request.ContentType = "application/json";
            request.Headers["X-Auth-Email"] = _apiEmailAddress;
            request.Headers["X-Auth-Key"] = _apiKey;
            request.UserAgent = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/54.0.2840.59 Safari/537.36";


            if (requestJson != null)
            {
                using (var writer = new StreamWriter(request.GetRequestStream()))
                {
                    writer.Write(requestJson);
                }
            }

            using (var response = request.GetResponse())
            {
                using (var stream = response.GetResponseStream())
                {
                    if (stream == null || stream == Stream.Null)
                    {
                        return null;
                    }

                    using (var reader = new StreamReader(stream))
                    {
                        return reader.ReadToEnd();
                    }
                }
            }
        }
    }
}
