﻿
using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;

namespace Curse.CloudQueue
{
    /// <summary>
    /// Routed messages added to a queue specific to a server and region
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public class BaseCloudQueueRoutedMessage<T> : ICloudQueueRoutedMessage
        where T : class, ICloudQueueRoutedMessage
    {
        /// <summary>
        /// This is purely for serialization. Don't use it!
        /// </summary>
        protected BaseCloudQueueRoutedMessage() {  } 

        public BaseCloudQueueRoutedMessage(string destinationServerName, int destinationRegionID)
        {
            if (string.IsNullOrEmpty(destinationServerName))
            {
                throw new ArgumentException("destinationServerName cannot be null or empty!");
            }

            if (destinationRegionID == 0)
            {
                throw new ArgumentException("destinationRegionID must be a valid region id!");
            }
            
            Completed = true;
            DestinationServerName = destinationServerName;
            DestinationRegionID = destinationRegionID;
            EnqueuedTimestamp = DateTime.UtcNow;
        }

        public DateTime EnqueuedTimestamp
        {
            get;
            set;
        }
        
        public int Retries
        {
            get;
            set;
        }

        public string DestinationServerName
        {
            get;
            set;
        }

        public int DestinationRegionID
        {
            get;
            set;
        }

        [JsonIgnore]
        public bool Completed
        {
            get;
            set;
        }

        static BaseCloudQueueRoutedMessage()
        {
            var type = typeof(T);
            string queueName = type.Name;

            var queueAttribute = typeof(T).GetCustomAttribute<CloudQueueAttribute>();
            var isPersistent = false;            
            if (queueAttribute != null)
            {
                isPersistent = queueAttribute.IsPersistent;                
            }

            QueueManager = new CloudQueueManager<T>(true, true, isPersistent, null, null);
            bool found = false;
            var ctors = type.GetConstructors();
            foreach (var ctor in ctors)
            {
                if (ctor.IsPublic && ctor.GetParameters().Count() == 0)
                {
                    found = true;
                    break;
                }
            }
            if (!found)
            {
                throw new Exception(type.Name + " class should have a public default constructor");
            }
        }       

        public static void StartProcessor(Action<T> handler)
        {

            var queueAttribute = typeof(T).GetCustomAttribute<CloudQueueAttribute>();
            var isPersistent = false;            
            if (queueAttribute != null)
            {
                isPersistent = queueAttribute.IsPersistent;                
            }

            var attribute = typeof(T).GetCustomAttribute<CloudQueueProcessorAttribute>();
            int? maxParallel = null;
            var isAcknowledged = false;
            if (attribute != null)
            {
                maxParallel = attribute.MaxParallel;
                isAcknowledged = attribute.IsAcknowledged;
            }
            _queueProcessor = new CloudQueueProcessor<T>(true, true, isPersistent, isAcknowledged, null, null, null, maxParallel);
            _queueProcessor.Start(handler);
        }

        public virtual void Enqueue()
        {            
            QueueManager.Enqueue(this as T, DestinationServerName, DestinationRegionID);
        }

        public static readonly CloudQueueManager<T> QueueManager;
        private static CloudQueueProcessor<T> _queueProcessor;
    }
}
