﻿using Newtonsoft.Json;
using System;
using System.Reflection;
using Curse.Logging;

namespace Curse.CloudQueue
{
    /// <summary>
    /// Routed messages added to a queue specific to a server and region
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public class BaseCloudQueueShoveledMessage<T> : ICloudQueueShoveledMessage
        where T : class, ICloudQueueShoveledMessage
    {
        protected static readonly LogCategory Logger;

        /// <summary>
        /// This is purely for serialization. Don't use it!
        /// </summary>
        [JsonConstructor]
        protected BaseCloudQueueShoveledMessage() {  } 

        public BaseCloudQueueShoveledMessage(int destinationRegionID)
        {            
            Completed = true;
            SourceRegionID = QueueConfiguration.LocalRegionID;
            DestinationRegionID = destinationRegionID;
            EnqueuedTimestamp = DateTime.UtcNow;            
        }

        /// <summary>
        /// Where this message originated from
        /// </summary>
        public int SourceRegionID
        {
            get;
            set;
        }

        public bool IsLocal()
        {
            return SourceRegionID == QueueConfiguration.LocalRegionID;
        }

        public DateTime EnqueuedTimestamp
        {
            get;
            set;
        }
        
        public int Retries
        {
            get;
            set;
        }       

        public int? DestinationRegionID
        {
            get;
            set;
        }

        [JsonIgnore]
        public bool Completed
        {
            get;
            set;
        }

        static BaseCloudQueueShoveledMessage()
        {            
            string queueName = typeof(T).Name;
            Logger = new LogCategory(queueName);

            var queueAttribute = typeof(T).GetCustomAttribute<CloudQueueAttribute>();
            if (queueAttribute != null)
            {
                IsPersistent = queueAttribute.IsPersistent;
            }

            var workerAttribute = typeof(T).GetCustomAttribute<CloudWorkerQueueAttribute>();

            if (workerAttribute != null)
            {
                MaxQueueLength = workerAttribute.MaxQueueLength;
                MaxQueueBytes = workerAttribute.MaxQueueBytes;
            }

            QueueManager = new CloudQueueManager<T>(true, false, IsPersistent, null, null, MaxQueueLength, MaxQueueBytes);            
        }       

        public static void StartProcessor(Action<T> handler)
        {
            var processorAttribute = typeof(T).GetCustomAttribute<CloudQueueProcessorAttribute>();
            int? maxParallel = null;
            var isAcknowledged = false;
            double? prefetchMultiplier = null;
            if (processorAttribute != null)
            {
                maxParallel = processorAttribute.MaxParallel;
                isAcknowledged = processorAttribute.IsAcknowledged;
                prefetchMultiplier = processorAttribute.PrefetchMultiplier;
            }

            _queueProcessor = new CloudQueueProcessor<T>(true, false, IsPersistent, isAcknowledged, null, null, MaxQueueLength, maxParallel, prefetchMultiplier, MaxQueueBytes);
            _queueProcessor.Start(handler);
        }

        private static readonly int? MaxQueueLength = null;
        private static readonly int? MaxQueueBytes = null;
        private static readonly bool IsPersistent = false;

        public virtual void Enqueue()
        {
            if (!DestinationRegionID.HasValue)
            {
                foreach(var regionID in QueueConfiguration.AllRegions)
                {
                    T message = this as T;
                    message.DestinationRegionID = regionID;
                    QueueManager.Enqueue(message, null, regionID);                    
                }
            }
            else
            {
                QueueManager.Enqueue(this as T, null, DestinationRegionID.Value);
            }
        }

        public static readonly CloudQueueManager<T> QueueManager;
        private static CloudQueueProcessor<T> _queueProcessor;
    }
}
