﻿
using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;
using Curse.Logging;

namespace Curse.CloudQueue
{

    /// <summary>
    ///  Worker messages are added to a shared queue, and processed by whichever worker dequeues it first.
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public class BaseCloudQueueWorkerMessage<T> : ICloudQueueMessage
        where T : class, ICloudQueueMessage
    {
        protected static readonly LogCategory Logger;                        

        public BaseCloudQueueWorkerMessage()
        {            
            Completed = true;
            EnqueuedTimestamp = DateTime.UtcNow;
        }

        public DateTime EnqueuedTimestamp
        {
            get;
            set;
        }

        [JsonIgnore]
        public bool Completed
        {
            get;
            set;
        }

        public int Retries
        {
            get;
            set;
        }

        public static void Initialize()
        {
            Logger.Info("Initializing " + typeof(T).Name + "...");
        }

        static BaseCloudQueueWorkerMessage()
        {
            try
            {                                                       
                Logger = new LogCategory(typeof (T).Name);

                var queueAttribute = typeof (T).GetCustomAttribute<CloudQueueAttribute>();

                if (queueAttribute != null)
                {
                    IsPersistent = queueAttribute.IsPersistent;
                }

                var workerAttribute = typeof (T).GetCustomAttribute<CloudWorkerQueueAttribute>();

                if (workerAttribute != null)
                {
                    MaxQueueLength = workerAttribute.MaxQueueLength;
                    MaxQueueBytes = workerAttribute.MaxQueueBytes;
                }
                
                QueueManager = new CloudQueueManager<T>(false, false, IsPersistent, null, null, MaxQueueLength, MaxQueueBytes);                
            }
            catch (Exception ex)
            {
                Logger.Fatal(ex, "Failed to initialize queue manager for '" + typeof(T).Name + "'");
                throw;
            }                      
        }

        public virtual void Enqueue()
        {
            T model = this as T;
            QueueManager.Enqueue(model);
        }

        private static readonly int? MaxQueueBytes = null;
        private static readonly int MaxQueueLength = 1000;
        private static readonly bool IsPersistent = false;
        
        public static void StartProcessor(Action<T> handler)
        {
            var attribute = typeof(T).GetCustomAttribute<CloudQueueProcessorAttribute>();
            int? maxParallel = null;
            var isAcknowledged = false;
            double? prefetchMultiplier = null;
            if (attribute != null)
            {
                maxParallel = attribute.MaxParallel;
                isAcknowledged = attribute.IsAcknowledged;
                prefetchMultiplier = attribute.PrefetchMultiplier;
            }
            _queueProcessor = new CloudQueueProcessor<T>(false, false, IsPersistent, isAcknowledged, null, null, MaxQueueLength, maxParallel, prefetchMultiplier, MaxQueueBytes);            
            _queueProcessor.Start(handler);
        }
        
        public static readonly CloudQueueManager<T> QueueManager;
        private static CloudQueueProcessor<T> _queueProcessor;
    }
}
