﻿using System.Xml.Serialization;
using System;
using System.Collections.Generic;
using System.Linq;

namespace Curse.CloudQueue
{
    [XmlType("Queue")]
    public class QueueConfiguration
    {
        // This must be increments if anything substantial changes, which would require the queues to be reset
        public static int Version = 4;

        public static bool SingleRegionMode = false;
        public static QueueConfiguration[] Configurations;
        public static QueueConfiguration LocalConfiguration;
        public static int LocalRegionID;
        public static HashSet<int> AllRegions;
        public static HashSet<int> RemoteRegions;
        public static bool CreateQueues = false;
        public static bool FastCreateQueues = false;

        [XmlElement()]
        public int TypeRouteID
        {
            get;
            set;
        }

        [XmlElement()]
        public int RegionIdentifier
        {
            get;
            set;
        }

        [XmlElement()]
        public string RegionKey
        {
            get;
            set;
        }

        [XmlArray()]
        [XmlArrayItem("Address")]
        public string[] Addresses
        {
            get;
            set;
        }

        [XmlElement()]
        public int Port
        {
            get;
            set;
        }

        [XmlElement()]
        public string Username
        {
            get;
            set;
        }

        [XmlElement()]
        public string Password
        {
            get;
            set;
        }
        
        [XmlIgnore]
        public bool IsLocal
        {
            get;
            private set;
        }
        
        public HashSet<string> Types
        {
            get;
            set;
        }

        public static IReadOnlyDictionary<int, QueueTypeRoute> ExplicitQueueConfig { get; private set; }

        public delegate bool QueueVersionCheckDelegate(string queueName);
        public delegate void QueueVersionUpdateDelegate(string queueName);
        private static QueueVersionCheckDelegate _queueVersionCheck;
        private static QueueVersionUpdateDelegate _queueVersionUpdate;
        
        public static bool QueueIsValid(string name)
        {
            if(_queueVersionCheck == null)
            {
                return false;
            }

            return _queueVersionCheck(name);
        }

        public static void MarkQueueAsValid(string name)
        {
            if (_queueVersionUpdate == null)
            {
                return;
            }

            _queueVersionUpdate(name);
        }

        public static HashSet<string> ExplcitQueueRoutes { get; private set; }

        public static bool ReplaceRemoteQueues { get; set; }

        public static void Initialize(string localRegionIdentifier, QueueConfigurationCollection configurations, QueueVersionCheckDelegate queueVersionCheck, QueueVersionUpdateDelegate queueVersionUpdate)
        {
            _queueVersionCheck = queueVersionCheck;
            _queueVersionUpdate = queueVersionUpdate;
            SingleRegionMode = configurations.SingleRegionMode;
            Configurations = configurations.Configurations.ToArray();

            ExplcitQueueRoutes = new HashSet<string>(configurations.TypeRouting.SelectMany(p => p.Types).ToArray());

            var typeRoutes = configurations.TypeRouting.ToDictionary(p => p.ID);
            ExplicitQueueConfig = typeRoutes;
            
            foreach (var config in Configurations)
            {
                QueueTypeRoute route;
                if (!typeRoutes.TryGetValue(config.TypeRouteID, out route))
                {
                    throw new InvalidOperationException("Type route could not be found for queue configuration!");
                }

                config.Types = new HashSet<string>(route.Types);

                if (config.RegionKey.Equals(localRegionIdentifier, StringComparison.InvariantCultureIgnoreCase))
                {
                    config.IsLocal = true;
                    LocalConfiguration = config;
                    LocalRegionID = config.RegionIdentifier;
                }                             
            }
            
            AllRegions = new HashSet<int>(Configurations.Select(p => p.RegionIdentifier));
            RemoteRegions = new HashSet<int>(Configurations.Select(p => p.RegionIdentifier).Where(p => p != LocalRegionID));

            RabbitConnectionManager.Initialize(Configurations, LocalRegionID);
        }
    }
}
