﻿using Curse.Logging;
using RabbitMQ.Client;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;

namespace Curse.CloudQueue
{
    public class RabbitConnectionManager
    {
        private static readonly Dictionary<string, RabbitConnectionManager>  ConnectionManagersByType 
            = new Dictionary<string, RabbitConnectionManager>(StringComparer.InvariantCultureIgnoreCase);

        private static Dictionary<int, string> _remoteRegionNames;

        private static readonly LogCategory Logger = new LogCategory("RabbitConnectionManager");

        private readonly object _syncRoot = new object();      
        private readonly QueueConfiguration _localConfiguration;

        public static int LocalRegionID;

        public QueueConfiguration LocalConfiguration
        {
            get { return _localConfiguration; }
        }

        private readonly Dictionary<int, QueueConfiguration> _remoteConfigurationsByRegion;
        
        private readonly List<RabbitHost> _hosts = new List<RabbitHost>();

        public static string GetRemoteRegionKey(int id)
        {
            string regionKey;
            if (!_remoteRegionNames.TryGetValue(id, out regionKey))
            {
                throw new InvalidOperationException("Failed to get region key for id: " + id);
            }

            return regionKey;
        }

        public static void Initialize(QueueConfiguration[] allConfigurations, int localRegionID)
        {
            LocalRegionID = localRegionID;
            _remoteRegionNames= new Dictionary<int, string>();
            
            foreach (var config in allConfigurations)
            {
                _remoteRegionNames[config.RegionIdentifier] = config.RegionKey;
            }             

            // For each local config, create a connection manager
            foreach (var config in allConfigurations.Where(p => p.IsLocal))
            {
                // Try to find the most suitable remote configs
                var remoteConfigs = allConfigurations.Where(p => !p.IsLocal && p.Types.SetEquals(config.Types)).ToArray();
                var manager = new RabbitConnectionManager(config, remoteConfigs);
                foreach (var type in config.Types)
                {
                    ConnectionManagersByType[type] = manager;
                }
            }

        }

        public RabbitConnectionManager(QueueConfiguration localConfiguration, QueueConfiguration[] remoteConfigurations)
        {
            lock (_syncRoot)
            {
                if (_localConfiguration != null) return;

                _localConfiguration = localConfiguration;
                _remoteConfigurationsByRegion = remoteConfigurations.ToDictionary(p => p.RegionIdentifier);

                Logger.Info("Initializing RabbitMQ Connection Manager", new { Configuration = localConfiguration });

                foreach (var address in localConfiguration.Addresses)
                {
                    Logger.Info("Creating a RabbitMQ host: " + address);

                    try
                    {
                        var host = new RabbitHost(address, localConfiguration.Port, localConfiguration.Username, localConfiguration.Password);
                        _hosts.Add(host);                       
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex);
                    }
                }

                Logger.Info("Setting initial host health...");
                foreach (var host in _hosts)
                {
                    Logger.Info("Determining health of host: " + host.Address);
                    host.CheckHealth();
                }
            }
        }
        
        public IConnection GetConnection(int maxAttempts = 30)
        {                        
            var attempts = 0;

            while (true)
            {
                ++attempts;
                if (!_hosts.Any())
                {
                    Logger.Error("No hosts found!");
                    return null;                        
                }

                var healthyHosts = _hosts.OrderByDescending(p => p.IsHealthy).ToArray();
                foreach (var host in healthyHosts)
                {                        
                    var connection = host.GetConnection();
                    if (connection != null)
                    {
                        return connection;
                    }
                }

                Logger.Warn("Unable to find a valid RabbitMQ connection after " + attempts + " attempts", _hosts);

                if (attempts > maxAttempts)
                {
                    Logger.Fatal("RabbitMQ connection manager cannot connect to any node in the cluster.", _hosts);
                    throw new Exception("Failed to get a valid connection from Rabbit Connection Manager.");
                }

                Thread.Sleep(100);
            }
            
        }

        public QueueConfiguration GetRemoteConfiguration(int regionID)
        {
            QueueConfiguration configuration;
            if (!_remoteConfigurationsByRegion.TryGetValue(regionID, out configuration))
            {
                throw new Exception("Failed to find a configuration for Rabbit in " + regionID);
            }
            return configuration;
        }

        public IConnection GetRemoteConnection(int regionID)
        {
            QueueConfiguration configuration;
            if (!_remoteConfigurationsByRegion.TryGetValue(regionID, out configuration))
            {
                throw new Exception("Failed to find a configuration for Rabbit in " + regionID);
            }

            foreach (var address in configuration.Addresses)
            {
                var factory = new ConnectionFactory()
                {
                    HostName = address,
                    Port = configuration.Port,
                    UserName = configuration.Username,
                    Password = configuration.Password,
                    RequestedConnectionTimeout = 5000
                };

                var connection = factory.CreateConnection();
                if (connection != null && connection.IsOpen)
                {
                    return connection;
                }
            }
            
            throw new Exception("Failed to find a working connection for Rabbit in " + regionID);            
        }

        public string CurrentHostName
        {
            get
            {

                var host = _hosts.FirstOrDefault(p => p.IsHealthy);
                if (host == null)
                {
                    return null;
                }

                return host.Address;
            }
        }

        public static RabbitConnectionManager GetConnectionManager(string typeName)
        {
            RabbitConnectionManager connectionManager;
            if (ConnectionManagersByType.TryGetValue(typeName, out connectionManager))
            {
                return connectionManager;
            }

            Logger.Fatal("RabbitMQ connection manager is unable to get a connection manager for type:" + typeName);
            throw new Exception("Failed to get a valid connection from Rabbit Connection Manager.");
        }

        public static IConnection GetConnection(string typeName, int maxAttempts = 30)
        {
            RabbitConnectionManager connectionManager;
            if (ConnectionManagersByType.TryGetValue(typeName, out connectionManager))
            {
                return connectionManager.GetConnection(maxAttempts);
            }

            Logger.Fatal("RabbitMQ connection manager is unable to get a connection manager for type:" + typeName);
            throw new Exception("Failed to get a valid connection from Rabbit Connection Manager.");
        }

        public int[] GetRemoteRegionIDs()
        {
            return _remoteConfigurationsByRegion.Keys.ToArray();
        }

        public string GetRegionKey(int id)
        {
            QueueConfiguration config;
            if (!_remoteConfigurationsByRegion.TryGetValue(id, out config))
            {
                throw new InvalidOperationException("Failed to get region key for id: " + id);
            }

            return config.RegionKey;
        }

        public static int[] GetRemoteRegions(string typeName)
        {
            RabbitConnectionManager connectionManager;
            if (ConnectionManagersByType.TryGetValue(typeName, out connectionManager))
            {
                return connectionManager.GetRemoteRegionIDs();
            }

            Logger.Fatal("RabbitMQ connection manager is unable to get a remote regions for type:" + typeName);
            throw new Exception("Failed to get a valid connection from Rabbit Connection Manager.");
            
        }
        
        public static IConnection GetRemoteConnection(string typeName, int regionID)
        {
            RabbitConnectionManager connectionManager;
            if (ConnectionManagersByType.TryGetValue(typeName, out connectionManager))
            {
                return connectionManager.GetRemoteConnection(regionID);
            }

            Logger.Fatal("RabbitMQ connection manager is unable to get a remote manager for type:" + typeName);
            throw new Exception("Failed to get a valid connection from Rabbit Connection Manager.");
        }
    }
}
