﻿using Curse.Logging;
using RabbitMQ.Client;
using System;

namespace Curse.CloudQueue
{
    public class RabbitHost
    {
        private static readonly LogCategory Logger = new LogCategory("RabbitConnectionManager");        
        private volatile bool _isHealthy = false;        
        private readonly ConnectionFactory _connectionFactory;
        
        public string Address
        {
            get;
            private set;
        }        

        public RabbitHost(string address, int port, string username, string password)
        {
            Address = address;

            _connectionFactory = new ConnectionFactory()
            {
                HostName = address,                 
                Port = port,
                UserName = username,
                Password = password,
                RequestedConnectionTimeout = 1000,  
                RequestedHeartbeat = 60, 
            };                        
        }

        public void CheckHealth()
        {
            Logger.Info("Attempting to validate connection to: " + Address);
            try
            {
                using (var connection = _connectionFactory.CreateConnection())
                {
                    if (connection.IsOpen)
                    {
                        Logger.Info("Found healthy Rabbit node: " + Address);
                        _isHealthy = true;
                    }
                    else
                    {
                        Logger.Warn("Unable to connect to RabbitMQ host: " + Address);
                        _isHealthy = false;
                    }
                } 
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Unable to connect to RabbitMQ host: " + Address);
                _isHealthy = false;
            }
            
        }

        public IConnection GetConnection()
        {

            IConnection connection = null;

            try
            {
                connection = _connectionFactory.CreateConnection();                
            }
            catch (Exception ex)
            {

                Logger.Error(ex, "Failed to create RabbitMQ connection on host: " + Address, new { Diagnostics = ex.ToString() });
                _isHealthy = false;
                return null;
            }

            if (connection != null && connection.IsOpen)
            {
                _isHealthy = true;
                return connection;
            }
            else
            {
                if (_isHealthy)
                {                    
                    Logger.Warn("Setting RabbitMQ connection to unhealthy: " + Address);
                }
                _isHealthy = false;
                return null;
            }
            
        }

        public bool IsHealthy
        {
            get
            {
                return _isHealthy;
            }
        }
    }
}
