﻿using System.Collections.Concurrent;
using Curse.Logging;
using Newtonsoft.Json;
using RabbitMQ.Client;
using RabbitMQ.Client.Events;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using RabbitMQ.Client.Framing;

namespace Curse.CloudQueue
{

    public class RabbitQueue<T> : ICloudQueue<T>
        where T : ICloudQueueMessage
    {

        private static volatile bool _shuttingDown = false;

        public Action<T> MessageProcessor;
        private readonly LogCategory Logger;
        private readonly string TypeName;
        private readonly string BaseQueueName;
        protected int RetryLimit;
        
        private string GetRemoteQueueName(string regionKey)
        {
            return "Remote-" + BaseQueueName + "-" + regionKey;
        }

        private readonly bool _isShoveled = false;
        private readonly bool _isMachineRouted = false;
        private readonly BasicProperties _basicPublishProperties = null;
        private readonly bool _isAcknowledged = false;
        private readonly CloudQueueMode _mode;

        public RabbitQueue(CloudQueueMode mode, string queueName, bool isShoveled, bool isMachineRouted, bool isPersistent, bool isAcknowledged = false, int? retryLimit = null, int? messageTtlMilliseconds = null, int? maxQueueLength = null, int? maxParallelProcessing = null, double? prefetchMultipler = null, int? maxQueueBytes = null)
            :this(RabbitConnectionManager.GetConnectionManager(typeof(T).Name),mode,queueName,isShoveled,isMachineRouted,isPersistent,isAcknowledged,retryLimit,messageTtlMilliseconds,maxQueueLength,maxParallelProcessing, prefetchMultipler, maxQueueBytes)
        {
        }

        public RabbitQueue(RabbitConnectionManager connectionManager, CloudQueueMode mode, string queueName, bool isShoveled, bool isMachineRouted, bool isPersistent, bool isAcknowledged = false, int? retryLimit = null, int? messageTtlMilliseconds = null, int? maxQueueLength = null, int? maxParallelProcessing = null, double? prefetchMultipler = null, int? maxQueueBytes = null)
        {
            TypeName = typeof(T).Name;
            Logger = new LogCategory("RabbitQueue-" + typeof(T).Name + "-" + mode);
            Logger.Trace("Setting up queue...");

            _mode = mode;
            _isShoveled = isShoveled;
            _isMachineRouted = isMachineRouted;
            _isAcknowledged = isAcknowledged;

#if DEBUGLOCAL
            if (!isMachineRouted)
            {
                BaseQueueName = queueName + "-" + Environment.MachineName;
            }
            else
            {
                BaseQueueName = queueName;
            }
#else
            BaseQueueName = queueName;
#endif

            RetryLimit = retryLimit ?? 5;

            var args = new Dictionary<string, object>();

            if (messageTtlMilliseconds.HasValue && messageTtlMilliseconds.Value > 0)
            {
                args.Add("x-message-ttl", messageTtlMilliseconds.Value);
            }

            if (maxQueueLength.HasValue && maxQueueLength.Value > 0)
            {
                args.Add("x-max-length", maxQueueLength);
            }

            if(maxQueueBytes.HasValue && maxQueueBytes.Value > 0)
            {
                args.Add("x-max-length-bytes", maxQueueBytes.Value);
            }

            if (maxParallelProcessing.HasValue && maxParallelProcessing.Value > 0)
            {
                _maxParallelProcessing = maxParallelProcessing.Value;
            }
            else
            {
                _maxParallelProcessing = Environment.ProcessorCount;
            }

            if (prefetchMultipler.HasValue && prefetchMultipler.Value > 1)
            {
                _prefetchMultipler = prefetchMultipler.Value;
            }

            if (_maxParallelProcessing < 1)
            {
                Logger.Warn("MaxParallelProcessing was set below the minimum value. It will default to 2.");
                _maxParallelProcessing = 2;
            }

#if DEBUGLOCAL
            _maxParallelProcessing = 1;
#endif

            if (isPersistent)
            {
                _basicPublishProperties = new BasicProperties { DeliveryMode = 2 };
            }

            var sw = Stopwatch.StartNew();

            try
            {
                SetupQueue(connectionManager, args);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to setup queue. We'll assume it's already setup, and continue.");
#if DEBUG
                throw;
#endif
            }


            CloudQueueInstanceManager.RegisterProcessor(this);

            if (mode == CloudQueueMode.Publisher)
            {
                new Thread(ProcessSendQueue) { IsBackground = true }.Start();
            }

            Logger.Trace("Finished initialization in " + sw.Elapsed.TotalSeconds.ToString("###,##0.00") + " seconds", new { _isShoveled, _isMachineRouted, messageTtlMilliseconds, maxQueueLength, _maxParallelProcessing });
        }

        public void SetupQueue(RabbitConnectionManager connManager, Dictionary<string, object> args)
        {
            if (QueueConfiguration.FastCreateQueues)
            {
                Logger.Trace("Fast create queue option enabled. Checking if queue is valid...");
                if (QueueConfiguration.QueueIsValid(BaseQueueName))
                {
                    return;                    
                }
            }
            else if (!QueueConfiguration.CreateQueues)
            {
                Logger.Trace("Skipping queue creation, due to configuration.");
                return;
            }

            using (var conn = connManager.GetConnection())
            {
                using (var channel = conn.CreateModel())
                {
                    if (_isShoveled) // If this is routed, we need to declare a local exchange as well as a named queue, for each remote region
                    {
                        if (_isMachineRouted)
                        {
                            Logger.Info("Creating exchange '" + BaseQueueName + "'");
                            channel.ExchangeDeclare(BaseQueueName, "direct", true);
                        }
                        else
                        {
                            CreateOrReplaceQueue(conn, BaseQueueName, true, args);
                        }


                        foreach (var regionID in connManager.GetRemoteRegionIDs())
                        {
                            try
                            {
                                using (var remoteConnection = connManager.GetRemoteConnection(regionID))
                                {
                                    using (var remoteChannel = remoteConnection.CreateModel())
                                    {
                                        if (_isMachineRouted)
                                        {
                                            Logger.Info("Creating remote exchange '" + BaseQueueName + "' for shoveling.");
                                            remoteChannel.ExchangeDeclare(BaseQueueName, "direct", true);
                                        }
                                        else
                                        {
                                            Logger.Info("Creating remote queue '" + BaseQueueName + "' for shoveling.");
                                            if (QueueConfiguration.ReplaceRemoteQueues)
                                            {
                                                CreateOrReplaceQueue(remoteConnection, BaseQueueName, true, args);
                                            }
                                            else
                                            {
                                                remoteChannel.QueueDeclare(BaseQueueName, true, false, false, args);
                                            }
                                        }
                                    }
                                }

                                var remoteQueueName = GetRemoteQueueName(connManager.GetRegionKey(regionID));

                                if (_isMachineRouted)
                                {
                                    Logger.Info("Creating local exchange '" + remoteQueueName + "' for shoveling.");
                                    channel.ExchangeDeclare(remoteQueueName, "topic", false, false, null);

                                }
                                else
                                {
                                    Logger.Info("Creating local queue '" + remoteQueueName + "' for shoveling.");
                                    if (QueueConfiguration.ReplaceRemoteQueues)
                                    {
                                        CreateOrReplaceQueue(conn, remoteQueueName, true, args);
                                    }
                                    else
                                    {
                                        channel.QueueDeclare(remoteQueueName, true, false, false, args);
                                    }
                                }

                                var shovelApi = "http://" + connManager.CurrentHostName + ":15672";
                                var remoteConfiguration = connManager.GetRemoteConfiguration(regionID);

                                Logger.Info("Creating shovel ", new
                                {
                                    shovelApi, 
                                    Source = string.Join(", ", connManager.LocalConfiguration.Addresses),
                                    Destination = string.Join(", ", remoteConfiguration.Addresses),
                                    LocalQueueName = remoteQueueName,
                                    RemoteQueueName = BaseQueueName
                                });

                                ConfigureShovel(
                                    shovelApi,
                                   connManager.LocalConfiguration.Addresses,
                                   remoteConfiguration.Addresses,
                                   connManager.LocalConfiguration.Username,
                                   connManager.LocalConfiguration.Password,
                                   remoteConfiguration.Port,
                                   "Shovel-" + BaseQueueName + "-" + remoteConfiguration.RegionKey,
                                   remoteQueueName,
                                   _isMachineRouted,
                                   BaseQueueName);
                            }
                            catch (Exception ex)
                            {
                                Logger.Error(ex, "Failed to setup queue '" + BaseQueueName + "' for remote region '" + regionID + "'");
                            }
                        }
                    }
                    else // Otherwise, we need to declare just a local queue
                    {
                        CreateOrReplaceQueue(conn, BaseQueueName, true, args);
                    }
                }
            }


            QueueConfiguration.MarkQueueAsValid(BaseQueueName);
        }

        private void CreateOrReplaceQueue(IConnection connection, string queueName, bool durable, IDictionary<string, object> args)
        {
            try
            {
                using (var channel = connection.CreateModel())
                {
                    channel.QueueDeclare(queueName, durable, false, false, args);
                }
            }
            catch
            {
                using (var channel = connection.CreateModel())
                {
                    channel.QueueDelete(queueName);
                    channel.QueueDeclare(queueName, durable, false, false, args);
                }
            }
        }

        private void WaitForCompletion()
        {
            var attempts = 0;
            // Wait a bit for existing processing operations to stop
            while (_currentlyProcessing > 0 || _currentlySending)
            {
                Thread.Sleep(250);
                if (++attempts >= 5)
                {
                    break;
                }
            }
        }

        public void CloseConnections()
        {            
            _shuttingDown = true;

            try
            {
                Logger.Info("Waiting for queue to finish...");
                WaitForCompletion();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to wait for completion!");
            }
        }

        public int CurrentlyProcessing
        {
            get { return _currentlyProcessing; }
        }

        public long TotalProcessed
        {
            get { return _totalProcessed; }
        }

        public TimeSpan AverageProcessTime
        {
            get
            {
                if (_totalProcessed == 0)
                {
                    return TimeSpan.Zero;
                }

                lock (_processingTimeLock)
                {
                    return TimeSpan.FromMilliseconds(_totalProcessingTime.TotalMilliseconds / _totalProcessed);
                }
            }
        }

        public TimeSpan BestProcessTime
        {
            get { return _bestProcessingTime; }
        }

        public TimeSpan WorstProcessTime
        {
            get { return _worstProcessingTime; }
        }

        public TimeSpan AverageQueuedTime
        {
            get
            {
                if (_totalProcessed == 0)
                {
                    return TimeSpan.Zero;
                }

                lock (_processingTimeLock)
                {
                    return TimeSpan.FromMilliseconds(_totalQueuedTime.TotalMilliseconds / _totalProcessed);
                }
            }
        }

        public TimeSpan BestQueuedTime
        {
            get { return _bestQueuedTime; }
        }

        public TimeSpan WorstQueuedTime
        {
            get { return _worstQueuedTime; }
        }

        public CloudQueueStats Stats
        {
            get
            {
                return new CloudQueueStats
                {
                    QueueName = Name,
                    CurrentlyProcessing = CurrentlyProcessing,
                    TotalMessagesProcessed = TotalProcessed,
                    AverageProcessTime = AverageProcessTime,
                    BestProcessTime = BestProcessTime,
                    WorstProcessTime = WorstProcessTime,
                    AverageQueuedTime = AverageQueuedTime,
                    BestQueuedTime = BestQueuedTime,
                    WorstQueuedTime = WorstQueuedTime
                };
            }
        }

        public string Name
        {
            get { return typeof(T).Name; }
        }

        public void Enqueue(T value, string destinationServerName = null, int? destinationRegionID = null)
        {
            if (_mode != CloudQueueMode.Publisher)
            {
                throw new InvalidOperationException("This queue instance is not a publisher: " + _mode);
            }

            if (_isShoveled && !destinationRegionID.HasValue)
            {
                throw new ArgumentException("destinationRegionID must be specified when the queue is machine specific. Queue type: " + typeof(T).Name);
            }

            if (QueueConfiguration.SingleRegionMode && _isShoveled && destinationRegionID != QueueConfiguration.LocalRegionID)
            {
                Logger.Trace("Single Region Mode: Rerouting message to local region!", new { DestinationRegion = destinationRegionID });
                destinationRegionID = QueueConfiguration.LocalRegionID;
            }

            // Set the timestamp for when this was inserted into RabbitMQ
            value.EnqueuedTimestamp = DateTime.UtcNow;

            var serialized = JsonConvert.SerializeObject(value);
            var message = Encoding.UTF8.GetBytes(serialized);

            if (_isShoveled)
            {
                EnqueueRouted(message, destinationRegionID.Value, destinationServerName);
            }
            else
            {
                EnqueueNormal(message);
            }

        }

        private void EnqueueRouted(byte[] serializedValue, int destinationRegionID, string destinationServerName)
        {
            try
            {
                if (QueueConfiguration.SingleRegionMode && destinationRegionID != QueueConfiguration.LocalRegionID)
                {
                    Logger.Warn("Unable to publish to queue. SingleRegionMode is enabled, and the destination supplied is not the local one.");                    
                    return;                    
                }

                
                if (destinationRegionID != RabbitConnectionManager.LocalRegionID) // This needs to be added to the queue
                {
                    var regionKey = RabbitConnectionManager.GetRemoteRegionKey(destinationRegionID);
                    var shovelQueueName = GetRemoteQueueName(regionKey);
                    // Add this to the local queue, for shoveling                
                    if (_isMachineRouted)
                    {
                        Logger.Trace("Publishing a '" + typeof(T).Name + "' to region " + destinationRegionID + " via exchange '" + shovelQueueName + "' with routing key '" + destinationServerName + "'");
                        TryBasicPublish(shovelQueueName, destinationServerName, _basicPublishProperties, serializedValue);
                    }
                    else
                    {
                        Logger.Trace("Publishing a '" + typeof(T).Name + "' to region " + destinationRegionID + " via queue '" + shovelQueueName + "'");
                        TryBasicPublish(string.Empty, shovelQueueName, _basicPublishProperties, serializedValue);
                    }
                }
                else if (_isMachineRouted) // Add this to the named exchange
                {
                    TryBasicPublish(BaseQueueName, destinationServerName, _basicPublishProperties, serializedValue);
                }
                else // Add this to the local queue
                {
                    EnqueueNormal(serializedValue);
                }
            }
            catch (KeyNotFoundException ex)
            {
                Logger.Error(ex, "EnqueueRouted failed. The configuration region key specified could not be found: " + destinationRegionID);
                throw;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "EnqueueRouted failed! Repairing connection...");
                throw;
            }
        }

        private class RabbitQueueItem
        {
            public string Exchange { get; set; }

            public string RoutingKey { get; set; }

            public IBasicProperties BasicProperties { get; set; }

            public byte[] Body { get; set; }
        }

        private readonly ConcurrentQueue<RabbitQueueItem> _sendQueue = new ConcurrentQueue<RabbitQueueItem>();

        private void ProcessSendQueue()
        {
            IConnection connection = null;
            IModel channel = null;
            var connectionIsValid = false;
            _currentlySending = true;

            while (!_shuttingDown)
            {
                try
                {
                    // Ensure we have a valid sending channel and connection

                    if (connection == null || !connection.IsOpen || !connectionIsValid || !channel.IsOpen)
                    {
                        Logger.Trace("Sender opening new connection...");

                        channel.CloseSafely(Logger);
                        connection.CloseSafely(Logger);

                        
                        connection = RabbitConnectionManager.GetConnection(TypeName);

                        if (connection == null || !connection.IsOpen)
                        {
                            Logger.Warn("Failed to acquire an available Rabbit connection!");
                            Thread.Sleep(1000);
                            continue;
                        }

                        Logger.Trace("Sender connection opened!");

                        connectionIsValid = true;
                        connection.ConnectionShutdown += (sender, args) =>
                        {
                            if (args.ReplyCode != Constants.ReplySuccess)
                            {
                                Logger.Warn("Sending channel connection has closed unexpectedly!", args);
                            }
                            else
                            {
                                Logger.Info("Sending channel connection has closed successfully", args);
                            }

                            connectionIsValid = false;
                        };

                        channel = connection.CreateModel();
                        Logger.Trace("Sender model created!");

                        if (channel == null)
                        {
                            Logger.Warn("Sender failed to create a receiving channel!");
                            connection.CloseSafely(Logger);
                            Thread.Sleep(1000);
                            continue;
                        }
                    }

                    RabbitQueueItem queueItem;

                    if (!_sendQueue.TryDequeue(out queueItem))
                    {
                        Thread.Sleep(100);
                        continue;
                    }

                    try
                    {
                        channel.BasicPublish(queueItem.Exchange, queueItem.RoutingKey, queueItem.BasicProperties, queueItem.Body);
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Failed to publish to queue!");
                    }
                }
                catch (ThreadAbortException) { }
                catch (Exception ex)
                {
                    Logger.Error(ex, "General error in send queue thread");
                }
            }

            Logger.Info("Send queue thread has completed, closing connections");

            channel.CloseSafely(Logger);
            connection.CloseSafely(Logger);
            _currentlySending = false;
        }

        private DateTime _lastLoggedQueueLimit = DateTime.MinValue;
        private readonly TimeSpan _queueLimitLogFrequency = TimeSpan.FromSeconds(30);

        private void TryBasicPublish(string exchange, string routingKey, IBasicProperties basicProperties, byte[] body)
        {
            try
            {
                var currentQueueCount = _sendQueue.Count;
                if (currentQueueCount > 10000)
                {
                    Logger.Error("Send queue is full!");
                    return;
                }

                if (currentQueueCount >= 500)
                {
                    if (DateTime.UtcNow - _lastLoggedQueueLimit >= _queueLimitLogFrequency)
                    {
                        _lastLoggedQueueLimit = DateTime.UtcNow;
                        Logger.Warn("Send queue has reached a high limit: " + _sendQueue.Count);
                    }                    
                }
                
                _sendQueue.Enqueue(new RabbitQueueItem
                {
                    Exchange = exchange,
                    RoutingKey = routingKey,
                    BasicProperties = basicProperties,
                    Body = body
                });
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to enqueue!");
            }
        }

        private void EnqueueNormal(byte[] serializedValue)
        {
            TryBasicPublish(string.Empty, BaseQueueName, _basicPublishProperties, serializedValue);
        }

        public void StartProcessor(Action<T> messageProcessor)
        {
            MessageProcessor = messageProcessor;
            new Thread(ProcessorThread) { IsBackground = true }.Start();
        }

        private QueueingBasicConsumer SetupProcessing(IModel receivingChannel)
        {
            var queueName = BaseQueueName;

            if (_isMachineRouted)
            {
                queueName = BaseQueueName + "-" + Environment.MachineName;

                // Declare a teporary queue to bind to this exchange
                receivingChannel.QueueDeclare(queueName, false, false, false, null);

                // Bind the queue to the exchange
                receivingChannel.QueueBind(queueName, BaseQueueName, Environment.MachineName);
            }

            var consumer = new QueueingBasicConsumer(receivingChannel);
            var prefetch = (ushort)(_maxParallelProcessing * _prefetchMultipler);

            Logger.Trace("Setting up receiving channel...", new { MaxParallel = _maxParallelProcessing, QueueName = queueName, PrefetchCount = prefetch });

            receivingChannel.BasicQos(0, prefetch, false);
            receivingChannel.BasicConsume(queueName, !_isAcknowledged, consumer);

            return consumer;
        }

        private volatile bool _processingConnectionValid = false;

        void ProcessingConnection_ConnectionShutdown(object connection, ShutdownEventArgs reason)
        {
            _processingConnectionValid = false;

            if (reason.ReplyCode != Constants.ReplySuccess)
            {
                Logger.Warn("Processing channel connection has closed unexpectedly!", reason);
            }
            else
            {
                Logger.Trace("Processing channel connection has closed successfully", reason);
            }
        }

        public void ProcessorThread()
        {
            IConnection connection = null;
            IModel receivingChannel = null;
            QueueingBasicConsumer consumer = null;
            _processorStarted = DateTime.UtcNow;
            
            try
            {

                while (!_shuttingDown)
                {
                    BasicDeliverEventArgs delivery = null;

                    try
                    {

                        if (MessageProcessor == null)
                        {
                            Logger.Warn("Message processor method has not been set. Waiting 1 second before re-attempting...");
                            Thread.Sleep(1000);
                            continue;
                        }

                        if (connection == null || !connection.IsOpen || !_processingConnectionValid)
                        {
                            receivingChannel.CloseSafely(Logger);
                            connection.CloseSafely(Logger);

                            Logger.Trace("Processor getting new connection...");
                            connection = RabbitConnectionManager.GetConnection(TypeName);

                            if (connection == null || !connection.IsOpen)
                            {
                                Logger.Warn("Failed to acquire an available Rabbit connection!");
                                Thread.Sleep(1000);
                                continue;
                            }

                            Logger.Trace("Processor connection resumed!");

                            _processingConnectionValid = true;
                            connection.ConnectionShutdown += ProcessingConnection_ConnectionShutdown;
                            receivingChannel = connection.CreateModel();

                            if (receivingChannel == null)
                            {
                                Logger.Warn("Processor failed to create a receiving channel!");
                                connection.Abort();
                                Thread.Sleep(1000);
                                continue;
                            }

                            consumer = SetupProcessing(receivingChannel);

                            if (consumer == null || consumer.Queue == null)
                            {
                                Logger.Warn("Processor failed to setup a consumer!");
                                connection.Abort();
                                Thread.Sleep(1000);
                                continue;
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "General failure in processing thread. Waiting 1 second before attempting to reconnect...");
                        Thread.Sleep(1000);
                    }

                    try
                    {
                        if (_currentlyProcessing >= _maxParallelProcessing)
                        {
                            Thread.Sleep(10);
                            continue;
                        }

                        if (consumer == null)
                        {
                            Logger.Warn("Consumer is null! Aborting connection...");
                            connection.Abort();
                            Thread.Sleep(1000);
                            continue;
                        }

                        if (consumer.Queue == null)
                        {
                            Logger.Warn("Consumer queue is null! Aborting connection...");
                            connection.Abort();
                            Thread.Sleep(1000);
                            continue;
                        }

                        if (!consumer.Queue.Dequeue(1000, out delivery))
                        {
                            continue;
                        }

                        if (delivery == null)
                        {
                            Logger.Warn("Dequeue produced null value.");
                            Thread.Sleep(500); // Wait half a second for the queue to recover
                            continue;
                        }

                        if (_shuttingDown)
                        {
                            if (_isAcknowledged)
                            {
                                Logger.Info("Re-queuing message...");
                                try
                                {
                                    receivingChannel.BasicNack(delivery.DeliveryTag, false, true);
                                }
                                catch (Exception ex)
                                {
                                    Logger.Warn(ex, "Failed to nack delivery");
                                }
                            }

                            Logger.Info("Closing receiving channel...");
                            receivingChannel.CloseSafely(Logger);
                            return;
                        }

                        if (_isAcknowledged)
                        {
                            receivingChannel.BasicAck(delivery.DeliveryTag, false);
                        }

                    }
                    catch (EndOfStreamException ex)
                    {
                        Logger.Error(ex, "EndOfStreamException exception in queue. The connection will be aborted, and re-established.");
                        if (connection != null)
                        {
                            try
                            {
                                connection.Abort();
                            }
                            catch { }
                            connection = null;
                        }
                        continue;
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Unhandled exception in task, while attempting to dequeue.");
                        Thread.Sleep(500); // Wait half a second for the queue to recover
                        continue;
                    }

                    Interlocked.Increment(ref _currentlyProcessing);
                    var timestamp = Stopwatch.StartNew();

                    Task.Factory.StartNew(() => ProcessQueueItem(timestamp, delivery.Body), TaskCreationOptions.LongRunning);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Processing thread produced an unhandled exception!");
            }
            finally
            {
                receivingChannel.CloseSafely(Logger);
                connection.CloseSafely(Logger);
            }
        }

        private readonly double _prefetchMultipler = 1.5;
        private readonly int _maxParallelProcessing = 10;
        private int _currentlyProcessing = 0;
        private long _totalProcessed = 0;
        private DateTime _processorStarted;
        private bool _currentlySending = false;

        private readonly object _processingTimeLock = new object();


        private TimeSpan _totalQueuedTime = TimeSpan.Zero;
        private TimeSpan _bestQueuedTime = TimeSpan.FromDays(1);
        private TimeSpan _worstQueuedTime = TimeSpan.Zero;

        private TimeSpan _totalProcessingTime = TimeSpan.Zero;
        private TimeSpan _bestProcessingTime = TimeSpan.FromDays(1);
        private TimeSpan _worstProcessingTime = TimeSpan.Zero;

        public void OnTaskCompleted(bool successful, TimeSpan processingTimespan, TimeSpan queuedTimespan)
        {
            Interlocked.Decrement(ref _currentlyProcessing);

            if (DateTime.UtcNow - _processorStarted < TimeSpan.FromSeconds(30)) // Skip first 30 seconds of processing, to allow for warmup
            {
                return;
            }

            Interlocked.Increment(ref _totalProcessed);

            if (!successful)
            {
                return;
            }

            lock (_processingTimeLock)
            {

                // Processing Stats
                _totalProcessingTime = _totalProcessingTime.Add(processingTimespan);

                if (processingTimespan < _bestProcessingTime)
                {
                    _bestProcessingTime = processingTimespan;
                }

                if (processingTimespan > _worstProcessingTime)
                {
                    _worstProcessingTime = processingTimespan;
                }


                // Queued Stats
                _totalQueuedTime = _totalQueuedTime.Add(queuedTimespan);

                if (queuedTimespan < _bestQueuedTime)
                {
                    _bestQueuedTime = queuedTimespan;
                }

                if (queuedTimespan > _worstQueuedTime)
                {
                    _worstQueuedTime = queuedTimespan;
                }


            }
        }

        private void ProcessQueueItem(Stopwatch sw, byte[] messageBody)
        {
            var queuedTimespan = TimeSpan.Zero;
            var successful = false;

            try
            {
                var message = Encoding.UTF8.GetString(messageBody);
                var value = JsonConvert.DeserializeObject<T>(message);

                if (value.EnqueuedTimestamp > DateTime.MinValue)
                {
                    queuedTimespan = DateTime.UtcNow - value.EnqueuedTimestamp;
                }

                try
                {
                    MessageProcessor(value);
                    successful = true;
                }
                catch (Exception ex)
                {
                    ++value.Retries;

                    var logMessage = "Failed to process message (Attempt " + value.Retries + ")";

                    if (value.Retries < RetryLimit)
                    {
                        Logger.Warn(ex, logMessage);
                        value.Enqueue();                       
                    }
                    else
                    {
                        Logger.Error(ex, logMessage);
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, string.Format("[{0}] Unhandled exception in task.", typeof(T).Name));
            }
            finally
            {
                OnTaskCompleted(successful, sw.Elapsed, queuedTimespan);
            }
        }

        public void ConfigureShovel(string apiUrl, string[] sourceHostNames, string[] destinationHostNames, string username, string password, int port, string shovelName, string localQueueName, bool isMachineRouted, string remoteName)
        {

            var amqpFormat = "amqp://{0}:{1}@{2}:{3}";

            var sourceUriList = sourceHostNames.Select(p => string.Format(amqpFormat, username, password, p, port)).ToArray();
            var destinationUriList = destinationHostNames.Select(p => string.Format(amqpFormat, username, password, p, port)).ToArray();

            var shovelRequest = new ShovelRequest
            {
                SourceUri = sourceUriList,
                DestinationUri = destinationUriList,
            };

            if (isMachineRouted)
            {
                shovelRequest.DestinationExchangeName = remoteName;
                shovelRequest.SourceExchangeName = localQueueName;
                shovelRequest.SourceExchangeKey = "*";
            }
            else
            {
                shovelRequest.DestinationQueueName = remoteName;
                shovelRequest.SourceQueueName = localQueueName;
            }


            var requestVaue = new
            {
                value = shovelRequest
            };

            string requestString = null;
            try
            {
                requestString = JsonConvert.SerializeObject(requestVaue);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to serialize shovel request!");
            }

            var fullApiUrl = apiUrl + "/api/parameters/shovel/%2f/" + shovelName;

            using (var webClient = new WebClient())
            {
                webClient.Credentials = new NetworkCredential(username, password);
                webClient.Headers.Set("content-type", "application/json");

                try
                {
                    webClient.UploadString(fullApiUrl, "PUT", requestString);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to create shovel!");
                }
            }
        }

        public class ShovelRequest
        {
            [JsonProperty(PropertyName = "src-uri")]
            public string[] SourceUri
            {
                get;
                set;
            }

            [JsonProperty(PropertyName = "src-exchange", NullValueHandling = NullValueHandling.Ignore)]
            public string SourceExchangeName
            {
                get;
                set;

            }

            [JsonProperty(PropertyName = "src-exchange-key", NullValueHandling = NullValueHandling.Ignore)]
            public string SourceExchangeKey
            {
                get;
                set;

            }
            [JsonProperty(PropertyName = "src-queue", NullValueHandling = NullValueHandling.Ignore)]
            public string SourceQueueName
            {
                get;
                set;
            }

            [JsonProperty(PropertyName = "dest-uri")]
            public string[] DestinationUri
            {
                get;
                set;
            }

            [JsonProperty(PropertyName = "dest-exchange", NullValueHandling = NullValueHandling.Ignore)]
            public string DestinationExchangeName
            {
                get;
                set;
            }

            [JsonProperty(PropertyName = "dest-queue", NullValueHandling = NullValueHandling.Ignore)]
            public string DestinationQueueName
            {
                get;
                set;
            }
        }

    }
}
