﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using Curse.CloudSearch.Tests.SearchTypes;
using Nest;

namespace Curse.CloudSearch.Tests
{
    static class LoadTest
    {
        private const int NumberOfSearchTerms = 5000;
        private const int NumberOfIndexedItems = 50000;

        private class ElapsedSearchTime
        {
            public long TermTimeStopwatch;
            public long TermTimeReported;
            public long MatchTimeStopwatch;
            public long MatchTimeReported;
        }

        private static readonly Dictionary<Type, List<string>> SearchTerms = new Dictionary<Type, List<string>>
        {
            {typeof (CharacterSearchType), new List<string>()},
            {typeof (EmailSearchType), new List<string>()},
            {typeof (UserSearchType), new List<string>()}
        };

        private static Dictionary<Type, ElapsedSearchTime> searchTimes = new Dictionary<Type, ElapsedSearchTime>
        {
            {typeof (CharacterSearchType), new ElapsedSearchTime()},
            {typeof (EmailSearchType), new ElapsedSearchTime()},
            {typeof (UserSearchType), new ElapsedSearchTime()}
        };

        private static readonly Random Random = new Random();
        private static List<string> _cachedCharacters = new List<string>();
        private static List<string> _cachedEmails = new List<string>();
        private static List<string> _cachedUsers = new List<string>();
 
        public static void Run()
        {
            Console.Clear();
            Console.Title = "ElasticSearch Load Test";
            Console.WriteLine("How many times would you like to run the test?");
            int numRuns;
            var response = Console.ReadLine();
            if (!int.TryParse(response, out numRuns))
            {
                Console.WriteLine("The number entered was not valid, running the test once.");
                numRuns = 1;
            }

            Console.WriteLine("\nPopulating Character Index");
            CharacterSearchTypeUtilities.PopulateIndex(NumberOfIndexedItems, Random, ref _cachedCharacters);
            var characterClient = CharacterSearchType.GetClient();

            Console.WriteLine("Populating Email Index");
            EmailSearchTypeUtilities.PopulateIndex(NumberOfIndexedItems, Random, ref _cachedEmails);
            var emailClient = EmailSearchType.GetClient();
            
            Console.WriteLine("Populating User Index");
            UserSearchTypeUtilities.PopulateIndex(NumberOfIndexedItems, Random, ref _cachedUsers);
            var userClient = UserSearchType.GetClient();

            WriteResults("---------------------------------------------------------------------------");
            WriteResults(string.Format("Test Results {0} - {1} Runs", DateTime.UtcNow.ToString("u"), numRuns));

            var originalTitle = Console.Title;
            for (int i = 0; i < numRuns; ++i)
            {
                Console.Clear();
                Console.WriteLine("Test Run {0} of {1}", i + 1, numRuns);

                Console.Title = string.Format("{0} ({1}/{2})", originalTitle, i + 1, numRuns);

                RandomizeSearchTerms();

                TestSearches(q => characterClient.SearchCharacters(q, CharacterSearchTypeUtilities.TestRegions, CharacterSearchTypeUtilities.TestServerNames, 100),
                    q => characterClient.SearchCharacters(q, CharacterSearchTypeUtilities.TestRegions, CharacterSearchTypeUtilities.TestServerNames, 100, true));

                TestSearches(q => userClient.SearchUsers(q, 100),
                    q => userClient.SearchUsers(q, 100, true));

                TestSearches(q => emailClient.SearchEmails(q), 
                    q => emailClient.SearchEmails(q, true));

            }
            Console.Title = string.Format("{0} (Complete)", originalTitle);

            WriteResults("Reported Times");
            WriteResults(string.Format("{0,-25}{1,-15}{2,-15}{3,-15}", 
                "Data Type", "Term Avg", "Match Avg", "Net Change"));
            WriteTestResults<CharacterSearchType>(numRuns);
            WriteTestResults<EmailSearchType>(numRuns);
            WriteTestResults<UserSearchType>(numRuns);

            WriteResults("---");
            WriteResults("Stopwatch Times");
            WriteResults(string.Format("{0,-25}{1,-15}{2,-15}{3,-15}",
                "Data Type", "Term Avg", "Match Avg", "Net Change"));
            WriteTestResults<CharacterSearchType>(numRuns, false);
            WriteTestResults<EmailSearchType>(numRuns, false);
            WriteTestResults<UserSearchType>(numRuns, false);
            
            WriteResults("---------------------------------------------------------------------------");

            Console.WriteLine("Press enter to exit.");
            Console.ReadLine();
        }


        private static void RandomizeSearchTerms()
        {
            SearchTerms[typeof(CharacterSearchType)] = new List<string>();
            SearchTerms[typeof(EmailSearchType)] = new List<string>();
            SearchTerms[typeof(UserSearchType)] = new List<string>();

            for (int i = 0; i < NumberOfSearchTerms; ++i)
            {
                var index = Random.Next(0, NumberOfIndexedItems-1);
                AddSearchTerm<CharacterSearchType>(_cachedCharacters[index]);
                AddSearchTerm<EmailSearchType>(_cachedEmails[index], false);
                AddSearchTerm<UserSearchType>(_cachedUsers[index]);
            }
        }
        private static void AddSearchTerm<T>(string originalTerm, bool randomized = true)
        {
            int substringLength = 1;
            if (originalTerm.Length > 1)
            {
                substringLength = randomized ? Random.Next(2, originalTerm.Length) : originalTerm.Length;
            }
            SearchTerms[typeof(T)].Add(originalTerm.Substring(0, substringLength));
        }

        private static void TestSearches<T>(Func<string, ISearchResponse<T>> termAction, Func<string, ISearchResponse<T>> matchAction) where T : class
        {
            var searchType = typeof (T);
            Console.WriteLine("Testing {0} Searching...", searchType.Name);

            var termStopwatch = new Stopwatch();
            var matchStopwatch = new Stopwatch();

            foreach (var searchTerm in SearchTerms[searchType])
            {
                termStopwatch.Start();
                var termResults = termAction(searchTerm);
                termStopwatch.Stop();

                matchStopwatch.Start();
                var matchResults = matchAction(searchTerm);
                matchStopwatch.Stop();

                searchTimes[searchType].TermTimeReported += termResults.ElapsedMilliseconds;
                searchTimes[searchType].TermTimeStopwatch += termStopwatch.ElapsedMilliseconds;

                searchTimes[searchType].MatchTimeReported += matchResults.ElapsedMilliseconds;
                searchTimes[searchType].MatchTimeStopwatch += matchStopwatch.ElapsedMilliseconds;

                var missingTerm = matchResults.Documents.FirstOrDefault(d => !termResults.Documents.Contains(d));
                var missingMatch = termResults.Documents.FirstOrDefault(d => !matchResults.Documents.Contains(d));
                if (!(missingMatch == null && missingTerm == null))
                {
                    // Both can be null in the case where the search term is a single character, not really an error in code
                    if (missingTerm == null)
                    {
                        WriteResults(string.Format("Term found at least one document Match did not find for {0}.", searchTerm));
                    }
                    else if (missingMatch == null)
                    {
                        WriteResults(string.Format("Match found at least one document Term did not find for {0}.", searchTerm));
                    }
                }
            }

        }

        private static void WriteTestResults<T>(int numRuns, bool useReportedTime=true)
        {
            var searchType = typeof (T);

            string termMillis;
            string matchMillis;
            string pctDiff;

            if (useReportedTime)
            {
                termMillis = string.Format("{0:D}", (searchTimes[searchType].TermTimeReported / numRuns));
                matchMillis = string.Format("{0:D}", (searchTimes[searchType].MatchTimeReported / numRuns));
                pctDiff = string.Format("{0:P2}", ((searchTimes[searchType].MatchTimeReported - searchTimes[searchType].TermTimeReported) / (double)searchTimes[searchType].TermTimeReported));
            }
            else
            {
                termMillis = string.Format("{0:D}", (searchTimes[searchType].TermTimeStopwatch / numRuns));
                matchMillis = string.Format("{0:D}", (searchTimes[searchType].MatchTimeStopwatch / numRuns));
                pctDiff = string.Format("{0:P2}", ((searchTimes[searchType].MatchTimeStopwatch - searchTimes[searchType].TermTimeStopwatch) / (double)searchTimes[searchType].TermTimeStopwatch));
            }

            var results = string.Format("{0,-25}{1,-15}{2,-15}{3,-15}",
                searchType.Name.Substring(0, Math.Min(25, searchType.Name.Length)), termMillis, matchMillis, pctDiff);

            WriteResults(results);
        }

        private static void WriteResults(string line)
        {
            Console.WriteLine(line);
            if (!Directory.Exists(@"C:\temp"))
            {
                Directory.CreateDirectory(@"C:\temp");
            }

            using (var stream = File.AppendText(@"C:\temp\results.txt"))
            {
                stream.WriteLineAsync(line);
            }
        }
    }
}
