﻿using System;
using System.Collections.Generic;
using Curse.CloudSearch.Tests.SearchTypes;
using Nest;

namespace Curse.CloudSearch.Tests
{
    static class SearchExtensions
    {
        public static ISearchResponse<CharacterSearchType> SearchCharacters(this ElasticClient client, string characterName,
            string[] regions, string[] serverNames, int rowLimit, bool useMatchSearch = false)
        {
            // Get the current user's character search hints            
            var shouldFilters = new List<Func<QueryDescriptor<CharacterSearchType>, QueryContainer>>();

            foreach (var region in regions)
            {
                shouldFilters.Add(qd => qd.Term(t => t.ServerRegion, region, 2.0));
            }

            shouldFilters.Add(qd => qd.Terms(t => t.ServerName, serverNames));

            var characterClient = CharacterSearchType.GetClient();

            ISearchResponse<CharacterSearchType> results;

            if (useMatchSearch)
            {
                results = characterClient.Search<CharacterSearchType>(s => s
                    .Query(q => q
                        .Bool(b => b
                            .Must(qd => qd
                                .Match(m => m
                                    .Query(characterName)
                                    .OnField(f => f.CharacterName.Suffix("autocomplete"))))
                            .Should(shouldFilters.ToArray())
                            .MinimumShouldMatch(0)
                            .Boost(2.0)))
                    .Take(rowLimit)
                    );
            }
            else
            {
                results = characterClient.Search<CharacterSearchType>(s => s
                    .Query(q => q
                        .Bool(b => b
                            .Must(
                                qd => qd.Term(t => t.CharacterName.Suffix("autocomplete"), characterName.ToLower())
                            )
                            .Should(shouldFilters.ToArray())
                            .MinimumShouldMatch(0)
                            .Boost(2.0)
                        )
                    )
                    .Take(rowLimit)
                    );
            }

            return results;
        }


        public static ISearchResponse<EmailSearchType> SearchEmails(this ElasticClient client, string emailAddress, bool useMatchSearch=false)
        {
            var emailSearchClient = EmailSearchType.GetClient();

            ISearchResponse<EmailSearchType> results;

            if (useMatchSearch)
            {
                results = emailSearchClient.Search<EmailSearchType>(s => s
                    .Query(q => q
                        .Match(m => m
                            .Query(emailAddress)
                            .OnField(f => f.EmailAddress.Suffix("lowercase")))));
            }
            else
            {
                results = emailSearchClient.Search<EmailSearchType>(s => s
                    .Query(q => q
                        .Bool(b => b
                            .Must(qd => qd
                                .Term(t => t.EmailAddress.Suffix("lowercase"), emailAddress.ToLower())))));
            }

            return results;
        }

        public static ISearchResponse<UserSearchType> SearchUsers(this ElasticClient client, string userName, int rowLimit, bool useMatchSearch=false)
        {
            // User Results
            var usernameSearchClient = UserSearchType.GetClient();

            ISearchResponse<UserSearchType> results;
            if (useMatchSearch)
            {
                results = usernameSearchClient.Search<UserSearchType>(s => s
                    .Query(q => q
                        .Match(m => m
                            .Query(userName)
                            .OnField(f => f.Username.Suffix("autocomplete"))))
                    .Take(rowLimit));
            }
            else
            {
                results = usernameSearchClient.Search<UserSearchType>(s => s
                    .Query(q => q
                        .Term(t => t.Username.Suffix("autocomplete"), userName.ToLower()))
                    .Take(rowLimit));
            }

            return results;
        }

    }
}
