﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using System.Threading;
using Nest;

namespace Curse.CloudSearch.Tests.SearchTypes
{
    class CharacterSearchTypeUtilities
    {
        public static readonly string[] TestRegions = {"NA", "EU", "OCE", "SA"};

        public static readonly string[] TestServerNames = 
        {
            "Wyrmrest Accord", "Emerald Dream", "Moon Guard", "Eldre'Thalas", "Winterhoof", "Kel'Thuzad", "Mannoroth",
            "Blackwater Raiders", "Feathermoon", "Shadow Council"
        };

        private const int ExactMatchRowLimit = 10;
        private const int AutocompleteRowLimit = 100;

        public static void PopulateIndex(int rowLimit, Random random, bool testSearching = false)
        {
            var throwaway = new List<string>();
            PopulateIndex(rowLimit, random, ref throwaway, testSearching);
        }

        public static void PopulateIndex(int rowLimt, Random random, ref List<string> possibleSearches, bool testSearching = false)
        {
            CharacterSearchType.DeleteIndex();
            CharacterSearchType.SetupIndex();

            var client = CharacterSearchType.GetClient();
            var list = new List<CharacterSearchType>();

            using (var conn = new SqlConnection("Data Source=sql02-dev.curse.us;Initial Catalog=AuthService;Integrated Security=true;MultipleActiveResultSets=True;Max Pool Size=1024;MultiSubnetFailover=True;"))
            {
                conn.Open();
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "select top " + rowLimt + " _uid, _username from users order by _uid";
                    using (var reader = cmd.ExecuteReader())
                    {
                        var i = 0;
                        while (reader.Read())
                        {
                            var friendCount = random.Next(0, 120);

                            var model = new CharacterSearchType
                            {
                                CharacterName = reader.GetString(1),
                                UserID = reader.GetInt32(0),
                                ServerName = TestServerNames[random.Next(0, TestServerNames.Length - 1)],
                                ServerRegion = TestRegions[random.Next(0, TestRegions.Length - 1)],
                                FriendCount = friendCount

                            };

                            model.GenerateUniqueKey();

                            list.Add(model);
                            possibleSearches.Add(model.CharacterName);

                            if (++i % 10000 == 0)
                            {
                                Console.WriteLine("Finished " + i.ToString("###,##0"));
                                var result = client.IndexMany(list);

                                if (!result.IsValid)
                                {
                                    Console.WriteLine("Index operation failed! Press any key to continue...");
                                    Console.ReadKey();
                                }

                                client.Refresh();

                                if (testSearching)
                                {
                                    Console.WriteLine("Testing Search Index ...");
                                    var searchCounter = 0;
                                    foreach (var user in list)
                                    {
                                        if (++searchCounter % 100 == 0)
                                        {
                                            Console.Title = (searchCounter / (double)list.Count).ToString("P0") + " completed...";
                                        }
                                        ValidateItem(user);
                                    }
                                }


                                list.Clear();
                            }
                        }
                    }
                }

                if (list.Any())
                {
                    client.IndexMany(list);
                }
            }

        }

        private static void ValidateItem(CharacterSearchType item)
        {
            if (item.CharacterName.Length < 2)
            {
                Console.WriteLine("Skipping character. It is not really very valid: " + item.CharacterName);
                return;
            }

            var characterClient = CharacterSearchType.GetClient();
            var regions = new[] {item.ServerRegion};
            var servers = new[] {item.ServerName};

            if (characterClient.SearchCharacters(item.CharacterName, regions, servers, ExactMatchRowLimit)
                .Documents.All(u => u.UserID != item.UserID))
            {
                Console.WriteLine("Full name search failed for: " + item.CharacterName);
            }

            if (item.CharacterName.Length < 3)
            {
                return;
            }

            var partialSearch = item.CharacterName.Substring(0, item.CharacterName.Length - 1);
            if (characterClient.SearchCharacters(partialSearch, regions, servers, AutocompleteRowLimit)
                .Documents.All(u => u.UserID != item.UserID))
            {
                Console.WriteLine("Partial name search failed for user '" +
                                  item.CharacterName + "' using search term '" +
                                  partialSearch + "'");
            }
        }


    }
}
