﻿using System.Linq;
using Curse.Logging;
using Nest;
using System;
using System.Collections.Generic;
using System.Reflection;
using Elasticsearch.Net.ConnectionPool;


namespace Curse.CloudSearch
{

    public abstract class CloudSearchManager<T> where T : class, new()
    {
        public const string AutocompleteSuffix = "autocomplete";
        public const string SortSuffix = "sort";
        public const string AutocompleteAnalyzerName = "autocomplete_analyzer";
        public const string LowercaseAnalyzerName = "lowercase_analyzer";

        private static readonly ConnectionSettings DefaultConnectionSettings;
        public static readonly string IndexName;
        public static readonly string IndexTypeName;
        public static readonly string IndexWildcard;
        public static readonly string TemplateName;

        static CloudSearchManager()
        {
            var attribute = typeof (T).GetCustomAttribute<CloudSearchModelAttribute>();
            if (attribute == null || attribute.UseDefaultIndex)
            {
                IndexName = typeof (T).FullName.Replace('.', '_').ToLowerInvariant();
            }

            if (attribute != null)
            {
                IndexTypeName = attribute.IndexTypeName;
                IndexWildcard = IndexName ?? (attribute.UseAlias ? "alias-" + IndexTypeName + "-*" : IndexTypeName + "-*");
                IndexName = IndexName ?? IndexTypeName + "-index";
                TemplateName = IndexTypeName + "-template";
            }

            DefaultConnectionSettings = GetDefaultConnectionSettings(SearchConfiguration.LocalConfigurations);

            if (IndexName != null && (attribute == null || attribute.AutoCreateIndex))
            {
                try
                {
                    SetupIndex();
                }
                catch
                {

                }
            }

        }

        public static ConnectionSettings GetDefaultConnectionSettings(SearchConfiguration[] regionalConfigs)
        {
            if (regionalConfigs == null || !regionalConfigs.Any() || !regionalConfigs.All(p => p.Types != null && p.Types.Any()))
            {
                throw new ArgumentException("Must have at least one config, and at least one type map.", "regionalConfigs");
            }
            var typeName = typeof(T).Name;
            var bestConfig = regionalConfigs.FirstOrDefault(p => p.Types.Any(t => string.Equals(t, typeName, StringComparison.InvariantCultureIgnoreCase)));
            if (bestConfig == null)
            {
                throw new InvalidOperationException("No search configuration found for type name: " + typeName);
            }

            var hosts = bestConfig.Hosts.Select(p => new Uri(p, UriKind.Absolute));
            var connectionPool = new SniffingConnectionPool(hosts, true);

            var connectionSettings = new ConnectionSettings(connectionPool);

            if (IndexName != null)
            {
                connectionSettings.SetDefaultIndex(IndexName);
            }

            return connectionSettings;
        }

        public static ElasticClient GetClient()
        {
            return new ElasticClient(DefaultConnectionSettings);
        }

        public static void DeleteIndex()
        {
            DeleteIndex(GetClient());
        }

        public static bool DeleteIndex(ElasticClient client)
        {
            var response = client.DeleteIndex(d => d.Index(IndexName));
            return response.ConnectionStatus.Success;
        }

        public static void SetupIndex()
        {
            SetupIndex(GetClient());
        }

        public static bool SetupIndex(ElasticClient client)
        {
            var indexName = IndexName;
            
            string idFieldName = null;
            var properties = typeof(T).GetProperties();
            var hasAutocompleteIndex = false;
            var hasCaseInsensitiveIndex = false;

            foreach (var property in properties)
            {
                var cloudSearchAttribute = property.GetCustomAttribute<CloudSearchPropertyAttribute>();
                if (cloudSearchAttribute == null)
                {
                    continue;
                }
                if (cloudSearchAttribute.SearchType == CloudSearchType.AutoComplete)
                {
                    hasAutocompleteIndex = true;
                }
                
                if(cloudSearchAttribute.SearchType == CloudSearchType.CaseInsensitiveMatch)
                {
                    hasCaseInsensitiveIndex = true;
                }

                if (cloudSearchAttribute.IsIdentifier)
                {
                    idFieldName = property.Name.ToCamelCase();
                }
            }

            Func<AnalysisDescriptor, AnalysisDescriptor> getAnalysis = (baseAnalysis) =>
            {
                if (hasAutocompleteIndex)
                {
                    baseAnalysis = baseAnalysis
                        .Analyzers(a => a
                            .Add("autocomplete",
                                    new Nest.CustomAnalyzer()
                                    {
                                        Tokenizer = "edgeNGram",
                                        Filter = new [] { "lowercase" }
                                    }
                            )
                        )
                        .Tokenizers(t => t
                            .Add(
                                "edgeNGram",
                                new Nest.EdgeNGramTokenizer()
                                {
                                    MinGram = 2,
                                    MaxGram = 128,
                                }
                            )
                        );
                }

                if (hasCaseInsensitiveIndex || hasAutocompleteIndex)
                {
                    baseAnalysis = baseAnalysis
                        .Analyzers(a => a
                            .Add("string_lowercase",
                                    new Nest.CustomAnalyzer()
                                    {
                                        Tokenizer = "keyword",
                                        Filter = new[] { "lowercase" }
                                    }
                            )
                        );
                        
                }
                                
                return baseAnalysis;
            };

            Func<PropertiesDescriptor<T>, PropertiesDescriptor<T>> getMapping = (a) =>
            {
                foreach (var prop in properties)
                {
                    var attrib = prop.GetCustomAttribute<CloudSearchPropertyAttribute>();
                    if (attrib == null)
                    {
                        continue;
                    }

                    switch (attrib.SearchType)
                    {
                        case CloudSearchType.CaseInsensitiveMatch:
                            var cname = prop.Name.ToCamelCase();
                            a = a.MultiField(p => p
                                .Name(cname)
                                .Fields(tf => tf
                                    .String(s => s
                                        .Name(cname)
                                        .Index(Nest.FieldIndexOption.NotAnalyzed)
                                    )
                                    .String(s => s
                                        .Name(cname + ".lowercase")
                                        .Index(Nest.FieldIndexOption.Analyzed)
                                        .Analyzer("string_lowercase")
                                    )
                                )
                            );
                            break;

                        case CloudSearchType.ExactMatchOnly:
                            a = a.String(s => s
                                    .Name(prop.Name.ToCamelCase())
                                    .Index(Nest.FieldIndexOption.NotAnalyzed)
                            );
                            break;

                        case CloudSearchType.AutoComplete:
                            var plainName = prop.Name.ToCamelCase();
                            a = a.MultiField(p => p
                                .Name(plainName)
                                .Fields(tf => tf
                                    .String(s => s
                                        .Name(plainName)
                                        .Index(Nest.FieldIndexOption.NotAnalyzed)
                                    )
                                    .String(s => s
                                        .Name(plainName + ".autocomplete")
                                        .Index(Nest.FieldIndexOption.Analyzed)
                                        .IndexAnalyzer("autocomplete")
                                        .SearchAnalyzer("string_lowercase")
                                    )
                                )
                            );
                            break;
                    }

                }
                return a;
            };

            var createResult = client.CreateIndex(indexName, index => index
                .Analysis(analysis => getAnalysis(analysis))
                .AddMapping<T>(tmd => tmd
                    .IdField(f => f.Path(idFieldName))
                    .Properties(props => getMapping(props))
                )
            );
            return createResult.ConnectionStatus.Success;
        }

        public void Index()
        {
            var client = GetClient();            
            client.Index<T>(this as T);
        }

        public static void IndexMany(List<T> models, int requiredCount = 0)
        {
            if (models.Count == 0)
            {
                return;
            }

            if(requiredCount > 0 && models.Count < requiredCount)
            {
                return;
            }

            var client = GetClient();
            var resp = client.IndexMany(models).ConnectionStatus;

            if (!resp.Success)
            {
                Logger.Warn("Failed to index " + typeof(T).Name + " search models!", resp);
            }

            models.Clear();            
        }

        public static void BulkIndex(List<T> models)
        {
            if (models.Count == 0)
            {
                return;
            }

            var client = GetClient();
            var modelsToSend = new List<T>();
            foreach (var model in models)
            {
                modelsToSend.Add(model);

                if (modelsToSend.Count == 1000)
                {
                    DoBulkIndex(client, modelsToSend);
                    modelsToSend.Clear();
                }
            }

            if (modelsToSend.Count > 0)
            {
                DoBulkIndex(client, modelsToSend);
            }
        }

        private static void DoBulkIndex(ElasticClient client, IEnumerable<T> modelsToSend)
        {
            var bulkRequest = new BulkRequest
            {
                Index = IndexName,
                Operations = modelsToSend.Select(m => (IBulkOperation)new BulkIndexOperation<T>(m)).ToArray(),
            };
            var resp = client.Bulk(bulkRequest);
            if (resp.Errors)
            {
                Logger.Warn("Failed to bulk index " + typeof(T).Name + " search models!", resp);
            }
        }

        public static void Delete(string id)
        {
            if (string.IsNullOrEmpty(id))
            {
                return;
            }
            
            var client = GetClient();
            var resp = client.Delete<T>(id).ConnectionStatus;
        }

        public static void DeleteMany(List<string> ids, int requiredCount = 0)
        {
            if (ids.Count == 0)
            {
                return;
            }

            if (requiredCount > 0 && ids.Count < requiredCount)
            {
                return;
            }

            var client = GetClient();
            var operations = new List<IBulkOperation>();
            foreach(var id in ids)
            {
                operations.Add(new BulkDeleteOperation<T>(id));
            }
            
            var resp = client.Bulk(new BulkRequest(){ Operations = operations });
            ids.Clear();
        }

        public static void BulkDelete(List<string> ids)
        {
            var idsToChange = new List<string>();
            var client = GetClient();
            foreach (var id in ids)
            {
                idsToChange.Add(id);

                if (idsToChange.Count == 1000)
                {
                    DoBulkDelete(client, idsToChange);
                    idsToChange.Clear();
                }
            }

            if (idsToChange.Count > 0)
            {
                DoBulkDelete(client, idsToChange);
            }
        }

        private static void DoBulkDelete(ElasticClient client, IEnumerable<string> ids)
        {
            var bulkRequest = new BulkRequest
            {
                Index = IndexName,
                Operations = ids.Select(id => (IBulkOperation) new BulkDeleteOperation<T>(id)).ToArray()
            };
            var resp = client.Bulk(bulkRequest);
            if (resp.Errors)
            {
                Logger.Debug("Failed to delete " + typeof (T).Name + " search models!", resp);
            }
        }

        #region Template Mappings

        public bool SetupTemplate()
        {
            return SetupTemplate(GetClient());
        }

        public bool SetupTemplate(ElasticClient client)
        {
            var managerAttribute = typeof(T).GetCustomAttribute<CloudSearchModelAttribute>();
            if (managerAttribute == null)
            {
                throw new InvalidOperationException("Cannot create a template for a manager without a CloudSearchModelAttribute");
            }

            if (managerAttribute.AutoCreateIndex || managerAttribute.UseDefaultIndex)
            {
                throw new InvalidOperationException("Cannot create a template when autocreate or default index is selected");
            }

            if (string.IsNullOrWhiteSpace(IndexTypeName))
            {
                throw new InvalidOperationException("Cannot create a template when an IndexTypeName is not specified");
            }

            const string autocompleteTokenizerName = "autocomplete_tokenizer";
            var response = client.PutTemplate(TemplateName, t =>
            {
                var resp = t.Template(IndexTypeName+"-*")
                    .Settings(s => s
                        .Add("analysis", new AnalysisSettings
                        {
                            Analyzers = new Dictionary<string, AnalyzerBase>
                            {
                                {AutocompleteAnalyzerName, new CustomAnalyzer {Tokenizer = autocompleteTokenizerName, Filter = new[] {"lowercase"}}},
                                {LowercaseAnalyzerName, new CustomAnalyzer {Tokenizer = "keyword", Filter = new[] {"lowercase"}}}
                            },
                            Tokenizers = new Dictionary<string, TokenizerBase>
                            {
                                {
                                    autocompleteTokenizerName, new EdgeNGramTokenizer
                                    {
                                        MinGram = 1,
                                        MaxGram = 64,
                                    }
                                }
                            }
                        })
                    )
                    .AddMapping<T>(m => m
                        .AllField(af => af.Enabled(false))
                        .SourceField(sf => sf.Enabled())
                        .MapFromAttributes()
                        .Properties(props => CustomPropertyMapping(props.CreateAutocompleteFields())));

                if (managerAttribute.UseAlias)
                {
                    resp = resp.AddAlias("alias-{index}");
                }
                return resp;
            });

            return response.ConnectionStatus.Success;
        }

        protected virtual PropertiesDescriptor<T> CustomPropertyMapping(PropertiesDescriptor<T> descriptor)
        {
            return descriptor;
        }

        public bool DeleteTemplate()
        {
            return DeleteTemplate(GetClient());
        }

        public bool DeleteTemplate(ElasticClient client)
        {
            var response = client.DeleteTemplate(TemplateName);
            return response.ConnectionStatus.Success;
        }


        #endregion
    }
}
