﻿using System;
using System.Web;
using Curse.CloudServices.Models;
using Curse.ServiceEncryption;

namespace Curse.CloudServices.Authentication.Web
{
    public static class AuthenticationContext
    {
        public static AuthenticationToken Current
        {
            get
            {
                if (HttpContext.Current == null)
                {
                    return new AuthenticationToken {IsAnonymous = true};
                }

                var token = HttpContext.Current.Items[AuthenticationToken.HeaderName] as AuthenticationToken;
                if (token == null)
                {
                    var tokenValue = HttpContext.Current.Request.Headers[AuthenticationToken.HttpHeaderName];

                    if (string.IsNullOrEmpty(tokenValue))
                    {
                        var cookie = HttpContext.Current.Request.Cookies.Get(AuthenticationToken.HttpCookieName);
                        if (cookie != null)
                        {
                            tokenValue = HttpUtility.UrlDecode(cookie.Value);
                        }
                    }

                    token = ParseToken(tokenValue);
                    HttpContext.Current.Items[AuthenticationToken.HeaderName] = token;
                }

                return token;
            }
        }

        private static AuthenticationToken ParseToken(string tokenString)
        {
            var token = new AuthenticationToken {IsAnonymous = true};

            if (tokenString == null)
            {
                return token;
            }

            var encryptedToken = EncryptionToken.FromValue(tokenString);
            if (!encryptedToken.IsValid)
            {
                return token;
            }

            var timestamp = encryptedToken.GetDateTime("Timestamp");

            if (DateTime.UtcNow - timestamp > TimeSpan.FromDays(60))
            {
                return token;
            }

            return new AuthenticationToken
            {
                Token = tokenString,
                Email = encryptedToken.GetValue("Email"),
                UserID = encryptedToken.GetInteger("UserID"),
                Username = encryptedToken.GetValue("Username")
            };
        }

        public static bool IsTokenValid(string tokenString)
        {
            var encryptedToken = EncryptionToken.FromValue(tokenString);
            return encryptedToken.IsValid;
        }
    }
}
