﻿using System;
using System.Collections.Generic;
using System.Linq;

namespace Curse.CloudServices.Authentication.Web
{
    public class DomainWhitelist
    {
        private readonly HashSet<string> _whiteList;

        public DomainWhitelist(IEnumerable<string> domainNames)
        {
            _whiteList = new HashSet<string>(domainNames);
        }

        private static readonly string[] SpecialSuffixes = { "com.cn", ".co.uk" };

        public static string GetDomainNameFromHost(Uri uri)
        {
            var host = uri.Host.ToLowerInvariant();
            var splitCount = 2;
            foreach (var suffix in SpecialSuffixes)
            {
                if (host.EndsWith(suffix))
                {
                    splitCount = 3;
                    break;
                }
            }

            // Get the last X parts of the host, depending on the split count
            var splitHostName = host.Split('.');
            return string.Join(".", splitHostName.Skip(splitHostName.Length - splitCount).Take(splitCount).ToArray());
        }

        public static string GetSanitizedUrl(Uri uri, string scheme)
        {
            var uriBuilder = new UriBuilder(uri);
            uriBuilder.Port = -1;
            uriBuilder.Scheme = scheme;
            uriBuilder.Host = uri.Host.ToLowerInvariant();            
            return uriBuilder.ToString().Replace(" ", "-"); // Replace any spaces with hyphens
        }

        public bool IsWhiteListed(string url, bool allowLocalhost = false)
        {
            if (string.IsNullOrWhiteSpace(url))
            {
                return false;
            }
          

            Uri uri;
            
            try
            {
                uri = new Uri(url, UriKind.Absolute);
            }
            catch
            {
                return false;
            }

            // Special case for development
            if (allowLocalhost && uri.Host.Equals("localhost", StringComparison.InvariantCultureIgnoreCase))
            {
                return true;
            }

            var domainName = GetDomainNameFromHost(uri);

            if (!_whiteList.Contains(domainName))
            {
                return false;
            }

            return true;
        }

        public bool IsWhiteListed(string url, string scheme, out string sanitizedUrl)
        {
            if (string.IsNullOrWhiteSpace(url))
            {
                sanitizedUrl = null;
                return false;
            }

            Uri uri;
            sanitizedUrl = null;
            
            try
            {
                uri = new Uri(url, UriKind.Absolute);
            }
            catch
            {
                return false;
            }


            var domainName = GetDomainNameFromHost(uri);

            if (!_whiteList.Contains(domainName))
            {
                return false;
            }

            sanitizedUrl = GetSanitizedUrl(uri, scheme);

            return true;

        }
    }

    
}
