﻿using System;
using System.ServiceModel.Web;
using Curse.CloudServices.Models;
using System.ServiceModel;
using Curse.Logging;
using Newtonsoft.Json;

namespace Curse.CloudServices.Authentication
{
    public class AuthenticationContext
    {
        public static AuthenticationToken Current
        {
            get
            {
                try
                {
                    if (UseWebOperationContext)
                    {
                        return GetForWebOperationContext(WebOperationContext.Current);
                    }
                    
                    return GetForOperationContext(OperationContext.Current);                     

                }
                catch (Exception ex)
                {
                    Logger.Warn(ex, "Failed to get authentication token.");
                    return new AuthenticationToken { IsAnonymous = true };
                }      
            }
        }

        private static AuthenticationToken GetForOperationContext(OperationContext operationContext)
        {
            var extension = operationContext.Extensions.Find<AuthenticationExtension>();

            if (extension == null)
            {
                if (operationContext.IncomingMessageHeaders.FindHeader(AuthenticationToken.HeaderName, AuthenticationToken.HeaderNamespace) != -1) // Soap 12 Version - Check standard Soap headers
                {
                    var authToken = operationContext.IncomingMessageHeaders.GetHeader<AuthenticationToken>(AuthenticationToken.HeaderName, AuthenticationToken.HeaderNamespace);
                    extension = new AuthenticationExtension(authToken);
                }
                else
                {
                    extension = new AuthenticationExtension(new AuthenticationToken { IsAnonymous = true });
                }

                OperationContext.Current.Extensions.Add(extension);
            }
            
            return extension.CurrentToken;
        }

        private static bool UseWebOperationContext
        {
            get
            {
                try
                {
                    return WebOperationContext.Current != null
                        && WebOperationContext.Current.IncomingRequest != null &&
                        WebOperationContext.Current.IncomingRequest.Headers.Get(AuthenticationToken.HeaderName) != null;
                }
                catch
                {
                    return false;
                }
            }
        }

        private static AuthenticationToken GetForWebOperationContext(WebOperationContext operationContext)
        {
            var extension = OperationContext.Current.Extensions.Find<AuthenticationExtension>();

            if (extension == null)
            {
                var header = operationContext.IncomingRequest.Headers.Get(AuthenticationToken.HeaderName);

                if (!string.IsNullOrEmpty(header))
                {
                    var token = JsonConvert.DeserializeObject<AuthenticationToken>(header);
                    extension = new AuthenticationExtension(token);                    
                }
                else
                {
                    extension = new AuthenticationExtension(new AuthenticationToken { IsAnonymous = true });
                }

                OperationContext.Current.Extensions.Add(extension);
            }

            return extension.CurrentToken;            
        }
    }
}
