﻿using System;
using System.Diagnostics;
using System.Net.Sockets;
using System.ServiceModel;
using System.ServiceModel.Security;
using System.Threading;
using System.Threading.Tasks;
using Curse.CloudServices.Models;
using Curse.Logging;

namespace Curse.CloudServices.Client
{
    public abstract class BaseServiceClient<TSelf, TInterface, TClient>
        where TSelf : class, new()
        where TClient : ClientBase<TInterface>, ICommunicationObject
        where TInterface : class
    {
        public static readonly TSelf Instance = new TSelf();
        protected static readonly LogCategory Logger;

        private bool _enableCallLogging = true;

        public bool EnableCallLogging
        {
            get { return _enableCallLogging; }
            set { _enableCallLogging = value; }
        }

        static BaseServiceClient()
        {
            Logger = new LogCategory(typeof(TSelf).Name);
        }

        protected TClient Client;

        protected abstract TClient CreateClient();        

        public delegate TResponse CallDelegate<out TResponse>(TClient client);

        public ServiceResult<TResponse> TryCall<TResponse>(CallDelegate<TResponse> call, string failureLogMessage = null, int attemptLimit = 2)
        {

#if DEBUG
            if (Thread.CurrentThread.Name == "MainThread")
            {
                // If you hit this, it means you tried to run an in stall task on the UI thread. This is not advisable.
                Debugger.Break();
            }
#endif
            try
            {
                var attemptCounter = 0;
                failureLogMessage = failureLogMessage ?? "Service Call Error";

                while (attemptCounter++ < attemptLimit)
                {
                    try
                    {
                        EnsureClient();
                        return new ServiceResult<TResponse>(call(Client));
                    }
                    catch (SecurityNegotiationException ex)
                    {
                        Logger.Warn("A security negotiation error has been detected.");

                        if (attemptCounter >= attemptLimit)
                        {
                            Logger.Error(ex, "A SecurityNegotiationException has occurred after " + attemptCounter + " attempts.");
                            return new ServiceResult<TResponse>(ServiceResultStatus.Security, ex);
                        }
                        
                        Logger.Warn(ex, "A SecurityNegotiationException has occurred. Attempting to reconnect...");                        
                        Reconnect();
                    }
                    catch (FaultException ex)
                    {
                        Logger.Error(ex, failureLogMessage);

                        var tryAgain = HandleServiceFault<TResponse>(ex);

                        if (tryAgain && attemptCounter < attemptLimit)
                        {
                            continue;
                        }

                        return new ServiceResult<TResponse>(ServiceResultStatus.Fault, ex);
                    }
                    catch (EndpointNotFoundException ex)
                    {
                        if (!IsConnectionAvailable)
                        {
                            Logger.Warn("An active Internet connection is not currently available.");
                        }
                        else
                        {
                            Logger.Error(ex, failureLogMessage);
                        }

                        if (attemptCounter >= attemptLimit)
                        {
                            return new ServiceResult<TResponse>(ServiceResultStatus.Communication, ex);
                        }

                        Reconnect();                        
                    }
                    catch (CommunicationException ex)
                    {
                        if (!IsConnectionAvailable)
                        {
                            Logger.Warn("An active Internet connection is not currently available.");
                            return new ServiceResult<TResponse>(ServiceResultStatus.Communication, ex);
                        }

                        Logger.Error(ex, failureLogMessage);

                        var exMessage = ex.Message.ToLowerInvariant();

                        // TODO: Find a better way to determine that this is a cert or protocol error... This will only work for English =/
                        if (exMessage.Contains("certificate") || exMessage.Contains("protocol"))
                        {
                            Logger.Warn("A certitficate or protocol error has been detected.");                            
                        }
                        
                        if (attemptCounter >= attemptLimit)
                        {
                            return new ServiceResult<TResponse>(ServiceResultStatus.Communication, ex);
                        }

                        Reconnect();
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, failureLogMessage);
                        return new ServiceResult<TResponse>(ServiceResultStatus.Exception, ex);
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Fatal error during TryCall: " + failureLogMessage);
            }

            return new ServiceResult<TResponse>(ServiceResultStatus.Exception);
        }

        public void CallAsync<TResponse>(string description, CallDelegate<TResponse> call, Action<TResponse> responseHandler, object logEntryArgs = null, int attemptLimit = 2)
        {
            if (EnableCallLogging)
            {
                try
                {
                    Logger.Info(description, logEntryArgs);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to write log entry!");
                }
            }

            Task.Factory.StartNew(() =>
            {
                TResponse response;

                try
                {
                    response = Call(description, call, attemptLimit);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, description, logEntryArgs);
                    return;
                }

                try
                {
                    responseHandler(response);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, description, logEntryArgs);
                }
            }, CancellationToken.None, TaskCreationOptions.None, TaskScheduler.Default);
        }

        public TResponse Call<TResponse>(string description, CallDelegate<TResponse> call, int attemptLimit = 2)
        {
            if (EnableCallLogging)
            {
                Logger.Info("Making service call: " + description);
            }

            var response = TryCall(call, null, attemptLimit);
            if (response.Status == ServiceResultStatus.Successful)
            {
                return response.Value;
            }

            Logger.Error(response.Exception, description);
            throw new Exception(description, response.Exception);
        }

        protected virtual bool HandleServiceFault<V>(FaultException ex)
        {
            CloseConnection();

            var reasonCode = -1;

            if (!int.TryParse(ex.Message, out reasonCode))
            {
                return false;
            }

            var status = (AuthenticationStatus)reasonCode;

            if (status == AuthenticationStatus.ExpiredSession || status == AuthenticationStatus.InvalidSession)
            {
                Logger.Info("Renewing token...");
                var success = RenewToken();
                if (success)
                {
                    Reconnect();
                    return true;
                }
            }
            else
            {
                Logger.Error(ex, "Unexpected service fault!");
            }

            return false;
        }

        protected string CurrentToken;

        protected virtual bool RenewToken()
        {
            return false;
        }

        protected void EnsureClient()
        {
            var client = Client;

            if (client != null && (client.State != CommunicationState.Faulted && client.State != CommunicationState.Closing && client.State != CommunicationState.Closed))
            {
                return;
            }

            Reconnect();
        }

        protected void Reconnect()
        {
            CloseConnection();
            Client = CreateClient();
        }

        private void CloseConnection()
        {
            var client = Client;
            
            try
            {
                if (client == null)
                {
                    return;
                }

                if (client.State == CommunicationState.Closed)
                {                    
                    return;
                }

                Logger.Info("Closing service client connection...");

                try
                {
                    client.Close(TimeSpan.FromSeconds(2));
                }
                catch
                {
                    client.Abort();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "General error when closing client.");
            }
            finally
            {
                if (client == Client)
                {
                    Client = null;
                }
            }
        }

        private bool IsConnectionAvailable
        {
            get
            {
                try
                {
                    using (var clnt = new TcpClient("www.google.com", 80))
                    {
                        clnt.Close();
                        return true;
                    }
                }
                catch
                {
                    return false;
                }
            }
        }

        public void Close()
        {
            CloseConnection();
        }
    }
}
