﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using System.Threading.Tasks;
using Curse.Logging;
using System.Threading;
using Curse.Aerospike;

namespace Curse.CloudServices.Jobs
{
    public static class JobScheduler
    {

        private static readonly Dictionary<string, Type> _nameToType = new Dictionary<string, Type>();
        private static HashSet<string> _loggedMissingJobs = new HashSet<string>();
        private static readonly LogCategory Logger = new LogCategory("Job Scheduler");

        public static void Initialize(bool isDefaultRegion)
        {
            try
            {
                var distinctTypes = GetDistinctTypes();
                var jobTypes = distinctTypes.Where(t => t.IsClass && !t.IsAbstract && typeof(BaseJob).IsAssignableFrom(t))
                    .ToArray();

                Logger.Info("Job scheduler found " + jobTypes.Length + " jobs to register.");

                foreach (var type in jobTypes)
                {
                    // Try to find the job info for it
                    var job = Activator.CreateInstance(type) as BaseJob;

                    if (job == null)
                    {
                        continue;
                    }

                    // Ensure that this job should run in the current region
                    if (job.OnlyDefaultRegion && !isDefaultRegion)
                    {
                        Logger.Info("Skipping job '" + type.Name + "'. It is configured to only run in the default region.");
                        continue;                        
                    }


                    var newJobInfo = new JobInfo
                    {
                        Name = type.FullName,
                        CurrentRunner = null,
                        ScheduleMode = job.ScheduleMode,
                        IndexMode = IndexMode.Default
                    };

                    // Try to get the job info
                    var jobInfo = JobInfo.GetLocal(type.FullName);

                    if (jobInfo != null)
                    {
                        Logger.Info("Updating job info for '" + type.Name + "'");
                        jobInfo.ScheduleMode = newJobInfo.ScheduleMode;
                        jobInfo.IndexMode = IndexMode.Default;
                        if (jobInfo.CurrentRunner == Environment.MachineName)
                        {
                            jobInfo.CurrentRunner = null;
                        }
                        jobInfo.Update();
                    }
                    else
                    {
                        Logger.Info("Registering new job info for '" + type.Name + "'");
                        newJobInfo.InsertLocal();
                    }

                    _nameToType[type.FullName] = type;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to initialize!");
                return;
            }

            new Thread(SchedulerThread) { IsBackground = true, Name = "Job Scheduler" }.Start();

        }


        private static void SchedulerThread()
        {
            while(true)
            {
                try
                {
                    SchedulerTick();
                    Thread.Sleep(TimeSpan.FromSeconds(30));
                }
                catch (ThreadAbortException)
                {

                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Job scheduler thread failure!");
                }
            }
        }

        private static void SchedulerTick()
        {
            // Go through each job, and see if we should run it
            var jobInfos = JobInfo.GetAllLocal(p => p.IndexMode, IndexMode.Default)
                                  .Where(p => !p.IsDeleted)
                                  .ToArray();
          

            foreach(var jobInfo in jobInfos)
            {
                if(!string.IsNullOrEmpty(jobInfo.CurrentRunner))
                {
                    continue;
                }

                if (_loggedMissingJobs.Contains(jobInfo.Name))
                {
                    continue;
                }

                Type jobType = null;
                if (!_nameToType.TryGetValue(jobInfo.Name, out jobType))
                {
                    Logger.Warn("Unable to find type for job: " + jobInfo.Name);
                    _loggedMissingJobs.Add(jobInfo.Name);
                    continue;
                }

                try
                {
                    var job = Activator.CreateInstance(jobType) as BaseJob;

                    // We need to run it
                    if (job.ShouldRun(jobInfo.LastRunTime))
                    {
                        Logger.Info("[" + jobInfo.Name + "] Scheduled to run.");
                        jobInfo.CurrentRunner = Environment.MachineName;
                        jobInfo.LastRunTime = DateTime.UtcNow;
                        try
                        {
                            jobInfo.Update(UpdateMode.Concurrent, p => p.CurrentRunner, p => p.LastRunTime);
                        }
                        catch (Exception ex)
                        {
                            if (ex.Message.Contains("Generation error"))
                            {
                                Logger.Info(ex, "[" + jobInfo.Name + "] Failed to start on this node, due to a concurrency conflict designed to prevent jobs from running multiple times. This is a normal occurence, and can be safely ignored.");
                                return;
                            }

                            throw;
                        }
                        
                        Task.Factory.StartNew(RunJob, job, TaskCreationOptions.LongRunning);
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to start job task!", jobInfo);
                }
                
            }
        }

        private static void RunJob(object o)
        {
            var job = o as BaseJob;

            if(job == null)
            {

                Logger.Error("Invalid argument supplied to run a job: " + o.GetType().FullName);
                return;
            }

            string jobName = job.GetType().FullName;

            var history = new JobHistory
            {
                ID = Guid.NewGuid(),
                JobName = jobName,
                DateStarted = DateTime.UtcNow,
                Status = JobRunStatus.Started,
                HostName = Environment.MachineName
            };
            history.InsertLocal();

            var startTime = DateTime.UtcNow;

            try
            {

                Logger.Debug("[" + jobName + "] Starting job");
                job.Run();
                var elapsed = DateTime.UtcNow - startTime;
                history.Status = JobRunStatus.Completed;
                history.DateCompleted = DateTime.UtcNow;
                history.DurationMilliseconds = (long)elapsed.TotalMilliseconds;
                history.Update();
                Logger.Info("[" + jobName + "] Job completed in " + elapsed.TotalSeconds.ToString("###,##0.00") + " seconds."); 
            }
            catch (Exception ex)
            {                                
                var elapsed = DateTime.UtcNow - startTime;
                history.Status = JobRunStatus.Failed;
                history.StatusMessage = ex.Message + Environment.NewLine + ex.StackTrace;
                history.DateCompleted = DateTime.UtcNow;
                history.DurationMilliseconds = (long)elapsed.TotalMilliseconds;
                history.Update();
                Logger.Error(ex, "Failed to run job: " + jobName);
            }
            finally
            {
                try
                {
                    var jobInfo = JobInfo.GetLocal(jobName);
                    jobInfo.CurrentRunner = null;
                    jobInfo.Update();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to cleanup job: " + jobName);
                }                
            }
        }

        private static Type[] GetDistinctTypes()
        {
            var allTypes = new List<Type>();
            // Using reflection get all classes which inherit from BaseJob
            var assemblies = AppDomain.CurrentDomain.GetAssemblies();
            foreach (var assembly in assemblies)
            {

                Type[] types = null;
                try
                {
                    types = assembly.GetTypes();
                    allTypes.AddRange(types);
                }
                catch
                {

                    continue;
                }
            }

            return allTypes.Distinct().ToArray();
        }

    }
}
