﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using System.ServiceModel.Channels;
using System.ServiceModel.Description;
using System.ServiceModel.Dispatcher;
using System.ServiceModel.Web;
using System.Text;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace Curse.CloudServices.Wcf.Json
{
    public class NewtonsoftJsonDispatchFormatter : IDispatchMessageFormatter
    {
        readonly OperationDescription _operation;
        readonly Dictionary<string, int> _parameterNames;
        private readonly bool _isResponseWrapped;
        private readonly bool _isRequestWrapped;
        private readonly string _wrapperName;

        public NewtonsoftJsonDispatchFormatter(OperationDescription operation, bool isRequest)
        {
            _operation = operation;

            var bodyStyle = operation.GetBodyStyle();
            if (isRequest)
            {
                var operationParameterCount = operation.Messages[0].Body.Parts.Count;
                _parameterNames = new Dictionary<string, int>();
                for (var i = 0; i < operationParameterCount; i++)
                {
                    _parameterNames.Add(operation.Messages[0].Body.Parts[i].Name, i);
                }
                _isRequestWrapped = bodyStyle == WebMessageBodyStyle.Wrapped || bodyStyle == WebMessageBodyStyle.WrappedRequest;
                _wrapperName = operation.Name + "Request";
            }
            else
            {
                _isResponseWrapped = bodyStyle == WebMessageBodyStyle.Wrapped || bodyStyle == WebMessageBodyStyle.WrappedResponse;
                _wrapperName = operation.Name + "Result";
            }
        }

        public void DeserializeRequest(Message message, object[] parameters)
        {
            object bodyFormatProperty;
            if (!message.Properties.TryGetValue(WebBodyFormatMessageProperty.Name, out bodyFormatProperty) ||
                ((WebBodyFormatMessageProperty)bodyFormatProperty).Format != WebContentFormat.Raw)
            {
                throw new InvalidOperationException("Incoming messages must have a body format of Raw. Is a ContentTypeMapper set on the WebHttpBinding?");
            }

            var bodyReader = message.GetReaderAtBodyContents();
            bodyReader.ReadStartElement("Binary");
            var rawBody = Convert.FromBase64String(bodyReader.ReadString());
            var serializer = new JsonSerializer();

            using(var ms = new MemoryStream(rawBody))
            using(var sr = new StreamReader(ms))
            using (var reader = new JsonTextReader(sr))
            {
                if (_isRequestWrapped)
                {
                    reader.Read();
                    if (reader.TokenType != JsonToken.StartObject)
                    {
                        throw new InvalidOperationException("Input was not wrapped");
                    }
                }
                else if (parameters.Length == 1)
                {
                    parameters[0] = serializer.Deserialize(sr, _operation.Messages[0].Body.Parts[0].Type);
                    return;
                }

                reader.Read();
                while (reader.TokenType == JsonToken.PropertyName)
                {
                    var parameterName = (string) reader.Value;
                    reader.Read();
                    if (_parameterNames.ContainsKey(parameterName))
                    {
                        var parameterIndex = _parameterNames[parameterName];
                        parameters[parameterIndex] = serializer.Deserialize(reader, _operation.Messages[0].Body.Parts[parameterIndex].Type);
                    }
                    else
                    {
                        reader.Skip();
                    }

                    reader.Read();
                }
            }
        }

        public Message SerializeReply(MessageVersion messageVersion, object[] parameters, object result)
        {
            byte[] body;
            var serializer = new JsonSerializer();

            using (var ms = new MemoryStream())
            {
                using (var sw = new StreamWriter(ms, Encoding.UTF8))
                {
                    using (JsonWriter writer = new JsonTextWriter(sw))
                    {
                        if (_isResponseWrapped)
                        {
                            var obj = new JObject();
                            obj.Add(new JProperty(_wrapperName, JToken.FromObject(result)));
                            obj.WriteTo(writer);
                        }
                        else
                        {
                            serializer.Serialize(writer, result);
                        }

                        sw.Flush();
                        body = ms.ToArray();
                    }
                }
            }

            var replyMessage = Message.CreateMessage(messageVersion, _operation.Messages[1].Action, new RawBodyWriter(body));
            replyMessage.Properties.Add(WebBodyFormatMessageProperty.Name, new WebBodyFormatMessageProperty(WebContentFormat.Raw));
            var respProp = new HttpResponseMessageProperty();
            respProp.Headers[HttpResponseHeader.ContentType] = "application/json";
            replyMessage.Properties.Add(HttpResponseMessageProperty.Name, respProp);
            return replyMessage;
        }
    }

}
