﻿using System;
using System.Linq;
using Curse.Extensions;
using Curse.Friends.AccountsWebService.Contracts;
using Curse.Friends.AuthenticationClient;
using Curse.Friends.AuthenticationClient.AuthService;
using Curse.Friends.Enums;
using Curse.Logging;

namespace Curse.Friends.AccountsWebService.Authentication
{
    internal class AccountsAuthenticationProvider : AuthenticationProvider
    {

        public static ChangePasswordStatus ChangePassword(int userID, string oldPassword, string newPassword)
        {
            try
            {
                using (var client = CreateServiceClient())
                {
                    var encryptedOldPassword = _outgoingCipher.Encrypt(oldPassword);
                    var encryptedNewPassword = _outgoingCipher.Encrypt(newPassword);
                    var result = client.v2SetUserPassword(_siteID, userID, encryptedOldPassword, encryptedNewPassword);

                    switch (result)
                    {
                        case ESetUserPasswordResult.Success:
                            return ChangePasswordStatus.Success;
                        case ESetUserPasswordResult.InvalidPassword:
                            return ChangePasswordStatus.InvalidOldPassword;

                        case ESetUserPasswordResult.Unsuccessful:
                        case ESetUserPasswordResult.UserNotFound:
                        default:
                            return ChangePasswordStatus.Error;
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to change password!");
                return ChangePasswordStatus.Error;
            }
        }

        public static ChangeEmailStatus ChangeEmail(int userID, string username, string password, string email)
        {
            try
            {
                using (var client = CreateServiceClient())
                {
                    var encryptedPassword = _outgoingCipher.Encrypt(password);
                    var validation = client.v2ValidateClientUser(_siteID, username, encryptedPassword);
                    if (validation.Status != ELoginStatus.Success)
                    {
                        return ChangeEmailStatus.InvalidPassword;
                    }

                    var status = client.v2UpdatePrimaryEmailAddress(_siteID, userID, email);
                    switch (status)
                    {
                        case EUpdatePrimaryEmailStatus.Successful:
                            return ChangeEmailStatus.Successful;
                        case EUpdatePrimaryEmailStatus.EmailInUse:
                            return ChangeEmailStatus.EmailInUse;
                        case EUpdatePrimaryEmailStatus.InvalidEmail:
                            return ChangeEmailStatus.InvalidEmail;
                        default:
                            return ChangeEmailStatus.Error;
                    }

                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to change password!");
                return ChangeEmailStatus.Error;
            }
        }

        public static ChangeUsernameStatus ChangeUsername(int userID, string username, string password, string newUsername)
        {
            try
            {
                using (var client = CreateServiceClient())
                {
                    var encryptedPassword = _outgoingCipher.Encrypt(password);
                    var validation = client.v2ValidateClientUser(_siteID, username, encryptedPassword);
                    if (validation.Status != ELoginStatus.Success)
                    {
                        return ChangeUsernameStatus.InvalidPassword;
                    }

                    var status = client.v2AdminRenameUser(_siteID, userID, newUsername);
                    switch (status)
                    {
                        case EAdminRenameUserStatus.Success:
                            return ChangeUsernameStatus.Success;
                        case EAdminRenameUserStatus.InvalidUsername:
                            return ChangeUsernameStatus.InvalidUsername;
                        case EAdminRenameUserStatus.UsernameInUse:
                            return ChangeUsernameStatus.UsernameInUse;
                        default:
                            return ChangeUsernameStatus.Error;
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to change username!");
                return ChangeUsernameStatus.Error;
            }
        }

        public static AccountEntitlement[] GetAccountEntitlements(int userID)
        {
            try
            {
                using (var client = CreateServiceClient())
                {
                    var res = client.v2GetUserProfile(_siteID, userID);
                    if (res == null || res.Entitlements == null)
                    {
                        return new AccountEntitlement[0];
                    }

                    return res.Entitlements.Select(p => new AccountEntitlement { Identifier = p.ID, IsActive = p.Active, ExpirationTimestamp = p.Expires.ToEpochMilliseconds() }).ToArray();                    
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to get account entitlements");
                return new AccountEntitlement[0];
            }
        }
    }

}