﻿using System;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Extensions;
using Curse.Friends.AccountsWebService.Authentication;
using Curse.Friends.AccountsWebService.Contracts;
using Curse.Friends.AuthenticationClient;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Statistics;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.Friends.AccountsWebService.Configuration;
using Curse.Friends.Configuration;
using Curse.Friends.UserEvents;
using Curse.Logging;
using System.Net;

namespace Curse.Friends.AccountsWebService.Controllers
{
    [RoutePrefix("account")]
    public class AccountController : MicroServiceController
    {

        [HttpGet]
        [Route("preferences/push")]
        public PushPreferencesResponse GetPushPreferences()
        {
            var me = GetCurrentUserAndRegion();
            return new PushPreferencesResponse
            {
                FriendMessagePushPreference = me.User.FriendMessagePushPreference,
                GroupMessagePushPreference = me.User.GroupMessagePushPreference,
                FriendRequestPushEnabled = me.User.FriendRequestPushEnabled,
                MentionsPushEnabled = !me.User.MentionsPushEnabled.HasValue || me.User.MentionsPushEnabled.Value
            };
        }
        
        [HttpPost]
        [Route("preferences/push")]
        public IHttpActionResult ChangePushPreferences(PushNotificationPreferencesRequest request)
        {

            var user = GetCurrentUserAndRegion().User;
            user.GroupMessagePushPreference = request.GroupMessagePushPreference;
            user.FriendMessagePushPreference = request.FriendMessagePushPreference;
            user.FriendRequestPushEnabled = request.FriendRequestPushEnabled;
            user.MentionsPushEnabled = request.MentionsPushEnabled;
            user.Update(u => u.GroupMessagePushPreference,
                u => u.FriendMessagePushPreference,
                u => u.FriendRequestPushEnabled,
                u => u.MentionsPushEnabled);

            // Let all of the user's endpoints know about this change
            RegionalUserChangeResolver.Create(user.UserID, false);

            return Ok();
        }

        [HttpGet]
        [Route("preferences/privacy")]
        [ResponseType(typeof(UserPrivacySettingsContract))]
        public IHttpActionResult GetPrivacyPreferences()
        {
            var me = GetCurrentUserAndRegion();
            return Ok(me.User.GetPrivacy().ToContract());
        }

        [HttpPost]
        [Route("preferences/privacy")]
        [ResponseType(typeof(void))]
        public IHttpActionResult ChangePrivacyPreferences(UserPrivacySettingsContract request)
        {
            var me = GetCurrentUserAndRegion();
            var privacy = me.User.GetPrivacy();
            var blockStrangerPMs = request.PrivateMessagePrivacy != PrivateMessagePrivacy.Anyone;

            var conversationPrivacyChanged = privacy.BlockStrangerPMs != blockStrangerPMs;
            var activitySharingChanged = privacy.ShareActivityPrivacy != request.ShareActivityPrivacy;

            privacy.FriendRequestPrivacy = request.FriendRequestPrivacy;
            privacy.BlockStrangerPMs = blockStrangerPMs;            
            privacy.ShareActivityPrivacy = request.ShareActivityPrivacy;

            privacy.Update();

            // Update the user statistics
            me.User.UpdateStatistics();

            if (conversationPrivacyChanged)
            {
                new ConversationSettingsEvent { UserID = me.User.UserID, RestrictWhispers = privacy.BlockStrangerPMs }.Enqueue();
            }

            if (activitySharingChanged)
            {                
                new PresenceSettingEventV2
                {
                    UserID = me.User.UserID,
                    Availability = me.User.LastConnectionStatus,
                    ShareActivity = request.ShareActivityPrivacy == ShareActivityPrivacy.Share
                }.Enqueue();                
            }

            Logger.Debug("User has changed privacy settings", new { stats = me.User.GetStatistics(), conversationPrivacyChanged, activitySharingChanged, request, privacy });
            
            // Let all of the user's endpoints know about this change
            RegionalUserChangeResolver.Create(me.User.UserID, activitySharingChanged);

            return Ok();
        }

        [HttpPost]
        [Route("status/idle")]
        public IHttpActionResult SetIdle(SetIdleRequest request)
        {
            request.Validate();

            var endpoint = ClientEndpoint.GetLocal(Token.UserID, request.MachineKey);

            if (endpoint == null)
            {
                return NotFound();
            }


            if (!endpoint.IsConnected)
            {
                return BadRequest("You cannot set the idle status of an endpoint that is not connected.");
            }

            if (endpoint.IsIdle)
            {
                return Ok("Endpoint is already idle.");
            }

            // From their storage region, get their record and update their status            
            var userRegion = UserRegion.GetByUserID(Token.UserID, false);

            if (userRegion == null)
            {
                return NotFound();
            }
            
            endpoint.ToggleIdle(true, userRegion.RegionID);
            UserPresenceResolver.CreateAvailabilityResolver(userRegion.UserID, userRegion.RegionID, endpoint.MachineKey);
            return Ok();
        }

        [HttpPost]
        [Route("status/active")]
        public IHttpActionResult SetActive(SetActiveRequest request)
        {
            request.Validate();

            var endpoint = ClientEndpoint.GetLocal(Token.UserID, request.MachineKey);

            if (endpoint == null)
            {
                return NotFound();
            }

            if (!endpoint.IsConnected)
            {
                return BadRequest("You cannot set the idle status of an endpoint that is not connected.");
            }

            if (!endpoint.IsIdle)
            {
                return BadRequest("Endpoint is already active.");
            }

            // From their storage region, get their record and update their status            
            var userRegion = UserRegion.GetByUserID(Token.UserID, false);

            if (userRegion == null)
            {
                return NotFound();
            }

            endpoint.ToggleIdle(false, userRegion.RegionID);
            UserPresenceResolver.CreateAvailabilityResolver(userRegion.UserID, userRegion.RegionID, endpoint.MachineKey);
            return Ok();
        }
       

        [HttpPost]
        [Route("status/connection")]
        public IHttpActionResult ChangeStatus(ChangeStatusRequest request)
        {
            request.Validate();
            
            var endpoint = ClientEndpoint.GetLocal(Token.UserID, request.MachineKey);

            if (endpoint == null)
            {
                return NotFound();
            }

            var userRegion = UserRegion.GetLocal(Token.UserID);
            if(userRegion == null)
            {
                return ErrorResponse(HttpStatusCode.NotFound, "user_not_found", "Could not retrieve user from the database");
            }

            // If the user's endpoint is not connected, do not permit this change
            if (!endpoint.IsConnected)
            {
                return Forbidden();
            }

            // Handle legacy API calls to set idle/active
            if (endpoint.IsIdle && request.Status != UserConnectionStatus.Idle)
            {
                return SetActive(new SetActiveRequest { MachineKey = request.MachineKey });
            }
            
            if (!endpoint.IsIdle && request.Status == UserConnectionStatus.Idle)
            {
                return SetIdle(new SetIdleRequest { MachineKey = request.MachineKey });
            }

            // Queue off a job to let their friends know their status            
            UserPresenceResolver.CreateAvailabilityOverrideResolver(userRegion.UserID, userRegion.RegionID, endpoint.MachineKey, request.Status);

            return Ok();
        }

        [HttpPost]
        [Route("status/game")]
        public IHttpActionResult ChangeGameStatus(ChangeGameStatusRequest request)
        {
            request.Validate();
            var userRegion = UserRegion.GetLocal(Token.UserID);
            if(userRegion == null)
            {
                return ErrorResponse(HttpStatusCode.NotFound, "user_not_found", "Could not retrieve the user from the database");
            }

            var ep = ClientEndpoint.GetLocal(Token.UserID, request.MachineKey);
            if (ep == null)
            {
                return ErrorResponse(System.Net.HttpStatusCode.NotFound, "endpoint_not_found", "Endpoint for the specified machine key could not be found");
            }

            ep.ChangePlaying(userRegion.RegionID, request.IsRunning ? request.GameID : 0, request.GameState, request.GameStatusMessage);
            return Ok();
        }

        [HttpPost]
        [Route("status/watching")]
        public IHttpActionResult ChangeWatchingStatus(ChangeWatchingStatusRequest request)
        {
            request.Validate();
            var userRegion = UserRegion.GetLocal(Token.UserID);
            if (userRegion == null)
            {
                return ErrorResponse(HttpStatusCode.NotFound, "user_not_found", "Could not retrieve the user from the database");
            }

            var ep = ClientEndpoint.GetLocal(Token.UserID, request.MachineKey);
            if (ep == null)
            {
                return ErrorResponse(System.Net.HttpStatusCode.NotFound, "endpoint_not_found", "Endpoint for the specified machine key could not be found");
            }

            ep.ChangeWatching(userRegion.RegionID, request.IsWatching ? request.ChannelID : string.Empty);
            return Ok();
        }

        [HttpPost]
        [Route("profile")]
        public IHttpActionResult ChangeProfile(ChangeProfileRequest request)
        {
            request.Validate();

            // From their storage region, get their record and update their status            
            var userAndRegion = GetCurrentUserAndRegion();
            
            // Update their profile info            
            userAndRegion.User.AboutMe = request.AboutMe;
            userAndRegion.User.City = request.City;
            userAndRegion.User.State = request.State;
            userAndRegion.User.CountryCode = request.CountryCode;
            userAndRegion.User.Name = request.Name;
            userAndRegion.User.AboutMe = request.AboutMe;
            userAndRegion.User.Update(p => p.AvatarUrl, p => p.AboutMe, p => p.City, p => p.State, p => p.CountryCode, p => p.Name, p => p.AboutMe);

            // Fan this change out to the user and their friends            
            RegionalUserChangeResolver.Create(userAndRegion.User.UserID);
            
            var hasAnyData = !string.IsNullOrEmpty(request.AboutMe)
                                || !string.IsNullOrEmpty(request.City)
                                || !string.IsNullOrEmpty(request.CountryCode)
                                || !string.IsNullOrEmpty(request.Name)
                                || !string.IsNullOrEmpty(request.State);

            if (hasAnyData)
            {
                FriendsStatsManager.Current.ProfilesUpdated.Track(userAndRegion.User.UserID);
            }

            return Ok();
        }

        [HttpGet]
        [Route("profile")]
        [ResponseType(typeof(UserProfileNotification))]
        public IHttpActionResult GetProfile()
        {
            var currentUser = GetCurrentUserAndRegion();
            return Ok(currentUser.User.GetUserProfile());
        }

        [HttpPost]
        [Route("password")]
        [ResponseType(typeof(ChangePasswordStatus))]
        public IHttpActionResult ChangePassword(ChangePasswordRequest request)
        {
            if (request == null)
            {
                return BadRequest();
            }

            var validation = request.Validate();

            if (validation != ChangePasswordStatus.Success)
            {
                return BadRequest(validation);
            }

            var currentUser = GetCurrentUserAndRegion();
            if (currentUser == null)
            {
                return NotFound();
            }

            if (AuthenticationContext.IsTempAccount)
            {
                return Forbidden();
            }

            if (currentUser.User.IsMerged)
            {
                return Forbidden();
            }

            var result = AccountsAuthenticationProvider.ChangePassword(currentUser.User.UserID, request.OldPassword, request.NewPassword);

            if (result != ChangePasswordStatus.Success)
            {
                return BadRequest(result);
            }

            // Disabled for now, until this endpoint is updated in the client to refresh token
            // var selfStats = UserStatistics.GetByUserOrDefault(currentUser.User.UserID);
            // selfStats.SetMinimumTokenDate(DateTime.UtcNow);

            return Ok(result);            

        }

        [HttpPost]
        [Route("email")]
        [ResponseType(typeof (ChangeEmailStatus))]
        public IHttpActionResult ChangeEmail([FromBody] ChangeEmailRequest request)
        {

            if (string.IsNullOrEmpty(request.Password))
            {
                return BadRequest(ChangeEmailStatus.InvalidPassword);
            }

            if (!AuthenticationValidation.IsValidEmail(request.Email))
            {
                return BadRequest(ChangeEmailStatus.InvalidEmail);
            }

            if (AuthenticationContext.IsTempAccount)
            {
                return Forbidden();
            }            

            var me = GetCurrentUserAndRegion();

            if (me.User.IsMerged)
            {
                return Forbidden();
            }

            var result = AccountsAuthenticationProvider.ChangeEmail(me.User.UserID, Token.Username, request.Password, request.Email);

            if (result != ChangeEmailStatus.Successful)
            {
                return BadRequest(result);
            }

            me.User.UpdateEmail(request.Email);

            return Ok(result);
        }

        [HttpGet]
        [Route("rename")]
        [ResponseType(typeof (GetChangeUsernameAvailabilityResponse))]
        public IHttpActionResult GetChangeUsernameAvailability()
        {
            var response = new GetChangeUsernameAvailabilityResponse();

            if (AuthenticationContext.IsTempAccount)
            {
                return Forbidden();
            }

            var me = GetCurrentUserAndRegion();

            if (DateTime.UtcNow - me.User.LastNameChangeDate < Data.User.AccountRenameThrottleTimespan)
            {
                response.Availability = ChangeUsernameAvailability.NotAvailable;
                response.MillisecondsBeforeRetry = (long) (me.User.LastNameChangeDate + Data.User.AccountRenameThrottleTimespan - DateTime.UtcNow).TotalMilliseconds;
            }
            else
            {
                response.Availability = ChangeUsernameAvailability.Available;
            }

            return Ok(response);
        }

        [HttpPost]
        [Route("rename")]
        [ResponseType(typeof (ChangeUsernameStatus))]
        public IHttpActionResult ChangeUsername([FromBody] ChangeUsernameRequest request)
        {
            if (string.IsNullOrEmpty(request.Password))
            {
                return BadRequest(ChangeUsernameStatus.InvalidPassword);
            }

            if (string.IsNullOrEmpty(request.NewUsername))
            {
                return BadRequest(ChangeUsernameStatus.InvalidUsername);
            }

            if (AuthenticationContext.IsTempAccount)
            {
                return Forbidden();
            }

            var user = GetCurrentUserAndRegion();

            // Prevent merged accounts from renaming
            if (user.User.IsMerged)
            {
                return BadRequest(ChangeUsernameStatus.TooManyAttempts);
            }

            if (DateTime.UtcNow - user.User.LastNameChangeDate < Data.User.AccountRenameThrottleTimespan)
            {
                return BadRequest(ChangeUsernameStatus.TooManyAttempts);
            }

            var result = AccountsAuthenticationProvider.ChangeUsername(Token.UserID, Token.Username, request.Password, request.NewUsername);

            if (result != ChangeUsernameStatus.Success)
            {
                return BadRequest(result);
            }

            user.User.ChangeUsername(request.NewUsername);
            return Ok(result);
        }

        [HttpGet]
        [Route("preferences/client-settings")]
        [ResponseType(typeof(UserClientSettingsNotification))]
        public IHttpActionResult GetClientSettings()
        {
            var currentUser = GetCurrentUserAndRegion();
            return Ok(currentUser.User.GetClientSettings().ToNotification());
        }

        [HttpPost]
        [Route("preferences/client-settings")]
        [ResponseType(typeof(UserClientSettingsNotification))]
        public IHttpActionResult ChangeClientSettings([FromBody] ChangeClientSettingsRequest request)
        {
            request.Validate();

            var currentUser = GetCurrentUserAndRegion();
            var settings = currentUser.User.GetClientSettings();
            settings.ChangeSettings(request.GlobalSettings, request.DesktopSettings, request.WebSettings, request.MobileSettings);
            return Ok(settings.ToNotification());
        }

        [HttpGet]
        [Route("entitlements")]
        [ResponseType(typeof (AccountEntitlementsResponse))]
        public IHttpActionResult GetAccountEntitlements()
        {

            var entitlements = new List<AccountEntitlement>(AccountsAuthenticationProvider.GetAccountEntitlements(Token.UserID));

            if (AccountsWebServiceConfiguration.Current.PremiumPromoThrough >= DateTime.UtcNow.ToEpochMilliseconds())
            {
                entitlements.Add(new AccountEntitlement { ExpirationTimestamp = AccountsWebServiceConfiguration.Current.PremiumPromoThrough, Identifier = "CPP", IsActive = true});
            }

            return Ok(new AccountEntitlementsResponse
            {
                Entitlements = entitlements.ToArray()
            });
        }

        [HttpGet]
        [Route("emotes")]
        [ResponseType(typeof (IEnumerable<EmoteContract>))]
        public IHttpActionResult GetAllowedEmotes()
        {
            var userStats = UserStatistics.GetByUserOrDefault(Token.UserID, true);
            userStats.EnsureEmotes(TimeSpan.FromMinutes(5));
            var emoticons = TwitchEmote.MultiGetLocal(userStats.EmoteMap.Select(kvp => new KeyInfo(kvp.Value)));
            var emotes = emoticons.Select(emoticon => new EmoteContract
            {
                EmoteID = emoticon.EmoteID,
                Text = emoticon.Regex,
                Url = string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, "twitch-emotes", emoticon.EmoteID),
                EmoteSet = emoticon.EmoticonSet,
                EmoteHeight = emoticon.Height,
                EmoteWidth = emoticon.Width,
            }).ToArray();
            return Ok(emotes);
        }

        [HttpGet]
        [Route("twitch-emotes")]
        [ResponseType(typeof (GetTwitchEmotesResponse))]
        public IHttpActionResult GetTwitchEmotes()
        {
            var userStats = UserStatistics.GetByUserOrDefault(Token.UserID, true);
            userStats.EnsureEmotes(TimeSpan.FromMinutes(5));
            TwitchEmote[] emotes;
            bool canUse;
            if (userStats.EmoteMap.Count > 0)
            {
                emotes = TwitchEmote.MultiGetLocal(userStats.EmoteMap.Select(kvp => new KeyInfo(kvp.Value))).Where(e => !e.IsDeleted && e.IsAlphanumeric).ToArray();
                canUse = true;
            }
            else
            {
                emotes = TwitchEmote.GetAllLocal(e => e.EmoticonSet, 0).Where(e => !e.IsDeleted).ToArray();
                canUse = false;
            }

            return Ok(new GetTwitchEmotesResponse
            {
                CanUseEmotes = canUse,
                Emotes = emotes.Select(e => EmoteContract.ToContract(e, FriendsServiceConfiguration.Instance.AvatarUrlFormat)).ToArray()
            });
        }

        [HttpGet]
        [Route("twitch-token")]
        [ResponseType(typeof (GetTwitchTokenResponse))]
        public IHttpActionResult GetTwitchToken()
        {
            var me = GetCurrentUserAndRegion();
            if (string.IsNullOrEmpty(me.User.TwitchID))
            {
                return NotFound(new GetTwitchTokenResponse
                {
                    Status = GetTwitchTokenStatus.NoAccount
                });
            }

            var account = ExternalAccount.GetLocal(me.User.TwitchID, AccountType.Twitch);
            if (account == null)
            {
                return NotFound(new GetTwitchTokenResponse
                {
                    Status = GetTwitchTokenStatus.NoAccount
                });
            }

            return Ok(new GetTwitchTokenResponse
            {
                Status = GetTwitchTokenStatus.Success,
                Token = account.AuthToken,
                NeedsReauthentication = account.NeedsReauthentication,
                ExternalID = account.ExternalID,
                ExternalUsername = account.ExternalUsername,
                ExternalDisplayName = account.ExternalDisplayName,
            });
        }
    }
}